<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Storefront\PageLoader;

use Doctrine\DBAL\Connection;
use Shopware\Core\Content\Category\Exception\CategoryNotFoundException;
use Shopware\Core\Content\Product\SalesChannel\SalesChannelProductEntity;
use Shopware\Core\Framework\DataAbstractionLayer\Exception\InconsistentCriteriaIdsException;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\System\SalesChannel\Entity\SalesChannelRepository;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Page\GenericPageLoader;
use Symfony\Component\HttpFoundation\Request;
use Topdata\TopdataTopFinderProSW6\Service\DeviceToCustomerService;
use Topdata\TopdataTopFinderProSW6\Service\SettingsService;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\CompatibleDevicesWidget\CompatibleDevicesWidgetPageV2;

/**
 * This class is responsible for loading the compatible devices widget data for a given product.
 * It fetches the product, searches for compatible devices, and retrieves settings for the widget's display.
 */
class CompatibleDevicesWidgetPageV2Loader
{
    /**
     * @readonly
     */
    private GenericPageLoader $genericLoader;
    /**
     * @readonly
     */
    private SalesChannelRepository $productRepository;
    /**
     * @readonly
     */
    private Connection $connection;
    /**
     * @readonly
     */
    private SettingsService $settingsService;
    /**
     * @readonly
     */
    private DeviceToCustomerService $deviceToCustomerService;
    public function __construct(GenericPageLoader       $genericLoader, SalesChannelRepository  $productRepository, Connection              $connection, SettingsService         $settingsService, DeviceToCustomerService $deviceToCustomerService)
    {
        $this->genericLoader = $genericLoader;
        $this->productRepository = $productRepository;
        $this->connection = $connection;
        $this->settingsService = $settingsService;
        $this->deviceToCustomerService = $deviceToCustomerService;
    }

    /**
     * Loads the compatible devices widget data.
     *
     * @param Request $request The request object.
     * @param SalesChannelContext $salesChannelContext The sales channel context.
     * @param string $productid The ID of the product.
     *
     * @throws CategoryNotFoundException
     * @throws InconsistentCriteriaIdsException
     */
    public function load(Request $request, SalesChannelContext $salesChannelContext, string $productid): CompatibleDevicesWidgetPageV2
    {
        // ---- Load the generic page data
        $page = $this->genericLoader->load($request, $salesChannelContext);
        /** @var CompatibleDevicesWidgetPageV2 $page */
        $page = CompatibleDevicesWidgetPageV2::createFrom($page);

        // ---- Search for the product
        $page->product = $this->_searchProduct($productid, $salesChannelContext);

        // ---- Search for compatible devices
        $page->setDevices($this->_searchDevices($productid, $page->product->getParentId()));

        // ---- Get settings for the widget
        $page->groupMode = $this->settingsService->getString('deviceTabMode');
        $page->deviceNameMode = $this->settingsService->getString('deviceNameMode');

        // ---- Get the list of devices for the customer
        $page->deviceList = $this->deviceToCustomerService->getDeviceListOfCustomerArray($salesChannelContext);

        return $page;
    }

    /**
     * Searches for a product by its ID.
     *
     * @param string $productid The ID of the product.
     * @param SalesChannelContext $salesChannelContext The sales channel context.
     *
     * @return SalesChannelProductEntity
     */
    private function _searchProduct(string $productid, SalesChannelContext $salesChannelContext): SalesChannelProductEntity
    {
        return $this
            ->productRepository
            ->search(
                (new Criteria([$productid]))->addAssociation('manufacturer'),
                $salesChannelContext
            )
            ->getEntities()
            ->first();
    }

    /**
     * Searches for compatible devices for a product.
     *
     * @param string $productid The ID of the product.
     * @param string|null $parentProductId The ID of the parent product (optional).
     *
     * @return array
     */
    private function _searchDevices(string $productid, string $parentProductId = null): array
    {
        // ---- Fetch devices associated with the product ID
        $devices = $this->connection->fetchAllAssociative('
SELECT topdata_device.model as name, 
       LOWER(HEX(topdata_device.id)) as id, 
       LOWER(HEX(topdata_device.brand_id)) as brand_id, 
       LOWER(HEX(topdata_device.series_id)) as series_id, 
       LOWER(HEX(topdata_device.type_id)) as type_id, 
       topdata_device.code as code, 
       topdata_brand.label as brand_name,
       topdata_series.label as series_name,
       topdata_device_type.label as type_name
  FROM topdata_device_to_product, 
       topdata_brand,
       topdata_device
  LEFT JOIN `topdata_series` ON `topdata_device`.series_id = `topdata_series`.`id`
  LEFT JOIN topdata_device_type ON `topdata_device`.`type_id` = `topdata_device_type`.`id`
  WHERE (topdata_device.is_enabled=1)
        AND(topdata_device_to_product.product_id=0x' . $productid . ')
        AND(topdata_device_to_product.device_id = topdata_device.id)
        AND(topdata_brand.id=topdata_device.brand_id)
  ORDER BY topdata_brand.label, topdata_series.label, topdata_device.model
            ');

        // ---- If no devices are found and a parent product ID is provided, fetch devices associated with the parent product ID
        if (!count($devices) && $parentProductId) {
            $devices = $this->connection->fetchAllAssociative('
SELECT topdata_device.model as name, 
       LOWER(HEX(topdata_device.id)) as id, 
       LOWER(HEX(topdata_device.brand_id)) as brand_id, 
       LOWER(HEX(topdata_device.series_id)) as series_id, 
       LOWER(HEX(topdata_device.type_id)) as type_id, 
       topdata_device.code as code, 
       topdata_brand.label as brand_name,
       topdata_series.label as series_name,
       topdata_device_type.label as type_name
  FROM topdata_device_to_product, 
       topdata_brand,
       topdata_device
  LEFT JOIN `topdata_series` ON `topdata_device`.series_id = `topdata_series`.`id`
  LEFT JOIN topdata_device_type ON `topdata_device`.`type_id` = `topdata_device_type`.`id`
  WHERE (topdata_device.is_enabled=1)
        AND(topdata_device_to_product.product_id=0x' . $parentProductId . ')
        AND(topdata_device_to_product.device_id = topdata_device.id)
        AND(topdata_brand.id=topdata_device.brand_id)
  ORDER BY topdata_brand.label, topdata_series.label, topdata_device.model
            ');
        }

        return $devices;
    }
}