import Plugin from 'src/plugin-system/plugin.class';
import DomAccess from 'src/helper/dom-access.helper';
import HttpClient from 'src/service/http-client.service';
import CookieStorage from 'src/helper/storage/cookie-storage.helper';
import PseudoModalUtil from 'src/utility/modal-extension/pseudo-modal.util';
import SlimSelect from "../slimselect/slimselect.min";

/**
 * TopdataTopFinderProFinder Plugin
 *
 * This plugin handles the functionality for the Topdata Top Finder Pro feature.
 * It manages various selectors for brands, series, types, and devices, and handles
 * their respective change events.
 */
export default class TopdataTopFinderProFinder extends Plugin {

    static options = {
        switchSelector:        'select.top-finder-switch',
        brandSelector:         'select.top-finder-brand',
        seriesSelector:        'select.top-finder-series',
        typeSelector:          'select.top-finder-types',
        deviceSelector:        'select.top-finder-devices',
        deviceHistorySelector: 'select.top-finder-device-history',

        seriesContainer: '.top-finder-series-container',
        typeContainer:   '.top-finder-types-container',
    };

    // _client; // HttpClient;
    // // these are not IDs, they are the "codes" of the entities, eg "canon"
    // BrandId; // string | null;
    // SeriesId; // string | null;
    // TypeId; // string | null;
    // modal; // PseudoModalUtil;


    /**
     * Initialize the plugin
     * Sets up the HTTP client and initializes the brand, series, and type IDs
     */
    init() {
        console.log('TopdataTopFinderProFinder::init()');
        this._client = new HttpClient();

        // Declare variables ONCE and query the DOM
        const brandElement = DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.brandSelector, false);
        const seriesElement = DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.seriesSelector, false);
        const typeElement = DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.typeSelector, false);
        const deviceElement = DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.deviceSelector, false); // Query device element too
        const historyElement = DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.deviceHistorySelector, false); // Query history element too

        // Set initial IDs using the queried elements
        this.BrandId = brandElement && brandElement.value ? brandElement.value : null;
        this.SeriesId = seriesElement && seriesElement.value ? seriesElement.value : null;
        this.TypeId = typeElement && typeElement.value ? typeElement.value : null;

        console.log({BrandId: this.BrandId, SeriesId: this.SeriesId, TypeId: this.TypeId});

        const ssConfig = { /* Add any common SlimSelect config here if needed */ };

        // Initialize SlimSelect only if the target element exists, using the variables declared above
        if (brandElement) {
            this.brandSlimSelect = new SlimSelect({'select': brandElement, ...ssConfig});
            console.log('SlimSelect initialized for Brand');
        } else { console.log('Brand select not found for SlimSelect'); }

        if (seriesElement) {
            this.seriesSlimSelect = new SlimSelect({'select': seriesElement, ...ssConfig});
            console.log('SlimSelect initialized for Series');
        } else { console.log('Series select not found for SlimSelect'); this.seriesSlimSelect = null; }

        if (typeElement) {
            this.typeSlimSelect = new SlimSelect({'select': typeElement, ...ssConfig});
            console.log('SlimSelect initialized for Type');
        } else { console.log('Type select not found for SlimSelect'); this.typeSlimSelect = null; }

        if (deviceElement) {
            this.deviceSlimSelect = new SlimSelect({'select': deviceElement, ...ssConfig});
            console.log('SlimSelect initialized for Device');
        } else { console.log('Device select not found for SlimSelect'); this.deviceSlimSelect = null; }

        if (historyElement) {
             new SlimSelect({'select': historyElement, ...ssConfig}); // History still not stored
             console.log('SlimSelect initialized for History');
        } else { console.log('History select not found for SlimSelect'); }

        this._registerEvents();

        // --- Trigger initial AJAX load if needed ---
        // Search globally for the first tab that needs AJAX loading
        const firstAjaxTab = document.querySelector('.top-finder-brand-devices-load-tab');
        if (firstAjaxTab) {
            // Use setTimeout to ensure the click happens after the current execution context
            setTimeout(() => {
                firstAjaxTab.click();
            }, 50); // Small delay might help ensure everything is ready
        }
        // --- End Trigger ---
    }

    /**
     * Register events
     * 
     */
    /**
     * Register all event listeners for the plugin
     * 
     */
    _registerEvents() {
        if (this.el.querySelector(TopdataTopFinderProFinder.options.switchSelector) !== null) {
            console.log("SwitchSelector found");
            this.el.querySelector(TopdataTopFinderProFinder.options.switchSelector).addEventListener('change', this.onChangeSwitchSelection.bind(this));
        }

        if (this.el.querySelector(TopdataTopFinderProFinder.options.brandSelector) !== null) {
            console.log("BrandSelector found");
            this.el.querySelector(TopdataTopFinderProFinder.options.brandSelector).addEventListener('change', this.onChangeBrandSelection.bind(this));
        }

        if (this.el.querySelector(TopdataTopFinderProFinder.options.typeSelector) !== null) {
            console.log("TypeSelector found");
            this.el.querySelector(TopdataTopFinderProFinder.options.typeSelector).addEventListener('change', this.onChangeTypeSelection.bind(this));
        }

        if (this.el.querySelector(TopdataTopFinderProFinder.options.seriesSelector) !== null) {
            console.log("SeriesSelector found");
            this.el.querySelector(TopdataTopFinderProFinder.options.seriesSelector).addEventListener('change', this.onChangeSeriesSelection.bind(this));
        }

        if (this.el.querySelector(TopdataTopFinderProFinder.options.deviceSelector) !== null) {
            console.log("DeviceSelector found");
            this.el.querySelector(TopdataTopFinderProFinder.options.deviceSelector).addEventListener('change', this.onChangeDeviceSelection.bind(this));
        }

        // ---- device history selector
        let deviceHistorySelector = this.el.querySelector(TopdataTopFinderProFinder.options.deviceHistorySelector);
        if (deviceHistorySelector) {
            console.log("DeviceHistorySelector found");
            deviceHistorySelector.addEventListener('change', this.onChangeDeviceSelection.bind(this));
        }

        let element;

        element = DomAccess.querySelector(this.el, '.top-finder-selectboxes-close.top-finder-selectboxes-hide', false);
        if (element) {
            element.addEventListener('click', this.closeSelectbox.bind(this));
        }

        element = DomAccess.querySelector(this.el, '.top-finder-selectboxes-close.top-finder-selectboxes-show', false);
        if (element) {
            element.addEventListener('click', this.openSelectbox.bind(this));
        }

        this._registerAjaxTabEvents(); // Add call to register AJAX tab listeners
    }

    /**
     * Register listeners for tabs that load content via AJAX
     * 
     */
    _registerAjaxTabEvents() {
        // Search globally instead of within this.el
        const ajaxTabs = document.querySelectorAll('.top-finder-brand-devices-load-tab');
        // *** ADD LOGGING HERE ***
        console.log('[TopFinder] Found AJAX tabs globally:', ajaxTabs.length, ajaxTabs);
        ajaxTabs.forEach(tab => {
            // *** ADD LOGGING HERE ***
            console.log('[TopFinder] Processing tab for listener:', tab);
            // Ensure listener isn't added multiple times if init runs again
            if (!tab.dataset.ajaxListenerAttached) {
                // *** ADD LOGGING HERE ***
                console.log('[TopFinder] Attaching listener to tab:', tab);
                tab.addEventListener('click', this.onAjaxTabClick.bind(this));
                tab.dataset.ajaxListenerAttached = 'true';
            } else {
                // *** ADD LOGGING HERE ***
                console.log('[TopFinder] Listener already attached to tab:', tab);
            }
        });
    }

    /**
     * Handle clicks on tabs that load content via AJAX
     * @param event
     * 
     */
    onAjaxTabClick(event) {
        const tab = event.currentTarget;
        const className = 'top-finder-brand-devices-load-tab';

        // Prevent multiple clicks if already processing
        if (!tab.classList.contains(className)) {
            return;
        }
        // Prevent default anchor behavior if needed, although Bootstrap tabs might handle this
        event.preventDefault();

        tab.classList.remove(className); // Remove class immediately

        const path = DomAccess.getAttribute(tab, 'data-path');
        const tabPaneSelector = DomAccess.getAttribute(tab, 'href'); // e.g., "#brandXXX-tab-pane"

        if (!path || !tabPaneSelector) {
            console.error('[TopFinder] Missing data-path or href attribute on AJAX tab', tab);
            return;
        }

        // Find the container within the corresponding tab pane
        // Search the whole document as the pane might not be inside this.el
        const tabPane = document.querySelector(tabPaneSelector);
        if (!tabPane) {
            console.error('[TopFinder] Could not find tab pane for selector:', tabPaneSelector);
            return;
        }

        const container = DomAccess.querySelector(tabPane, '.topfinder-devices-compact', false);
        if (!container) {
            console.error('[TopFinder] Could not find .topfinder-devices-compact container in tab pane:', tabPaneSelector);
            // Optionally, show an error message in the tab pane
            tabPane.innerHTML = '<p>Error: Content container not found.</p>';
            return;
        }

        // Ensure loading indicator is visible (it should be there from Twig)
        // container.innerHTML = 'Loading...'; // Or manage spinner visibility

        // *** ADD LOGGING HERE ***
        console.log('[TopFinder] Attempting AJAX load:', { path: path, targetPane: tabPaneSelector });

        this._client.get(path, (responseText, request) => {
            if (request.status >= 400) {
                console.error('[TopFinder] Error loading AJAX tab content:', request.status, responseText);
                container.innerHTML = `<p>Error loading content (${request.status}). Please try again later.</p>`;
                // Optionally re-add the class so the user can try again?
                // tab.classList.add(className);
                return;
            }
            try {
                const response = JSON.parse(responseText);
                if (response.success === true && typeof response.html !== 'undefined') {
                    container.innerHTML = response.html;
                    // Reinitialize Shopware plugins within the newly loaded content
                    // Cast window to any to satisfy TypeScript about PluginManager
                    window.PluginManager.initializePlugins();
                } else {
                    console.error('[TopFinder] AJAX tab response unsuccessful or missing HTML:', response);
                    container.innerHTML = '<p>Error: Invalid response from server.</p>';
                }
            } catch (error) {
                console.error('[TopFinder] Error parsing AJAX tab response:', error, responseText);
                container.innerHTML = '<p>Error: Could not process server response.</p>';
            }
        });
    }

    /**
     * Close the selectbox and set a cookie to remember this state
     * @param event - The event object
     */
    closeSelectbox(event) {
        let element = DomAccess.querySelector(this.el, '.top-finder-selectboxes');
        element.classList.add('top-finder-selectboxes-hidden');

        CookieStorage.setItem('topdata-device-selectboxes', 'hidden', 30);
        return false;
    }

    /**
     * Open the selectbox and remove the cookie that remembers the closed state
     * @param event - The event object
     */
    openSelectbox(event) {
        let element = DomAccess.querySelector(this.el, '.top-finder-selectboxes');
        element.classList.remove('top-finder-selectboxes-hidden');

        CookieStorage.setItem('topdata-device-selectboxes', '', 30);
        return false;
    }


    /**
     * @param event
     */
    /**
     * Handle the change event for the switch selection
     * @param event - The change event object
     */
    onChangeSwitchSelection(event) {
        event.preventDefault();
        console.log(event.currentTarget.value);

        CookieStorage.setItem('topdata-switch', event.currentTarget.value, 30);

        if (event.currentTarget.value === 'types') {
            DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.seriesContainer).style.display = 'none';
            DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.typeContainer).style.display = 'block';
        } else {
            DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.typeContainer).style.display = 'none';
            DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.seriesContainer).style.display = 'block';
        }
    }


    /**
     * Open a modal with the given content
     * @param content - The content to display in the modal
     */
    openModal(content) {
        this.modal = new PseudoModalUtil(content);

        // open the modal window and make it visible
        this.modal.open();
    }

    /**
     * @param event
     */
    /**
     * Handle the change event for brand selection
     * @param event - The change event object
     */
    onChangeBrandSelection(event) {
        event.preventDefault();
        let path;
        this.BrandId = event.currentTarget.value;

        let seriesSelector = DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.seriesSelector, false);
        let typeSelector = DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.typeSelector, false);
        let deviceSelector = DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.deviceSelector);

        if (typeSelector.length !== 0) {
            path = event.currentTarget.dataset.pathloadtypes;
            // this.resetSelectOptions(typeSelector); // Reset handled by loadNewSelectOptions
            this.loadNewSelectOptions(path.replace('brandcode', this.BrandId), typeSelector, this.typeSlimSelect); // Pass instance
        }

        if (seriesSelector.length !== 0) {
            path = event.currentTarget.dataset.pathloadseries;
            // this.resetSelectOptions(seriesSelector); // Reset handled by loadNewSelectOptions
            this.loadNewSelectOptions(path.replace('brandcode', this.BrandId), seriesSelector, this.seriesSlimSelect); // Pass instance
        }

        if (deviceSelector.length !== 0) {
            path = event.currentTarget.dataset.pathloaddevices;
            // this.resetSelectOptions(deviceSelector); // Reset handled by loadNewSelectOptions
            this.loadNewSelectOptions(path.replace('brandcode', this.BrandId), deviceSelector, this.deviceSlimSelect); // Pass instance
        }
    }

    /**
     * @param event
     */
    /**
     * Handle the change event for type selection
     * @param event - The change event object
     */
    onChangeTypeSelection(event) {
        event.preventDefault();
        let path;
        this.TypeId = event.currentTarget.value;
        let deviceSelector = DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.deviceSelector);

        if (deviceSelector.length !== 0) {
            path = event.currentTarget.dataset.pathloaddevices;
            path = path.replace('brandcode', this.BrandId);
            path = path.replace('typecode', this.TypeId);

            // this.resetSelectOptions(deviceSelector); // Reset handled by loadNewSelectOptions
            this.loadNewSelectOptions(path, deviceSelector, this.deviceSlimSelect); // Pass instance
        }
    }

    /**
     * @param event
     */
    /**
     * Handle the change event for series selection
     * @param event - The change event object
     */
    onChangeSeriesSelection(event) {
        event.preventDefault();
        let path;
        this.SeriesId = event.currentTarget.value;
        let deviceSelector = DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.deviceSelector);

        if (deviceSelector.length !== 0) {
            path = event.currentTarget.dataset.pathloaddevices;
            path = path.replace('brandcode', this.BrandId);
            path = path.replace('seriescode', this.SeriesId);

            // this.resetSelectOptions(deviceSelector); // Reset handled by loadNewSelectOptions
            this.loadNewSelectOptions(path, deviceSelector, this.deviceSlimSelect); // Pass instance
        }
    }


    /**
     * @param event
     */
    /**
     * Handle the change event for device selection
     * @param event - The change event object
     */
    onChangeDeviceSelection(event) {
        event.preventDefault();
        let path;
        let DeviceId = event.currentTarget.value;
        // Get the placeholder value from the first option
        const placeholderValue = event.currentTarget.options[0] ? event.currentTarget.options[0].value : '';

        // Only redirect if a non-empty, non-placeholder value is selected
        if (DeviceId !== '' && DeviceId !== placeholderValue) {
            path = event.currentTarget.dataset.pathgotodevice;
            path = path.replace('devicecode', DeviceId);
            window.location.href = path;
        }
        return false;
    }


    /**
     * Reset the options of a select element, keeping only the first option
     * @param selectBox - The select element to reset
     * @returns The reset select element
     */
    resetSelectOptions(selectBox) {
        if (selectBox && selectBox.length !== 0) {
            while (selectBox.options.length > 1) {
                selectBox.remove(1);
            }
        }
        return selectBox;
    }

    /**
     * Load new options for a select element from a given URL
     * @param url - The URL to fetch new options from
     * @param selectElement - The original select DOM element
     * @param slimSelectInstance - The SlimSelect instance for this element (might be null/invalid)
     */
    loadNewSelectOptions(url, selectElement, slimSelectInstance) {
        // First, check if the target select element actually exists in the DOM
        if (!selectElement) {
            console.log('Target select element does not exist for SlimSelect update, skipping URL:', url);
            return; // Exit early if the element isn't there
        }

        // --- Add Placeholder Option ---
        const placeholderText = selectElement.options[0] ? selectElement.options[0].text : 'Select...';
        const placeholderValue = selectElement.options[0] ? selectElement.options[0].value : '';
        const placeholderOption = { text: placeholderText, value: placeholderValue, placeholder: true };

        // More robust check if slimSelectInstance is valid and has expected methods
        const isSlimSelectValid = slimSelectInstance && typeof slimSelectInstance.setData === 'function';

        // Disable and set placeholder while loading
        if (isSlimSelectValid) {
            slimSelectInstance.setData([placeholderOption]);
            if (typeof slimSelectInstance.disable === 'function') {
                slimSelectInstance.disable();
            } else {
                 console.error('SlimSelect instance missing .disable() method', slimSelectInstance);
                 selectElement.disabled = true; // Fallback disable
            }
        } else {
             this.resetSelectOptions(selectElement); // Use old reset
             selectElement.disabled = true;
        }

        this._client.get(url, (responseText, request) => {
            let newOptionsData = [placeholderOption]; // Start data array with placeholder

            if (request.status < 400) { // Check for successful HTTP status
                try {
                    const response = JSON.parse(responseText);
                    if (response.found === true && response.items) {
                        Object.entries(response.items).forEach(entry => {
                            const [key, value] = entry;
                            newOptionsData.push({ text: String(value), value: key });
                        });
                    } else {
                        console.log('Response not found or empty items for ' + selectElement.name, response);
                    }
                } catch (error) {
                    console.error('Error parsing JSON for ' + selectElement.name + ':', error, responseText);
                }
            } else {
                 console.error('Error fetching options for ' + selectElement.name + ':', request.status, responseText);
            }

            // Update SlimSelect or standard select
            // Re-check instance validity inside callback
            const isSlimSelectStillValid = slimSelectInstance && typeof slimSelectInstance.setData === 'function';

            if (isSlimSelectStillValid) {
                slimSelectInstance.setData(newOptionsData); // Update SlimSelect with new data

                // Check if 'set' method exists before calling
                if (typeof slimSelectInstance.set === 'function') {
                     slimSelectInstance.set(placeholderValue); // Set placeholder value
                } else {
                     console.error('SlimSelect instance missing .set() method', slimSelectInstance);
                }
                // Check if 'enable' method exists
                if (typeof slimSelectInstance.enable === 'function') {
                    slimSelectInstance.enable(); // Re-enable the dropdown
                } else {
                     console.error('SlimSelect instance missing .enable() method', slimSelectInstance);
                     selectElement.disabled = false; // Fallback enable
                }
            } else { // Fallback for standard select or invalid instance
                this.resetSelectOptions(selectElement);
                 newOptionsData.forEach(optionData => {
                     if (!optionData.placeholder) {
                        let newOption = new Option(optionData.text, optionData.value);
                        selectElement.add(newOption, undefined);
                     }
                 });
                selectElement.disabled = false;
            }
        });
    }

}