<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\Topfinder\Popup;

use Psr\Log\LoggerInterface;
use Shopware\Core\Framework\Adapter\Translation\Translator;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Page\GenericPageLoader;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Topdata\TopdataTopFinderProSW6\Constants\TopfinderPluginConstants;
use Topdata\TopdataTopFinderProSW6\Service\TopdataDeviceDataService;
use Topdata\TopdataTopFinderProSW6\Service\SettingsService;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Topfinder\MyList\ListPage;
use Topdata\TopdataTopFinderProSW6\Util\UtilSearchTerm;

/**
 * Class PopupSearchPageLoader
 *
 * This class is responsible for loading the popup search page.
 * It retrieves devices matching a search term and prepares the page for display in a popup.
 */
class PopupSearchPageLoader
{

    /**
     * @readonly
     */
    private GenericPageLoader $genericPageLoader;
    /**
     * @readonly
     */
    private TopdataDeviceDataService $deviceDataService;
    /**
     * @readonly
     */
    private SettingsService $settingsService;
    /**
     * @readonly
     */
    private Translator $translator;
    /**
     * @readonly
     */
    private UrlGeneratorInterface $router;
    /**
     * @readonly
     */
    private LoggerInterface $logger;
    /**
     * Constructor dependency injection with property promotion
     */
    public function __construct(GenericPageLoader        $genericPageLoader, TopdataDeviceDataService $deviceDataService, SettingsService          $settingsService, Translator               $translator, UrlGeneratorInterface    $router, LoggerInterface          $logger)
    {
        $this->genericPageLoader = $genericPageLoader;
        $this->deviceDataService = $deviceDataService;
        $this->settingsService = $settingsService;
        $this->translator = $translator;
        $this->router = $router;
        $this->logger = $logger;
    }



    /**
     * Load the popup search page
     *
     * This method loads the generic page, retrieves devices matching the search term,
     * and configures the page for display in a popup.
     *
     * @param Request $request The current request
     * @param SalesChannelContext $salesChannelContext The sales channel context
     * @return ListPage The loaded popup search page
     */
    public function load($request, $salesChannelContext): ListPage
    {
        try {
            // Get offset and term from request
            $offset = (int)$request->query->get('offset', 0);
            $term = $request->query->get('term', '');

            // Filter the search term
            $term = UtilSearchTerm::filterTerm($term);

            // Load the generic page
            $page = $this->genericPageLoader->load($request, $salesChannelContext);

            // Create the list page from the generic page
            $listPage = new ListPage();
            // Copy all properties from the generic page to the list page
            foreach (get_object_vars($page) as $property => $value) {
                $listPage->$property = $value;
            }

            // Set the title for the popup search page
            $listPage->setTitle($this->translator->trans('topdata-topfinder.popup.deviceSearchTitle', ['%term%' => $term]));

            // Set the search term
            $listPage->searchTerm = $term;

            // Set the popup path
            $popupPath = [
                [
                    'name' => $this->translator->trans('topdata-topfinder.popup.deviceSearchTitle', ['%term%' => $term]),
                ]
            ];
            $listPage->popupPath = $popupPath;

            // Return early if the search term is empty
            if (empty($term)) {
                return $listPage;
            }

            // Count total matching devices
            $total = $this->deviceDataService->countFindDevices($term);
            $listPage->devicesTotal = $total;

            // Implement pagination
            $limit = TopfinderPluginConstants::POPUP_DEVICES_PRELOAD_LENGTH;

            // Fetch paginated device results
            $devices = $this->deviceDataService->findDevices($term, $limit, $offset);

            // Load device content (enrich the device data)
            $this->deviceDataService->loadDeviceContent($devices, $salesChannelContext, 'search', $term);

            // Set the devices to the page
            $listPage->setDevices($devices);

            // Set the page type to search
            $listPage->setPageType('search');

            // Set the page to show devices
            $listPage->setShow(['devices' => true]);

            return $listPage;
        } catch (\Exception $e) {
            $this->logger->error('Error loading popup search page: ' . $e->getMessage());
            throw $e;
        }
    }
}