<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\Topfinder\Popup;

use Psr\Log\LoggerInterface;
use Shopware\Core\Framework\Adapter\Translation\Translator;
use Shopware\Storefront\Page\GenericPageLoader;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Topdata\TopdataTopFinderProSW6\Service\TopdataDeviceDataService;
use Topdata\TopdataTopFinderProSW6\Service\SettingsService;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Topfinder\MyList\ListPage;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\Topfinder\MyList\TRASH____ListPageLoader;

/**
 * Class PopupBrandDevicesPageLoader
 *
 * This class is responsible for loading the popup brand devices page.
 * It retrieves devices for a specific brand and prepares the page for display in a popup.
 */
class PopupBrandDevicesPageLoader
{
    /**
     * @readonly
     */
    private GenericPageLoader $genericPageLoader;
    /**
     * @readonly
     */
    private TopdataDeviceDataService $deviceDataService;
    /**
     * @readonly
     */
    private SettingsService $settingsService;
    /**
     * @readonly
     */
    private Translator $translator;
    /**
     * @readonly
     */
    private UrlGeneratorInterface $router;
    /**
     * @readonly
     */
    private LoggerInterface $logger;
    public function __construct(GenericPageLoader        $genericPageLoader, TopdataDeviceDataService $deviceDataService, SettingsService          $settingsService, Translator               $translator, UrlGeneratorInterface    $router, LoggerInterface          $logger)
    {
        $this->genericPageLoader = $genericPageLoader;
        $this->deviceDataService = $deviceDataService;
        $this->settingsService = $settingsService;
        $this->translator = $translator;
        $this->router = $router;
        $this->logger = $logger;
    }

    /**
     * Load the popup brand devices page
     *
     * This method loads the generic page, retrieves a brand by code,
     * gets the devices for that brand, and configures the page for display in a popup.
     *
     * @param mixed $request The current request
     * @param mixed $salesChannelContext The sales channel context
     * @return ListPage The loaded popup brand devices page
     */
    public function load($request, $salesChannelContext): ListPage
    {
        try {
            // Get the brand code from the request attributes
            $brandCode = $request->attributes->get('code');
            
            // Load the generic page
            $page = $this->genericPageLoader->load($request, $salesChannelContext);
            
            // Create the list page from the generic page
            $listPage = new ListPage();
            // Copy all properties from the generic page to the list page
            foreach (get_object_vars($page) as $property => $value) {
                $listPage->$property = $value;
            }
            
            // Get the brand by code
            $brand = $this->brandDataService->getBrandByCode($brandCode, $salesChannelContext);
            
            // If the brand is not found, return the empty page
            if (!$brand) {
                $this->logger->warning('Brand not found with code: ' . $brandCode);
                return $listPage;
            }
            
            // Set the title for the popup brand devices page
            $listPage->setTitle($this->translator->trans('topdata-topfinder.SEO.brandPageTitle', [
                '%brand%' => '<b>' . $brand->getName() . '</b>',
            ]));
            
            // Get devices for the brand
            $devices = $this->deviceDataService->getDevicesArray($brand->getCode());
            
            // Load device content (enrich the device data)
            $this->deviceDataService->loadDeviceContent($devices, $salesChannelContext, 'brand');
            
            // Set the devices to the page
            $listPage->setDevices($devices);
            
            // Set the page type to brand
            $listPage->setPageType(TRASH____ListPageLoader::TYPE_BRAND);
            
            // Set the page to show devices
            $listPage->setShow(['devices' => true]);
            
            return $listPage;
        } catch (\Exception $e) {
            $this->logger->error('Error loading popup brand devices page: ' . $e->getMessage());
            throw $e;
        }
    }
}