<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Service;

use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Session\SessionInterface; // Type hint for session
use Psr\Log\LoggerInterface; // <-- Add Logger

/**
 * Manages the device history USING THE SESSION
 *
 * 05/2025 created (for showing chosen printer/s in cart)
 */
class DeviceHistoryService_Session
{
    /**
     * @readonly
     */
    private RequestStack $requestStack;
    /**
     * @readonly
     */
    private LoggerInterface $logger;
    public const SESSION_KEY = 'topfinder_device_history'; // Made public for debugging
    private const MAX_HISTORY_SIZE = 20;

    public function __construct(RequestStack $requestStack, LoggerInterface $logger)
    {
        $this->requestStack = $requestStack;
        $this->logger = $logger;
    }

    private function _getSession(): SessionInterface
    {
        return $this->requestStack->getSession();
    }

    /**
     * Updates the device history in the session.
     * Adds the deviceId to the beginning and limits history size.
     *
     * @param string $deviceId The ID of the device to add.
     */
    public function updateSessionHistory(string $deviceId): void
    {
        $history = $this->_getHistoryInternal(); // Use internal getter

        // Remove existing entry if present to avoid duplicates and move to front
        $history = array_diff($history, [$deviceId]);
        // Add the new device ID to the beginning of the array
        array_unshift($history, $deviceId);

        // Limit the history size
        $limitedHistory = array_slice($history, 0, self::MAX_HISTORY_SIZE);

        $this->_getSession()->set(self::SESSION_KEY, $limitedHistory);
    }

    /**
     * Retrieves the latest device ID from the session history.
     *
     * @return string|null The latest device ID or null if history is empty or not set.
     */
    public function getLatestDeviceFromHistory(): ?string
    {
        $history = $this->_getHistoryInternal();

        // Return the first element (the most recent one)
        return $history[0] ?? null;
    }

    /**
     * Retrieves the full device history array from the session.
     * Ensures the result is always an array.
     *
     * @return array<int, string> The device history array (most recent first).
     */
    public function getHistory(): array
    {
        return $this->_getHistoryInternal();
    }

    /**
     * Internal helper to get history and ensure it's an array.
     */
    private function _getHistoryInternal(): array
    {
        $history = $this->_getSession()->get(self::SESSION_KEY, []);
        // Log what is read from the session
        $this->logger->info('DeviceHistoryService_Session: Reading history from session key "' . self::SESSION_KEY . '". Found: ', is_array($history) ? $history : [$history]);

        return is_array($history) ? $history : [];
    }


    /**
     * Clears the device history from the session.
     */
    public function clearHistory(): void
    {
        $this->_getSession()->remove(self::SESSION_KEY);
    }
}
