<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Storefront\PageLoader;

use Doctrine\DBAL\Connection;
use Shopware\Core\Framework\Adapter\Translation\Translator;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Page\GenericPageLoader;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Topdata\TopdataTopFinderProSW6\Service\SettingsService;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\BrandLetter\BrandLetterPage;

/**
 * This class is responsible for loading the brand letter page.
 * It fetches brands or devices based on the given letter and settings,
 * and prepares the page with necessary data for rendering.
 */
class BrandLetterPageLoader
{

    /**
     * @readonly
     */
    private GenericPageLoader $genericLoader;
    /**
     * @readonly
     */
    private Connection $connection;
    /**
     * @readonly
     */
    private SettingsService $settingsService;
    /**
     * @readonly
     */
    private UrlGeneratorInterface $router;
    /**
     * @readonly
     */
    private Translator $translator;
    public function __construct(GenericPageLoader     $genericLoader, Connection            $connection, SettingsService       $settingsService, UrlGeneratorInterface $router, Translator            $translator)
    {
        $this->genericLoader = $genericLoader;
        $this->connection = $connection;
        $this->settingsService = $settingsService;
        $this->router = $router;
        $this->translator = $translator;
    }

    /**
     * Loads the brand letter page with the given letter.
     *
     * @param Request $request The request object.
     * @param SalesChannelContext $salesChannelContext The sales channel context.
     * @param string $letter The letter to filter brands or devices by.
     * @return BrandLetterPage The loaded brand letter page.
     */
    public function load(Request $request, SalesChannelContext $salesChannelContext, string $letter): BrandLetterPage
    {
        $page = $this->genericLoader->load($request, $salesChannelContext);
        /** @var BrandLetterPage $page */
        $page = BrandLetterPage::createFrom($page);

        $letterStr = ($letter == '0') ? '0-9' : strtoupper($letter);

        // ---- Check if devices should be displayed instead of brands
        if ($this->settingsService->getBool('letterDisplayDevices')) {
            $page->devices = $this->getDevices($letter);
            $page->displayDevices = true;
        } else {
            $page->brands = $this->getBrands($letter);
            $page->displayDevices = false;
        }

        // ---- Set SEO information
        $page->setTitle($this->translator->trans('topdata-topfinder.SEO.brandLetterPageTitle', [
            '%letter%' => $letterStr,
        ]));

        $page->getMetaInformation()->setMetaTitle($this->translator->trans('topdata-topfinder.SEO.brandLetterMetaTitle', [
            '%letter%' => $letterStr,
        ]));

        $page->getMetaInformation()->setMetaDescription($this->translator->trans('topdata-topfinder.SEO.brandLetterMetaDescription', [
            '%letter%' => $letterStr,
        ]));

        $page->getMetaInformation()->setRobots($this->translator->trans('topdata-topfinder.SEO.brandLetterMetaRobots'));

        // ---- Add breadcrumbs
        $page->addBreadcrumb(
            $this->translator->trans('topdata-topfinder.SEO.brandsPageTitle'),
            $this->router->generate('frontend.top-finder.brands')
        );

        $page->addBreadcrumb(
            $page->getTitle(),
            $this->router->generate('frontend.top-finder.letter', ['letter' => $letter])
        );

        return $page;
    }

    /**
     * Retrieves brands based on the given letter.
     *
     * @param string $letter The letter to filter brands by.
     * @return array The array of brands.
     */
    private function getBrands($letter): array
    {
        $condition = '';
        // ---- Build the SQL condition based on the letter
        if ($letter == '0') {
            $orCondition = [];
            for ($i = 0; $i < 10; $i++) {
                $orCondition[] = '(code LIKE "' . $i . '%")';
            }
            $condition = implode(' OR ', $orCondition);
        } else {
            $condition = 'code LIKE "' . $letter . '%"';
        }

        $devices = $this->connection->fetchAllAssociative('
SELECT LOWER(HEX(id)) as id, 
       code, 
       label as name,
       sort
  FROM topdata_brand
  WHERE (is_enabled=1)
        AND(' . $condition . ')
  ORDER BY sort DESC, code
            ');

        return $devices;
    }

    /**
     * Retrieves devices based on the given letter.
     *
     * @param string $letter The letter to filter devices by.
     * @return array The array of devices.
     */
    private function getDevices($letter): array
    {
        $return = [];
        $condition = '';
        // ---- Build the SQL condition based on the letter
        if ($letter == '0') {
            $orCondition = [];
            for ($i = 0; $i < 10; $i++) {
                $orCondition[] = '(topdata_device.code LIKE "' . $i . '%")';
            }
            $condition = implode(' OR ', $orCondition);
        } else {
            $condition = 'topdata_device.code LIKE "' . $letter . '%"';
        }

        $devices = $this->connection->fetchAllAssociative('
SELECT topdata_device.model as name, 
       LOWER(HEX(topdata_device.id)) as id, 
       LOWER(HEX(topdata_device.brand_id)) as brand_id, 
       LOWER(HEX(topdata_device.series_id)) as series_id, 
       LOWER(HEX(topdata_device.type_id)) as type_id, 
       topdata_device.code as code, 
       topdata_brand.label as brand_name,
       topdata_brand.code as brand_code
  FROM topdata_brand,
       topdata_device
  WHERE (topdata_device.is_enabled=1)
        AND(' . $condition . ')
        AND(topdata_brand.id=topdata_device.brand_id)
  ORDER BY topdata_device.code
            ');

        $current_brand = [
            'brand'      => $devices[0]['brand_name'],
            'brand_id'   => $devices[0]['brand_id'],
            'brand_code' => $devices[0]['brand_code'],
            'devices'    => [],
            'series'     => [],
            'types'      => [],
        ];

        $series = $this->getSeries();
        $types = $this->getTypes();

        // ---- Group devices by brand
        foreach ($devices as $device) {
            if ($current_brand['brand'] != $device['brand_name']) {
                if ($current_brand['devices']) {
                    sort($current_brand['series'], SORT_STRING);
                    if ($current_brand['series'][0] == '') {
                        unset($current_brand['series'][0]);
                        $current_brand['series'][] = '';
                    }
                    sort($current_brand['types'], SORT_STRING);
                    if ($current_brand['types'][0] == '') {
                        unset($current_brand['types'][0]);
                        $current_brand['types'][] = '';
                    }
                    $return[] = $current_brand;
                }

                $current_brand = [
                    'brand'      => $device['brand_name'],
                    'brand_id'   => $device['brand_id'],
                    'brand_code' => $device['brand_code'],
                    'devices'    => [],
                    'series'     => [],
                    'types'      => [],
                ];
            }

            if ($device['series_id'] && isset($series[$device['series_id']])) {
                $device['series_name'] = $series[$device['series_id']]['name'];
            } else {
                $device['series_name'] = '';
            }

            if ($device['type_id'] && isset($types[$device['type_id']])) {
                $device['type_name'] = $types[$device['type_id']]['name'];
            } else {
                $device['type_name'] = '';
            }

            $current_brand['devices'][] = $device;
            if (!in_array($device['series_name'], $current_brand['series'])) {
                $current_brand['series'][] = $device['series_name'];
            }
            if (!in_array($device['type_name'], $current_brand['types'])) {
                $current_brand['types'][] = $device['type_name'];
            }
        }

        if ($current_brand['devices']) {
            sort($current_brand['series'], SORT_STRING);
            if ($current_brand['series'][0] == '') {
                unset($current_brand['series'][0]);
                $current_brand['series'][] = '';
            }
            sort($current_brand['types'], SORT_STRING);
            if ($current_brand['types'][0] == '') {
                unset($current_brand['types'][0]);
                $current_brand['types'][] = '';
            }
            $return[] = $current_brand;
        }

        return $return;
    }

    /**
     * Retrieves all series.
     *
     * @return array The array of series.
     */
    private function getSeries(): array
    {
        $return = [];

        $series = $this->connection->fetchAllAssociative('
SELECT label as name, 
       LOWER(HEX(id)) as id, 
       code as code
  FROM topdata_series
  WHERE is_enabled=1
  ORDER BY label
        ');

        foreach ($series as $serie) {
            $return[$serie['id']] = $serie;
        }

        return $return;
    }

    /**
     * Retrieves all types.
     *
     * @return array The array of types.
     */
    private function getTypes(): array
    {
        $return = [];

        $types = $this->connection->fetchAllAssociative('
SELECT label as name, 
       LOWER(HEX(id)) as id, 
       code as code
  FROM topdata_device_type
  WHERE is_enabled=1
  ORDER BY label
        ');

        foreach ($types as $type) {
            $return[$type['id']] = $type;
        }

        return $return;
    }

    /**
     * Loads the brand letter page for JSON response with the given letter.
     *
     * @param Request $request The request object.
     * @param SalesChannelContext $salesChannelContext The sales channel context.
     * @param string $letter The letter to filter brands by.
     * @return BrandLetterPage The loaded brand letter page.
     */
    public function loadJson(Request $request, SalesChannelContext $salesChannelContext, string $letter): BrandLetterPage
    {
        $page = $this->genericLoader->load($request, $salesChannelContext);
        /** @var BrandLetterPage $page */
        $page = BrandLetterPage::createFrom($page);

        $letterStr = ($letter == '0') ? '0-9' : strtoupper($letter);

        $page->brands = $this->getBrands($letter);

        $page->setTitle($this->translator->trans('topdata-topfinder.SEO.brandLetterPageTitle', [
            '%letter%' => $letterStr,
        ]));

        $page->popupPath[] = [
            'name' => $letterStr,
//            'path' => $this->router->generate('frontend.top-finder-api.popup-letter', ['letter'=>$letter])
        ];

        return $page;
    }
}