<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\Topfinder\MyList;

use Psr\Log\LoggerInterface;
use Shopware\Core\Framework\Adapter\Translation\Translator;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Page\GenericPageLoader;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Topdata\TopdataTopFinderProSW6\Component\BreadcrumbsTrait;
use Topdata\TopdataTopFinderProSW6\Service\TopdataDeviceDataService;
use Topdata\TopdataTopFinderProSW6\Service\SettingsService;
use Topdata\TopdataTopFinderProSW6\Service\TopdataDeviceTypeService;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Topfinder\MyList\ListPage;

/**
 * Class TypeListPageLoader
 *
 * This class is responsible for loading a specific type page.
 * It retrieves type data and devices based on the type code from the request.
 */
class TypeListPageLoader
{
    /**
     * @readonly
     */
    private GenericPageLoader $genericPageLoader;
    /**
     * @readonly
     */
    private TopdataDeviceTypeService $typeDataService;
    /**
     * @readonly
     */
    private TopdataDeviceDataService $deviceDataService;
    /**
     * @readonly
     */
    private SettingsService $settingsService;
    /**
     * @readonly
     */
    private Translator $translator;
    /**
     * @readonly
     */
    private UrlGeneratorInterface $router;
    /**
     * @readonly
     */
    private LoggerInterface $logger;
    /**
     * @readonly
     */
    private RequestStack $requestStack;
    use BreadcrumbsTrait;

    public function __construct(GenericPageLoader        $genericPageLoader, TopdataDeviceTypeService $typeDataService, TopdataDeviceDataService $deviceDataService, SettingsService          $settingsService, Translator               $translator, UrlGeneratorInterface    $router, LoggerInterface          $logger, RequestStack             $requestStack)
    {
        $this->genericPageLoader = $genericPageLoader;
        $this->typeDataService = $typeDataService;
        $this->deviceDataService = $deviceDataService;
        $this->settingsService = $settingsService;
        $this->translator = $translator;
        $this->router = $router;
        $this->logger = $logger;
        $this->requestStack = $requestStack;
    }

    /**
     * Load the type page for a specific type code
     *
     * This method loads the generic page, retrieves the type data,
     * and configures the page with appropriate devices, metadata, and breadcrumbs.
     *
     * @param Request $request The current request
     * @param SalesChannelContext $salesChannelContext The sales channel context
     * @return ListPage The loaded type page
     * @throws \Exception If an error occurs during loading
     */
    public function load(string $typeCode, SalesChannelContext $salesChannelContext): ListPage
    {
        try {

            // Load the generic page
            $request = $this->requestStack->getCurrentRequest();
            $page = $this->genericPageLoader->load($request, $salesChannelContext);

            // Create the list page from the generic page
            $listPage = new ListPage();
            // Copy all properties from the generic page to the list page
            foreach (get_object_vars($page) as $property => $value) {
                $listPage->$property = $value;
            }

            // Set the page type to type
            $listPage->setPageType(TRASH____ListPageLoader::TYPE_TYPE);

            // Get the type by code
            $type = $this->typeDataService->getDeviceTypeByCode($typeCode, $salesChannelContext);

            // If the type is not found, throw an exception
            if (!$type) {
                $this->logger->error('Type not found', ['code' => $typeCode]);
                throw new \Exception('Type not found: ' . $typeCode);
            }

            // Set the type model to the page
            $listPage->model = $type;

            // Get the associated brand
            $brand = $type->getBrand();

            // If the brand is not loaded, log a warning
            if (!$brand) {
                $this->logger->warning('Brand association not loaded for type', ['typeCode' => $typeCode]);
                throw new \Exception('Brand association not loaded for type: ' . $typeCode);
            }

            // Get devices for this type
            $devices = $this->deviceDataService->getDevicesArray($brand->getCode(), null, $type->getId());

            // Set devices to the page
            $listPage->setDevices($devices);

            // Set the page to show devices
            $listPage->setShow(['devices' => true]);

            // Set the SEO title for the type page
            $listPage->setTitle($this->translator->trans('topdata-topfinder.SEO.typePageTitle', [
                '%brand%' => $brand->getName(),
                '%type%'  => $type->getLabel(),
            ]));

            // Set the SEO meta information
            try {
                if (method_exists($listPage, 'getMetaInformation')) {
                    // Set the SEO meta title for the type page
                    $listPage->getMetaInformation()->setMetaTitle($this->translator->trans('topdata-topfinder.SEO.typeMetaTitle', [
                        '%brand%' => $brand->getName(),
                        '%type%'  => $type->getLabel(),
                    ]));

                    // Set the SEO meta description for the type page
                    $listPage->getMetaInformation()->setMetaDescription($this->translator->trans('topdata-topfinder.SEO.typeMetaDescription', [
                        '%brand%' => $brand->getName(),
                        '%type%'  => $type->getLabel(),
                    ]));

                    // Set the SEO robots meta tag for the type page
                    $listPage->getMetaInformation()->setRobots($this->translator->trans('topdata-topfinder.SEO.typeMetaRobots'));
                } else {
                    $this->logger->warning('Meta information methods not available on ListPage class');
                }
            } catch (\Exception $e) {
                $this->logger->warning('Error setting meta information: ' . $e->getMessage());
            }

            // Add breadcrumbs for the type page
            $listPage->addBreadcrumb(
                $this->translator->trans('topdata-topfinder.SEO.brandsPageTitle'),
                $this->router->generate('frontend.top-finder.brands')
            );
            $listPage->addBreadcrumb(
                $this->translator->trans('topdata-topfinder.SEO.brandPageTitle', ['%brand%' => $brand->getName()]),
                $this->router->generate('frontend.top-finder.brandq', ['code' => $brand->getCode()])
            );
            $listPage->addBreadcrumb(
                $listPage->getTitle(),
                $this->router->generate('frontend.top-finder.type', ['code' => $type->getCode()])
            );

            // Apply compact mode limit if configured
            if ($this->settingsService->getInt('searchCompactLimit', true)) {
                $listPage->setCompactModeLimit($this->settingsService->getInt('searchCompactLimit'));
            }

            // If devices are shown, load their content
            if (isset($listPage->getDevices()[0])) {
                $devices = $listPage->getDevices();
                $this->deviceDataService->loadDeviceContent($devices, $salesChannelContext, 'type');
                $listPage->setDevices($devices);
            }

            return $listPage;
        } catch (\Exception $e) {
            $this->logger->error('Error loading type page: ' . $e->getMessage(), [
                'exception' => $e,
                'typeCode'  => $request->attributes->get('code')
            ]);
            throw $e;
        }
    }
}