<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Sitemap;

use Shopware\Core\Content\Sitemap\Provider\AbstractUrlProvider;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Topdata\TopdataTopFinderProSW6\Service\SettingsService;
use Topdata\TopdataConnectorSW6\Core\Content\DeviceType\DeviceTypeCollection;
use Topdata\TopdataConnectorSW6\Core\Content\DeviceType\DeviceTypeEntity;
use Shopware\Core\Content\Sitemap\Struct\Url;
use Shopware\Core\Content\Sitemap\Struct\UrlResult;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Core\Content\Sitemap\Service\ConfigHandler;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\NotFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsAnyFilter;
use Shopware\Core\Content\Seo\SeoUrlPlaceholderHandlerInterface;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Sorting\FieldSorting;
use Shopware\Core\Framework\Plugin\Exception\DecorationPatternException;

/**
 * Provides URLs for device types to be included in the sitemap.
 */
class TypeUrlProvider extends AbstractUrlProvider
{
    /**
     * @readonly
     */
    private EntityRepository $topdataDeviceTypeRepository;
    /**
     * @readonly
     */
    private SettingsService $settingsService;
    /**
     * @readonly
     */
    private SeoUrlPlaceholderHandlerInterface $seoUrlPlaceholderHandler;
    /**
     * @readonly
     */
    private ConfigHandler $configHandler;
    public const CHANGE_FREQ = 'daily';

    public function __construct(EntityRepository $topdataDeviceTypeRepository, SettingsService $settingsService, SeoUrlPlaceholderHandlerInterface $seoUrlPlaceholderHandler, ConfigHandler $configHandler)
    {
        $this->topdataDeviceTypeRepository = $topdataDeviceTypeRepository;
        $this->settingsService = $settingsService;
        $this->seoUrlPlaceholderHandler = $seoUrlPlaceholderHandler;
        $this->configHandler = $configHandler;
    }

    /**
     * @throws DecorationPatternException
     */
    public function getDecorated(): AbstractUrlProvider
    {
        throw new DecorationPatternException(self::class);
    }

    /**
     * Returns the name of this URL provider.
     */
    public function getName(): string
    {
        return 'device_type';
    }

    /**
     * Retrieves URLs for device types to be included in the sitemap.
     *
     * @param SalesChannelContext $salesChannelContext The sales channel context
     * @param int $limit The maximum number of URLs to return
     * @param int|null $offset The offset for pagination
     * @return UrlResult The result containing the URLs and the next offset
     */
    public function getUrls(SalesChannelContext $salesChannelContext, int $limit, ?int $offset = null): UrlResult
    {
        if ($this->settingsService->getBool('showTypes')) {
            return new UrlResult([], null);
        }
        $types = $this->_getTypes($salesChannelContext, $limit, $offset);

        $urls = [];
        $url  = new Url();
        foreach ($types as $type) {
            /** @var \DateTimeInterface $lastmod */
            $lastmod = $type->getUpdatedAt() ?: $type->getCreatedAt();

            $newUrl = clone $url;
            $newUrl->setLoc($this->seoUrlPlaceholderHandler->generate('frontend.top-finder.type', ['code' => $type->getCode()]));
            $newUrl->setLastmod($lastmod);
            $newUrl->setChangefreq(self::CHANGE_FREQ);
            $newUrl->setResource(DeviceTypeEntity::class);
            $newUrl->setIdentifier($type->getId());

            $urls[] = $newUrl;
        }

        if (\count($urls) < $limit) { // last run
            $nextOffset = null;
        } elseif ($offset === null) { // first run
            $nextOffset = $limit;
        } else { // 1+n run
            $nextOffset = $offset + $limit;
        }

        return new UrlResult($urls, $nextOffset);
    }

    /**
     * Retrieves a collection of device types based on the given criteria.
     *
     * @param SalesChannelContext $salesChannelContext The sales channel context
     * @param int $limit The maximum number of device types to return
     * @param int|null $offset The offset for pagination
     * @return DeviceTypeCollection The collection of device types
     */
    private function _getTypes(SalesChannelContext $salesChannelContext, int $limit, ?int $offset): DeviceTypeCollection
    {
        $criteria = new Criteria();
        $criteria->setLimit($limit);

        if ($offset !== null) {
            $criteria->setOffset($offset);
        }

        $criteria->addFilter(new EqualsFilter('enabled', true));
        $criteria->addSorting(new FieldSorting('wsId'));

        $excludedTypeIds = $this->_getExcludedTypeIds($salesChannelContext);
        if (!empty($excludedTypeIds)) {
            $criteria->addFilter(new NotFilter(NotFilter::CONNECTION_AND, [new EqualsAnyFilter('id', $excludedTypeIds)]));
        }

        return $this->topdataDeviceTypeRepository->search($criteria, $salesChannelContext->getContext())->getEntities();
    }

    /**
     * Retrieves the IDs of device types that should be excluded from the sitemap.
     *
     * @param SalesChannelContext $salesChannelContext The sales channel context
     * @return array An array of excluded device type IDs
     */
    private function _getExcludedTypeIds(SalesChannelContext $salesChannelContext): array
    {
        $salesChannelId = $salesChannelContext->getSalesChannel()->getId();

        $excludedUrls = $this->configHandler->get(ConfigHandler::EXCLUDED_URLS_KEY);
        if (empty($excludedUrls)) {
            return [];
        }

        $excludedUrls = array_filter($excludedUrls, static function (array $excludedUrl) use ($salesChannelId) {
            if ($excludedUrl['resource'] !== DeviceTypeEntity::class) {
                return false;
            }

            if ($excludedUrl['salesChannelId'] !== $salesChannelId) {
                return false;
            }

            return true;
        });

        return array_column($excludedUrls, 'identifier');
    }
}
