<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Storefront\PageLoader;

use Shopware\Core\Content\Product\Aggregate\ProductVisibility\ProductVisibilityDefinition;
use Shopware\Core\Content\Product\SalesChannel\ProductAvailableFilter;
use Shopware\Core\Framework\Adapter\Translation\Translator;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\System\SalesChannel\Entity\SalesChannelRepository;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Page\GenericPageLoader;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\Request;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\ListProducts\ListProductsLoadedEvent;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\ListProducts\ListProductsPage;

/**
 * This class is responsible for loading the list products page.
 */
class ListProductsPageLoader
{
    /**
     * @readonly
     */
    private GenericPageLoader $genericLoader;
    /**
     * @readonly
     */
    private Translator $translator;
    /**
     * @readonly
     */
    private SalesChannelRepository $productRepository;
    /**
     * @readonly
     */
    private EventDispatcherInterface $eventDispatcher;
    public function __construct(GenericPageLoader        $genericLoader, Translator               $translator, SalesChannelRepository   $productRepository, EventDispatcherInterface $eventDispatcher)
    {
        $this->genericLoader = $genericLoader;
        $this->translator = $translator;
        $this->productRepository = $productRepository;
        $this->eventDispatcher = $eventDispatcher;
    }

    /**
     * Loads the list products page with the given product IDs.
     *
     * @param Request $request
     * @param SalesChannelContext $salesChannelContext
     * @param array $productIds
     * @return ListProductsPage
     */
    public function loadPage(Request $request, SalesChannelContext $salesChannelContext, array $productIds): ListProductsPage
    {
        // ---- Load the generic page
        $page = $this->genericLoader->load($request, $salesChannelContext);

        $page = ListProductsPage::createFrom($page);
        $page->products = $this->loadProducts($productIds, $salesChannelContext);

        // ---- Set the title and meta title
        $page->setTitle($this->translator->trans('topdata-topfinder.listProductsTitle'));
        $page->getMetaInformation()->setMetaTitle($page->getTitle());

        // ---- Dispatch the list products loaded event
        $this->eventDispatcher->dispatch(
            new ListProductsLoadedEvent($page, $salesChannelContext, $request)
        );

        return $page;
    }

    /**
     * Loads the products with the given product IDs.
     *
     * @param array $productIds
     * @param SalesChannelContext $salesChannelContext
     * @return mixed
     */
    private function loadProducts(array $productIds, SalesChannelContext $salesChannelContext)
    {
        // ---- Create the criteria
        $criteria = (new Criteria($productIds))
            ->addFilter(new ProductAvailableFilter($salesChannelContext->getSalesChannel()->getId(), ProductVisibilityDefinition::VISIBILITY_LINK))
            ->addAssociation('prices')
            ->addAssociation('cover');

        // ---- Search for the products
        return $this->productRepository->search($criteria, $salesChannelContext)->getEntities();
    }
}