<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Service;

use Psr\Log\LoggerInterface;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\EntitySearchResult;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Sorting\FieldSorting;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Topdata\TopdataConnectorSW6\Core\Content\DeviceType\DeviceTypeEntity;
use Topdata\TopdataTopFinderProSW6\Core\Content\TopdataDeviceType\TopdataDeviceTypeEntity;

/**
 * Class TypeDataService
 *
 * This service is responsible for handling device type data operations.
 * It provides methods for retrieving and processing device type data.
 */
class TopdataDeviceTypeService
{
    /**
     * @readonly
     */
    private EntityRepository $topdataDeviceTypeRepository;
    /**
     * @readonly
     */
    private LoggerInterface $logger;
    public function __construct(EntityRepository $topdataDeviceTypeRepository, LoggerInterface  $logger)
    {
        $this->topdataDeviceTypeRepository = $topdataDeviceTypeRepository;
        $this->logger = $logger;
    }

    /**
     * Get a device type by its code
     *
     * This method retrieves a specific device type entity based on the provided code.
     * It returns the device type entity if found, or null if not found.
     * The brand association is included in the returned entity.
     *
     * @param string $code The device type code to search for aka slug
     * @param SalesChannelContext $context The sales channel context
     * @return DeviceTypeEntity|null The device type entity or null if not found
     */
    public function getDeviceTypeByCode(string $code, SalesChannelContext $context): ?DeviceTypeEntity
    {
        try {
            $criteria = new Criteria();
            $criteria->addFilter(new EqualsFilter('enabled', true));
            $criteria->addFilter(new EqualsFilter('code', $code));
            $criteria->addAssociation('brand');

            $type = $this->topdataDeviceTypeRepository->search($criteria, $context->getContext())->first();

            return $type;
        } catch (\Exception $e) {
            $this->logger->error('Error fetching device type by code: ' . $e->getMessage(), [
                'code' => $code
            ]);
            return null;
        }
    }

    /**
     * Get all device types for a specific brand
     *
     * This method retrieves all device type entities associated with the provided brand ID.
     * The results are sorted by the 'label' field in ascending order.
     *
     * @param string $brandId The brand ID to get device types for
     * @param SalesChannelContext $context The sales channel context
     * @return EntitySearchResult The search result containing device type entities
     */
    public function getTypesForBrand(string $brandId, SalesChannelContext $context): EntitySearchResult
    {
        try {
            $criteria = new Criteria();
            $criteria->addFilter(new EqualsFilter('enabled', true));
            $criteria->addFilter(new EqualsFilter('brandId', $brandId));
            $criteria->addSorting(new FieldSorting('label', FieldSorting::ASCENDING));

            return $this->topdataDeviceTypeRepository->search($criteria, $context->getContext());
        } catch (\Exception $e) {
            $this->logger->error('Error fetching device types for brand: ' . $e->getMessage(), [
                'brandId' => $brandId
            ]);

            // Return an empty result in case of error
            return $this->topdataDeviceTypeRepository->search(new Criteria(), $context->getContext());
        }
    }
}