import Plugin from 'src/plugin-system/plugin.class';
import DomAccess from 'src/helper/dom-access.helper';
import HttpClient from 'src/service/http-client.service';
import CookieStorage from 'src/helper/storage/cookie-storage.helper';
import PseudoModalUtil from 'src/utility/modal-extension/pseudo-modal.util';
//import SlimSelect from "../slimselect/slimselect.min";

/**
 * TopdataTopFinderProFinder Plugin
 *
 * This plugin handles the functionality for the Topdata Top Finder Pro feature.
 * It manages various selectors for brands, series, types, and devices, and handles
 * their respective change events.
 */
export default class TopdataTopFinderProFinder extends Plugin {

    static options = {
        switchSelector:        'select.top-finder-switch',
        brandSelector:         'select.top-finder-brand',
        seriesSelector:        'select.top-finder-series',
        typeSelector:          'select.top-finder-types',
        deviceSelector:        'select.top-finder-devices',
        deviceHistorySelector: 'select.top-finder-device-history',

        seriesContainer: '.top-finder-series-container',
        typeContainer:   '.top-finder-types-container',
    };

    private _client: HttpClient;
    // these are not IDs, they are the "codes" of the entities, eg "canon"
    private BrandId: string | null;
    private SeriesId: string | null;
    private TypeId: string | null;
    private modal: PseudoModalUtil;


    /**
     * Initialize the plugin
     * Sets up the HTTP client and initializes the brand, series, and type IDs
     */
    init() {
        console.log('TopdataTopFinderProFinder::init()');
        this._client = new HttpClient();

        // Using the traditional approach instead of optional chaining (?.) and nullish coalescing (??) operators
        const brandElement = DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.brandSelector, false);
        this.BrandId = brandElement && brandElement.value ? brandElement.value : null;

        const seriesElement = DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.seriesSelector, false);
        this.SeriesId = seriesElement && seriesElement.value ? seriesElement.value : null;

        const typeElement = DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.typeSelector, false);
        this.TypeId = typeElement && typeElement.value ? typeElement.value : null;

        console.log({BrandId: this.BrandId, SeriesId: this.SeriesId, TypeId: this.TypeId});

        /*
        this.brandSlimSelect = new SlimSelect({'select': TopdataTopFinderProFinder.options.brandSelector});
        this.seriesSlimSelect = new SlimSelect({'select': TopdataTopFinderProFinder.options.seriesSelector});
        new SlimSelect({'select': TopdataTopFinderProFinder.options.typeSelector});
        this.deviceSlimSelect = new SlimSelect({'select': TopdataTopFinderProFinder.options.deviceSelector});
        new SlimSelect({'select': TopdataTopFinderProFinder.options.deviceHistorySelector});
         */

        this._registerEvents();

        // --- Trigger initial AJAX load if needed ---
        // Search globally for the first tab that needs AJAX loading
        const firstAjaxTab = document.querySelector('.top-finder-brand-devices-load-tab');
        if (firstAjaxTab) {
            // Use setTimeout to ensure the click happens after the current execution context
            setTimeout(() => {
                (firstAjaxTab as HTMLElement).click();
            }, 50); // Small delay might help ensure everything is ready
        }
        // --- End Trigger ---
    }

    /**
     * Register events
     * @private
     */
    /**
     * Register all event listeners for the plugin
     * @private
     */
    _registerEvents() {
        if (this.el.querySelector(TopdataTopFinderProFinder.options.switchSelector) !== null) {
            console.log("SwitchSelector found");
            this.el.querySelector(TopdataTopFinderProFinder.options.switchSelector).addEventListener('change', this.onChangeSwitchSelection.bind(this));
        }

        if (this.el.querySelector(TopdataTopFinderProFinder.options.brandSelector) !== null) {
            console.log("BrandSelector found");
            this.el.querySelector(TopdataTopFinderProFinder.options.brandSelector).addEventListener('change', this.onChangeBrandSelection.bind(this));
        }

        if (this.el.querySelector(TopdataTopFinderProFinder.options.typeSelector) !== null) {
            console.log("TypeSelector found");
            this.el.querySelector(TopdataTopFinderProFinder.options.typeSelector).addEventListener('change', this.onChangeTypeSelection.bind(this));
        }

        if (this.el.querySelector(TopdataTopFinderProFinder.options.seriesSelector) !== null) {
            console.log("SeriesSelector found");
            this.el.querySelector(TopdataTopFinderProFinder.options.seriesSelector).addEventListener('change', this.onChangeSeriesSelection.bind(this));
        }

        if (this.el.querySelector(TopdataTopFinderProFinder.options.deviceSelector) !== null) {
            console.log("DeviceSelector found");
            this.el.querySelector(TopdataTopFinderProFinder.options.deviceSelector).addEventListener('change', this.onChangeDeviceSelection.bind(this));
        }

        // ---- device history selector
        let deviceHistorySelector = this.el.querySelector(TopdataTopFinderProFinder.options.deviceHistorySelector);
        if (deviceHistorySelector) {
            console.log("DeviceHistorySelector found");
            deviceHistorySelector.addEventListener('change', this.onChangeDeviceSelection.bind(this));
        }

        let element;

        element = DomAccess.querySelector(this.el, '.top-finder-selectboxes-close.top-finder-selectboxes-hide', false);
        if (element) {
            element.addEventListener('click', this.closeSelectbox.bind(this));
        }

        element = DomAccess.querySelector(this.el, '.top-finder-selectboxes-close.top-finder-selectboxes-show', false);
        if (element) {
            element.addEventListener('click', this.openSelectbox.bind(this));
        }

        this._registerAjaxTabEvents(); // Add call to register AJAX tab listeners
    }

    /**
     * Register listeners for tabs that load content via AJAX
     * @private
     */
    _registerAjaxTabEvents() {
        // Search globally instead of within this.el
        const ajaxTabs = document.querySelectorAll('.top-finder-brand-devices-load-tab');
        // *** ADD LOGGING HERE ***
        console.log('[TopFinder] Found AJAX tabs globally:', ajaxTabs.length, ajaxTabs);
        ajaxTabs.forEach(tab => {
            // *** ADD LOGGING HERE ***
            console.log('[TopFinder] Processing tab for listener:', tab);
            // Ensure listener isn't added multiple times if init runs again
            if (!(tab as HTMLElement).dataset.ajaxListenerAttached) {
                // *** ADD LOGGING HERE ***
                console.log('[TopFinder] Attaching listener to tab:', tab);
                tab.addEventListener('click', this.onAjaxTabClick.bind(this));
                (tab as HTMLElement).dataset.ajaxListenerAttached = 'true';
            } else {
                // *** ADD LOGGING HERE ***
                console.log('[TopFinder] Listener already attached to tab:', tab);
            }
        });
    }

    /**
     * Handle clicks on tabs that load content via AJAX
     * @param {Event} event
     * @private
     */
    onAjaxTabClick(event) {
        const tab = event.currentTarget as HTMLElement;
        const className = 'top-finder-brand-devices-load-tab';

        // Prevent multiple clicks if already processing
        if (!tab.classList.contains(className)) {
            return;
        }
        // Prevent default anchor behavior if needed, although Bootstrap tabs might handle this
        event.preventDefault();

        tab.classList.remove(className); // Remove class immediately

        const path = DomAccess.getAttribute(tab, 'data-path');
        const tabPaneSelector = DomAccess.getAttribute(tab, 'href'); // e.g., "#brandXXX-tab-pane"

        if (!path || !tabPaneSelector) {
            console.error('[TopFinder] Missing data-path or href attribute on AJAX tab', tab);
            return;
        }

        // Find the container within the corresponding tab pane
        // Search the whole document as the pane might not be inside this.el
        const tabPane = document.querySelector(tabPaneSelector);
        if (!tabPane) {
            console.error('[TopFinder] Could not find tab pane for selector:', tabPaneSelector);
            return;
        }

        const container = DomAccess.querySelector(tabPane, '.topfinder-devices-compact', false);
        if (!container) {
            console.error('[TopFinder] Could not find .topfinder-devices-compact container in tab pane:', tabPaneSelector);
            // Optionally, show an error message in the tab pane
            tabPane.innerHTML = '<p>Error: Content container not found.</p>';
            return;
        }

        // Ensure loading indicator is visible (it should be there from Twig)
        // container.innerHTML = 'Loading...'; // Or manage spinner visibility

        // *** ADD LOGGING HERE ***
        console.log('[TopFinder] Attempting AJAX load:', { path: path, targetPane: tabPaneSelector });

        this._client.get(path, (responseText, request) => {
            if (request.status >= 400) {
                console.error('[TopFinder] Error loading AJAX tab content:', request.status, responseText);
                container.innerHTML = `<p>Error loading content (${request.status}). Please try again later.</p>`;
                // Optionally re-add the class so the user can try again?
                // tab.classList.add(className);
                return;
            }
            try {
                const response = JSON.parse(responseText);
                if (response.success === true && typeof response.html !== 'undefined') {
                    container.innerHTML = response.html;
                    // Reinitialize Shopware plugins within the newly loaded content
                    // Cast window to any to satisfy TypeScript about PluginManager
                    (window as any).PluginManager.initializePlugins();
                } else {
                    console.error('[TopFinder] AJAX tab response unsuccessful or missing HTML:', response);
                    container.innerHTML = '<p>Error: Invalid response from server.</p>';
                }
            } catch (error) {
                console.error('[TopFinder] Error parsing AJAX tab response:', error, responseText);
                container.innerHTML = '<p>Error: Could not process server response.</p>';
            }
        });
    }

    /**
     * Close the selectbox and set a cookie to remember this state
     * @param {Event} event - The event object
     */
    closeSelectbox(event) {
        let element = DomAccess.querySelector(this.el, '.top-finder-selectboxes');
        element.classList.add('top-finder-selectboxes-hidden');

        CookieStorage.setItem('topdata-device-selectboxes', 'hidden', 30);
        return false;
    }

    /**
     * Open the selectbox and remove the cookie that remembers the closed state
     * @param {Event} event - The event object
     */
    openSelectbox(event) {
        let element = DomAccess.querySelector(this.el, '.top-finder-selectboxes');
        element.classList.remove('top-finder-selectboxes-hidden');

        CookieStorage.setItem('topdata-device-selectboxes', '', 30);
        return false;
    }


    /**
     * @param {Event} event
     */
    /**
     * Handle the change event for the switch selection
     * @param {Event} event - The change event object
     */
    onChangeSwitchSelection(event) {
        event.preventDefault();
        console.log(event.currentTarget.value);

        CookieStorage.setItem('topdata-switch', event.currentTarget.value, 30);

        if (event.currentTarget.value === 'types') {
            DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.seriesContainer).style.display = 'none';
            DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.typeContainer).style.display = 'block';
        } else {
            DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.typeContainer).style.display = 'none';
            DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.seriesContainer).style.display = 'block';
        }
    }


    /**
     * Open a modal with the given content
     * @param {string} content - The content to display in the modal
     */
    openModal(content) {
        this.modal = new PseudoModalUtil(content);

        // open the modal window and make it visible
        this.modal.open();
    }

    /**
     * @param {Event} event
     */
    /**
     * Handle the change event for brand selection
     * @param {Event} event - The change event object
     */
    onChangeBrandSelection(event) {
        event.preventDefault();
        let path;
        this.BrandId = event.currentTarget.value;

        let seriesSelector = DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.seriesSelector, false);
        let typeSelector = DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.typeSelector, false);
        let deviceSelector = DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.deviceSelector);

        if (typeSelector.length !== 0) {
            path = event.currentTarget.dataset.pathloadtypes;
            this.resetSelectOptions(typeSelector);
            this.loadNewSelectOptions(path.replace('brandcode', this.BrandId), typeSelector);
        }

        if (seriesSelector.length !== 0) {
            path = event.currentTarget.dataset.pathloadseries;
            this.resetSelectOptions(seriesSelector);
            this.loadNewSelectOptions(path.replace('brandcode', this.BrandId), seriesSelector);
        }

        if (deviceSelector.length !== 0) {
            path = event.currentTarget.dataset.pathloaddevices;
            this.resetSelectOptions(deviceSelector);
            this.loadNewSelectOptions(path.replace('brandcode', this.BrandId), deviceSelector);
        }
    }

    /**
     * @param {Event} event
     */
    /**
     * Handle the change event for type selection
     * @param {Event} event - The change event object
     */
    onChangeTypeSelection(event) {
        event.preventDefault();
        let path;
        this.TypeId = event.currentTarget.value;
        let deviceSelector = DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.deviceSelector);

        if (deviceSelector.length !== 0) {
            path = event.currentTarget.dataset.pathloaddevices;
            path = path.replace('brandcode', this.BrandId);
            path = path.replace('typecode', this.TypeId);

            this.resetSelectOptions(deviceSelector);
            this.loadNewSelectOptions(path, deviceSelector);
        }
    }

    /**
     * @param {Event} event
     */
    /**
     * Handle the change event for series selection
     * @param {Event} event - The change event object
     */
    onChangeSeriesSelection(event) {
        event.preventDefault();
        let path;
        this.SeriesId = event.currentTarget.value;
        let deviceSelector = DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.deviceSelector);

        if (deviceSelector.length !== 0) {
            path = event.currentTarget.dataset.pathloaddevices;
            path = path.replace('brandcode', this.BrandId);
            path = path.replace('seriescode', this.SeriesId);

            this.resetSelectOptions(deviceSelector);
            this.loadNewSelectOptions(path, deviceSelector);
        }
    }


    /**
     * @param {Event} event
     */
    /**
     * Handle the change event for device selection
     * @param {Event} event - The change event object
     */
    onChangeDeviceSelection(event) {
        event.preventDefault();
        let path;
        let DeviceId = event.currentTarget.value;

        if (DeviceId !== '') {
            path = event.currentTarget.dataset.pathgotodevice;
            path = path.replace('devicecode', DeviceId);
            window.location.href = path;
        }
        return false;
    }


    /**
     * Reset the options of a select element, keeping only the first option
     * @param {HTMLSelectElement} selectBox - The select element to reset
     * @returns {HTMLSelectElement} The reset select element
     */
    resetSelectOptions(selectBox) {
        if (selectBox && selectBox.length !== 0) {
            while (selectBox.options.length > 1) {
                selectBox.remove(1);
            }
        }
        return selectBox;
    }

    /**
     * Load new options for a select element from a given URL
     * @param {string} url - The URL to fetch new options from
     * @param {HTMLSelectElement} selectElement - The select element to populate with new options
     */
    loadNewSelectOptions(url, selectElement) {
        this._client.get(url, (responseText, request) => {
            if (request.status >= 400) {
                console.log(responseText);
            }
            try {
                const response = JSON.parse(responseText);
                if (response.found === true) {
                    Object.entries(response.items).forEach(entry => {
                        const [key, value] = entry;
                        // Ensure value is treated as string for Option constructor
                        let newOption = new Option(String(value), key);
                        selectElement.add(newOption, undefined);
                    });
                } else {
                    console.log(response);
                }
            } catch (error) {
                console.log(error);
            }
        });
    }
}