<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Controller;

use Doctrine\DBAL\Connection;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\Validation\DataBag\RequestDataBag;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Controller\StorefrontController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Annotation\Route;
use Topdata\TopdataConnectorSW6\Core\Content\Device\Agregate\DeviceCustomer\DeviceCustomerEntity;
use Topdata\TopdataConnectorSW6\Core\Content\Device\DeviceEntity;
use Topdata\TopdataTopFinderProSW6\FoundationHack\Util\UtilJsonResponse;
use Topdata\TopdataTopFinderProSW6\FoundationHack\Util\UtilUuid;
use Topdata\TopdataTopFinderProSW6\Service\DeviceToCustomerService;

/**
 * Controller for managing the association between devices and customers in the storefront.
 *
 * This controller provides API endpoints for adding, removing, and updating device information
 * related to a customer's personal device list.
 * @\Symfony\Component\Routing\Annotation\Route(defaults={"_routeScope"={"storefront"}})
 */
class ApiController_DeviceToCustomer extends StorefrontController
{
    /**
     * @readonly
     */
    private EntityRepository $topdataDeviceRepository;
    /**
     * @readonly
     */
    private EntityRepository $topdataDeviceToCustomerRepository;
    /**
     * @readonly
     */
    private Connection $connection;
    /**
     * @readonly
     */
    private DeviceToCustomerService $deviceToCustomerService;
    public function __construct(EntityRepository        $topdataDeviceRepository, EntityRepository        $topdataDeviceToCustomerRepository, Connection              $connection, DeviceToCustomerService $deviceToCustomerService)
    {
        $this->topdataDeviceRepository = $topdataDeviceRepository;
        $this->topdataDeviceToCustomerRepository = $topdataDeviceToCustomerRepository;
        $this->connection = $connection;
        $this->deviceToCustomerService = $deviceToCustomerService;
    }
    /**
     * Add device to customer's personal Device List.
     *
     * This method handles the addition of a device to a customer's personal device list.
     * It validates the device ID, checks if the customer is logged in, and then creates
     * an association between the device and the customer in the database.
     *
     * @param Request $request The current request
     * @param SalesChannelContext $context The sales channel context
     * @return JsonResponse JSON response indicating success or failure
     * @\Symfony\Component\Routing\Annotation\Route(path="/top-finder-api/add-device/{deviceId}", name="frontend.top-finder-api.add-device", defaults={"XmlHttpRequest": true}, methods={"GET", "POST"})
     */
    public function addDevice(Request $request, SalesChannelContext $context): JsonResponse
    {
        $deviceId = $request->get('deviceId');
        // ---- Prepare a success response
        $returnSuccess = new JsonResponse([
            'success'  => true,
            'deviceId' => $deviceId,
        ]);
        // ---- Validate the device ID
        if (!UtilUuid::isValidUuid($deviceId)) {
            return UtilJsonResponse::error('Something went wrong - invalid deviceId');
        }
        // ---- Check if the customer is logged in
        if (!$context->getCustomer() || $context->getCustomer()->getGuest()) {
            return UtilJsonResponse::error('Something went wrong - not logged in');
        }
        // ---- Retrieve the device from the database
        $criteria = new Criteria();
        $criteria->addFilter(new EqualsFilter('id', $deviceId))
            ->addFilter(new EqualsFilter('enabled', true))
            ->setLimit(1);
        /** @var DeviceEntity $device */
        $device = $this->topdataDeviceRepository->search($criteria, $context->getContext())->getEntities()->first();
        // ---- Check if the device exists
        if ($device === null) {
            return UtilJsonResponse::error('Something went wrong - device not found');
        }
        // ---- Check if the device is already associated with the customer
        if ($this->deviceToCustomerService->deviceToCustomerId($deviceId, $context->getCustomer()->getId())) {
            return $returnSuccess;
        }
        // ---- Create the association between the device and the customer
        $this->topdataDeviceToCustomerRepository->create([
            [
                'deviceId'   => $deviceId,
                'customerId' => $context->getCustomer()->getId(),
                'extraInfo'  => json_encode(DeviceCustomerEntity::defaultExtraInfo()),
            ],
        ], $context->getContext());
        // ---- Verify that the association was created successfully
        if ($this->deviceToCustomerService->deviceToCustomerId($deviceId, $context->getCustomer()->getId())) {
            return $returnSuccess;
        } else {
            return UtilJsonResponse::error('Something went wrong');
        }
    }
    /**
     * Remove device from customer's personal Device List.
     *
     * This method handles the removal of a device from a customer's personal device list.
     * It validates the device ID, checks if the customer is logged in, and then removes
     * the association between the device and the customer from the database.
     *
     * @param Request $request The current request
     * @param RequestDataBag $data The request data bag
     * @param SalesChannelContext $context The sales channel context
     * @return JsonResponse JSON response indicating success or failure
     * @\Symfony\Component\Routing\Annotation\Route(path="/top-finder-api/remove-device/{deviceId}", name="frontend.top-finder-api.remove-device", defaults={"XmlHttpRequest": true}, methods={"GET", "POST"})
     */
    public function removeDevice(Request $request, RequestDataBag $data, SalesChannelContext $context): JsonResponse
    {
        $deviceId = $request->get('deviceId');
        // ---- Prepare a success response
        $returnSuccess = new JsonResponse([
            'success'  => true,
            'deviceId' => $deviceId,
        ]);
        // ---- Validate the device ID
        if (!UtilUuid::isValidUuid($deviceId)) {
            return UtilJsonResponse::error('Something went wrong - invalid deviceId');
        }
        // ---- Check if the customer is logged in
        if (!$context->getCustomer() || $context->getCustomer()->getGuest()) {
            return UtilJsonResponse::error('Something went wrong - not logged in');
        }
        // ---- Retrieve the device from the database
        $criteria = (new Criteria())
            ->addFilter(new EqualsFilter('id', $deviceId))
            ->setLimit(1);
        /** @var DeviceEntity $device */
        $device = $this->topdataDeviceRepository->search($criteria, $context->getContext())->getEntities()->first();
        // ---- Check if the device exists
        if ($device === null) {
            return UtilJsonResponse::error('Something went wrong - device not found');
        }
        // ---- Delete the association between the device and the customer
        $this->connection->createQueryBuilder()
            ->delete('topdata_device_to_customer')
            ->where('(device_id=0x' . $deviceId . ') AND (customer_id = 0x' . $context->getCustomer()->getId() . ')')
            ->execute();
        return $returnSuccess;
    }
    /**
     * Update device information
     *
     * This method handles the update of device information associated with a customer.
     * It validates the device ID, checks if the customer is logged in, and then updates
     * the extra information for the device-customer association in the database.
     *
     * @param Request $request The current request
     * @param SalesChannelContext $salesChannelContext The sales channel context
     * @return JsonResponse JSON response indicating success or failure of the update
     * @\Symfony\Component\Routing\Annotation\Route(path="/top-finder-api/device-info-update", name="frontend.top-finder-api.device-info-update", defaults={"XmlHttpRequest": true}, methods={"GET", "POST"})
     */
    public function popupDeviceInfoUpdate(Request $request, SalesChannelContext $salesChannelContext): JsonResponse
    {
        $deviceId = $request->get('deviceId');
        // ---- Validate the device ID
        if (!UtilUuid::isValidUuid($deviceId)) {
            return UtilJsonResponse::error('Device Id ' . $deviceId . ' is invalid');
        }
        // ---- Check if the customer is logged in
        if (!$salesChannelContext->getCustomer() || $salesChannelContext->getCustomer()->getGuest()) {
            return UtilJsonResponse::error('no customer');
        }
        $userId = $salesChannelContext->getCustomer()->getId();
        $data = $request->get('formData');
        // ---- Retrieve the device-to-customer association ID
        $deviceToCustomerId = $this->deviceToCustomerService->deviceToCustomerId($deviceId, $userId);
        // ---- Check if the association exists
        if (!$deviceToCustomerId) {
            return UtilJsonResponse::error('no device to customer');
        }
        // ---- Update the extra information for the device-customer association
        $this->topdataDeviceToCustomerRepository->update([
            [
                'id'        => $deviceToCustomerId,
                'extraInfo' => json_encode($data),
            ],
        ], $salesChannelContext->getContext());
        return new JsonResponse([
            'success' => true,
        ]);
    }
}