<?php declare(strict_types=1);

namespace Topdata\TopdataTopFeedSW6\Service;

use Doctrine\DBAL\ArrayParameterType;
use Doctrine\DBAL\Connection;
use Shopware\Core\Framework\Uuid\Uuid;
use Topdata\TopdataConnectorSW6\Enum\ProductRelationshipTypeEnumV2;


/**
 * Helper class for retrieving product relationship data from the unified relationships table.
 *
 * This service provides utility methods for retrieving and counting
 * various types of product relationships such as alternates, bundles, etc.
 * It operates on the single `topdata_product_relationships` table and uses the
 * ProductRelationshipTypeEnumV2 for type-safe internal operations.
 *
 * This class is backward-compatible with the public API of the previous multi-table version.
 */
class TopfeedHelperServiceV2
{
    private const RELATIONSHIPS_TABLE = 'topdata_product_relationships';

    public function __construct(
        private readonly Connection $connection,
    )
    {
    }

    // ========================================================================
    // PUBLIC API METHODS (Backward Compatible)
    // ========================================================================

    public function getAlternateProductIds(string $productId, ?string $parentProductId = null): array
    {
        return $this->getLinkedIds(ProductRelationshipTypeEnumV2::ALTERNATE, $productId, $parentProductId);
    }

    public function getAlternateProductsCount(string $productId, ?string $parentProductId = null): int
    {
        return $this->getRelationshipCount(ProductRelationshipTypeEnumV2::ALTERNATE, $productId, $parentProductId);
    }

    public function getBundledProductIds(string $productId, ?string $parentProductId = null): array
    {
        return $this->getLinkedIds(ProductRelationshipTypeEnumV2::BUNDLED, $productId, $parentProductId);
    }

    public function getBundledProductsCount(string $productId, ?string $parentProductId = null): int
    {
        return $this->getRelationshipCount(ProductRelationshipTypeEnumV2::BUNDLED, $productId, $parentProductId);
    }

    public function getRelatedProductIds(string $productId, ?string $parentProductId = null): array
    {
        return $this->getLinkedIds(ProductRelationshipTypeEnumV2::RELATED, $productId, $parentProductId);
    }

    public function getRelatedProductsCount(string $productId, ?string $parentProductId = null): int
    {
        return $this->getRelationshipCount(ProductRelationshipTypeEnumV2::RELATED, $productId, $parentProductId);
    }

    public function getBundleIds(string $productId, ?string $parentProductId = null): array
    {
        return $this->getLinkedIds(ProductRelationshipTypeEnumV2::BUNDLED, $productId, $parentProductId, 'reverse');
    }

    public function getBundlesCount(string $productId, ?string $parentProductId = null): int
    {
        return $this->getRelationshipCount(ProductRelationshipTypeEnumV2::BUNDLED, $productId, $parentProductId, 'reverse');
    }

    public function getSimilarProductIds(string $productId, ?string $parentProductId = null): array
    {
        return $this->getLinkedIds(ProductRelationshipTypeEnumV2::SIMILAR, $productId, $parentProductId);
    }

    public function getSimilarProductsCount(string $productId, ?string $parentProductId = null): int
    {
        return $this->getRelationshipCount(ProductRelationshipTypeEnumV2::SIMILAR, $productId, $parentProductId);
    }

    public function getColorVariantProductIds(string $productId, ?string $parentProductId = null): array
    {
        return $this->getLinkedIds(ProductRelationshipTypeEnumV2::COLOR_VARIANT, $productId, $parentProductId);
    }

    public function getColorVariantProductsCount(string $productId, ?string $parentProductId = null): int
    {
        return $this->getRelationshipCount(ProductRelationshipTypeEnumV2::COLOR_VARIANT, $productId, $parentProductId);
    }

    public function getCapacityVariantProductIds(string $productId, ?string $parentProductId = null): array
    {
        return $this->getLinkedIds(ProductRelationshipTypeEnumV2::CAPACITY_VARIANT, $productId, $parentProductId);
    }

    public function getCapacityVariantProductsCount(string $productId, ?string $parentProductId = null): int
    {
        return $this->getRelationshipCount(ProductRelationshipTypeEnumV2::CAPACITY_VARIANT, $productId, $parentProductId);
    }

    public function getVariantProductIds(string $productId, ?string $parentProductId = null): array
    {
        return $this->getLinkedIds(ProductRelationshipTypeEnumV2::VARIANT, $productId, $parentProductId);
    }

    public function getVariantProductsCount(string $productId, ?string $parentProductId = null): int
    {
        return $this->getRelationshipCount(ProductRelationshipTypeEnumV2::VARIANT, $productId, $parentProductId);
    }

    // ========================================================================
    // PRIVATE "ENGINE" METHODS (Consolidated Logic)
    // ========================================================================

    /**
     * Generic engine to get all linked product IDs for a given product and relationship type.
     * Handles the fallback logic to a parent product.
     *
     * @param ProductRelationshipTypeEnumV2 $relationshipType The type of relationship to query.
     * @param 'forward'|'reverse' $lookupDirection 'forward' finds products linked *from* the ID, 'reverse' finds products linked *to* the ID.
     */
    private function getLinkedIds(ProductRelationshipTypeEnumV2 $relationshipType, string $productId, ?string $parentProductId, string $lookupDirection = 'forward'): array
    {
        if (!Uuid::isValid($productId)) {
            return [];
        }

        $ids = $this->fetchIdsForProduct($productId, $relationshipType, $lookupDirection);

        if (empty($ids) && $parentProductId && Uuid::isValid($parentProductId)) {
            return $this->fetchIdsForProduct($parentProductId, $relationshipType, $lookupDirection);
        }

        return $ids;
    }

    /**
     * Generic engine to count relationships for a given product and type.
     * Handles the fallback logic to a parent product.
     *
     * @param ProductRelationshipTypeEnumV2 $relationshipType The type of relationship to query.
     * @param 'forward'|'reverse' $lookupDirection 'forward' counts products linked *from* the ID, 'reverse' counts products linked *to* the ID.
     */
    private function getRelationshipCount(ProductRelationshipTypeEnumV2 $relationshipType, string $productId, ?string $parentProductId, string $lookupDirection = 'forward'): int
    {
        if (!Uuid::isValid($productId)) {
            return 0;
        }

        $count = $this->fetchCountForProduct($productId, $relationshipType, $lookupDirection);

        if ($count === 0 && $parentProductId && Uuid::isValid($parentProductId)) {
            return $this->fetchCountForProduct($parentProductId, $relationshipType, $lookupDirection);
        }

        return $count;
    }

    // ========================================================================
    // PRIVATE DATABASE HELPERS (Secure & Parameterized)
    // ========================================================================

    /**
     * Performs the actual secure database query to fetch linked IDs.
     *
     * @param 'forward'|'reverse' $lookupDirection
     */
    private function fetchIdsForProduct(string $productId, ProductRelationshipTypeEnumV2 $relationshipType, string $lookupDirection): array
    {
        $sourceColumn = ($lookupDirection === 'reverse') ? 'linked_product_id' : 'product_id';
        $targetColumn = ($lookupDirection === 'reverse') ? 'product_id' : 'linked_product_id';

        $ids = $this->connection->fetchFirstColumn(
            "SELECT LOWER(HEX(`{$targetColumn}`)) FROM `" . self::RELATIONSHIPS_TABLE . "` WHERE `{$sourceColumn}` = :productId AND `relationship_type` = :relationshipType",
            [
                'productId'        => Uuid::fromHexToBytes($productId),
                'relationshipType' => $relationshipType->value,
            ]
        );

        return $ids ?: [];
    }

    /**
     * Performs the actual secure database query to fetch the count.
     *
     * @param 'forward'|'reverse' $lookupDirection
     */
    private function fetchCountForProduct(string $productId, ProductRelationshipTypeEnumV2 $relationshipType, string $lookupDirection): int
    {
        $sourceColumn = ($lookupDirection === 'reverse') ? 'linked_product_id' : 'product_id';

        $count = $this->connection->fetchOne(
            "SELECT COUNT(*) FROM `" . self::RELATIONSHIPS_TABLE . "` WHERE `{$sourceColumn}` = :productId AND `relationship_type` = :relationshipType",
            [
                'productId'        => Uuid::fromHexToBytes($productId),
                'relationshipType' => $relationshipType->value,
            ]
        );

        return (int)$count;
    }

    /**
     * Get product IDs with other devices for a given product and its linked products.
     * This method has been updated to use secure parameterized queries.
     */
    public function getProductIdsWithOtherDevices(string $productId, array $linkedIds): array
    {
        $allProductIds = array_filter(array_unique([...$linkedIds, $productId]), fn($id) => Uuid::isValid($id));

        if (empty($allProductIds)) {
            return [];
        }

        $deviceMappings = $this->connection->fetchAllAssociative(
            'SELECT LOWER(HEX(product_id)) as product_id, LOWER(HEX(device_id)) as device_id 
             FROM `topdata_device_to_product` 
             WHERE product_id IN (:ids) 
             ORDER BY device_id',
            ['ids' => $allProductIds],
            ['ids' => ArrayParameterType::STRING]
        );

        $productDevices = [];
        foreach ($deviceMappings as $mapping) {
            $productDevices[$mapping['product_id']][] = $mapping['device_id'];
        }

        if (!isset($productDevices[$productId])) {
            return [];
        }

        $originalDeviceSignature = implode('|', $productDevices[$productId]);
        unset($productDevices[$productId]);

        $differentDeviceIds = [];
        foreach ($productDevices as $id => $devices) {
            if (implode('|', $devices) !== $originalDeviceSignature) {
                $differentDeviceIds[] = $id;
            }
        }

        return $differentDeviceIds;
    }
}