<?php declare(strict_types=1);

namespace Topdata\TopdataTopFeedSW6\Subscriber;

use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Topdata\TopdataProductMenu\Storefront\Page\ProductMenu\ProductMenuLoadedEvent;
use Shopware\Core\Framework\Adapter\Translation\Translator;
use Topdata\TopdataTopFeedSW6\Service\TopfeedHelperServiceV1;
use Topdata\TopdataTopFeedSW6\Service\TopfeedHelperServiceV2;
use Doctrine\DBAL\Connection;
use Topdata\TopdataTopFeedSW6\Storefront\Page\LinkedProductsPopup\LinkTypeEnum;
use Topdata\TopdataTopFeedSW6\Service\SettingsService;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Shopware\Core\Framework\Uuid\Uuid;

/**
 * NOTE: ONLY RELEVANT IF THE PLUGIN topdata-product-listing-menu-sw6 IS INSTALLED
 *
 * This subscriber is responsible for adding menu items to the product menu
 * based on various product relationships such as variants, alternates, bundles, etc.
 * It listens to the ProductMenuLoadedEvent and modifies the menu accordingly.
 */
class ProductMenuLoadedEventSubscriber implements EventSubscriberInterface
{


    public function __construct(
        private readonly Translator            $translator,
        private readonly Connection            $connection,
        private readonly SettingsService       $settingsService,
        private readonly UrlGeneratorInterface $router,
        private readonly TopfeedHelperServiceV1  $topfeedHelperServiceV2,
    )
    {
    }

    /**
     * Returns an array of event names this subscriber wants to listen to.
     *
     * @return array The event names to listen to
     */
    public static function getSubscribedEvents(): array
    {
        return [
            ProductMenuLoadedEvent::class => 'addMenuItems',
        ];
    }

    /**
     * Adds menu items to the product menu based on various product relationships.
     *
     * This method checks for different types of related products (variants, alternates, bundles, etc.)
     * and adds corresponding menu items if they exist and are enabled in the settings.
     *
     * @param ProductMenuLoadedEvent $event The event containing the product menu
     */
    public function addMenuItems(ProductMenuLoadedEvent $event): void
    {
        $productId = $event->getPage()->getProductId();
        if (!Uuid::isValid($productId)) {
            return;
        }
        $parentId = $this->connection->executeQuery('
                SELECT LOWER(HEX(parent_id)) as parentId FROM `product`
                 WHERE id = 0x' . $productId . '
                 LIMIT 1')->fetchOne();
        $buttonClasses = ['topdata-top-feed-linked-products'];

        if (!($this->settingsService->getBool('menuHideForVariated') && $parentId)) {
            if ($this->settingsService->getBool('showColorVariantProducts')) {
                $colorVariantsCount = $this->topfeedHelperServiceV2->getColorVariantProductsCount($productId, $parentId);
                if ($colorVariantsCount !== 0) {
                    $event->getPage()->setMenuItem(
                        $this->translator->trans('topdata-topfeed.menu.colors') . " ($colorVariantsCount)",
                        '#',
                        [
                            'path' => $this->router->generate(
                                'frontend.topdata_top_feed.products',
                                [
                                    'productId' => $productId,
                                    'linkType'  => LinkTypeEnum::COLOR_VARIANTS->value
                                ]
                            ),
                        ],
                        $buttonClasses
                    );
                }
            }

            if ($this->settingsService->getBool('showCapacityVariantProducts')) {
                $capacityVariantsCount = $this->topfeedHelperServiceV2->getCapacityVariantProductsCount($productId, $parentId);
                if ($capacityVariantsCount !== 0) {
                    $event->getPage()->setMenuItem(
                        $this->translator->trans('topdata-topfeed.menu.capacity') . " ($capacityVariantsCount)",
                        '#',
                        [
                            'path' => $this->router->generate(
                                'frontend.topdata_top_feed.products',
                                [
                                    'productId' => $productId,
                                    'linkType'  => LinkTypeEnum::CAPACITY_VARIANTS->value
                                ]
                            ),
                        ],
                        $buttonClasses
                    );
                }
            }
        }

        if ($this->settingsService->getBool('showVariantProducts')) {
            $variantsCount = $this->topfeedHelperServiceV2->getVariantProductsCount($productId, $parentId);
            if ($variantsCount !== 0) {
                $event->getPage()->setMenuItem(
                    $this->translator->trans('topdata-topfeed.menu.variants') . " ($variantsCount)",
                    '#',
                    [
                        'path' => $this->router->generate(
                            'frontend.topdata_top_feed.products',
                            [
                                'productId' => $productId,
                                'linkType'  => LinkTypeEnum::VARIANTS->value
                            ]
                        ),
                    ],
                    $buttonClasses
                );
            }
        }

        if ($this->settingsService->getBool('showAlternateProducts')) {
            $alternateCount = $this->topfeedHelperServiceV2->getAlternateProductsCount($productId, $parentId);
            if ($alternateCount !== 0) {
                $event->getPage()->setMenuItem(
                    $this->translator->trans('topdata-topfeed.menu.alternate') . " ($alternateCount)",
                    '#',
                    [
                        'path' => $this->router->generate(
                            'frontend.topdata_top_feed.products',
                            [
                                'productId' => $productId,
                                'linkType'  => LinkTypeEnum::ALTERNATE->value
                            ]
                        ),
                    ],
                    $buttonClasses
                );
            }
        }

        if ($this->settingsService->getBool('showBundledProducts')) {
            $bundledCount = $this->topfeedHelperServiceV2->getBundledProductsCount($productId, $parentId);
            if ($bundledCount !== 0) {
                $event->getPage()->setMenuItem(
                    $this->translator->trans('topdata-topfeed.menu.bundled') . " ($bundledCount)",
                    '#',
                    [
                        'path' => $this->router->generate(
                            'frontend.topdata_top_feed.products',
                            [
                                'productId' => $productId,
                                'linkType'  => LinkTypeEnum::BUNDLED->value
                            ]
                        ),
                    ],
                    $buttonClasses
                );
            }
        }

        if ($this->settingsService->getBool('showBundles')) {
            $bundlesCount = $this->topfeedHelperServiceV2->getBundlesCount($productId, $parentId);
            if ($bundlesCount !== 0) {
                $event->getPage()->setMenuItem(
                    $this->translator->trans('topdata-topfeed.menu.bundles') . " ($bundlesCount)",
                    '#',
                    [
                        'path' => $this->router->generate(
                            'frontend.topdata_top_feed.products',
                            [
                                'productId' => $productId,
                                'linkType'  => LinkTypeEnum::BUNDLES->value
                            ]
                        ),
                    ],
                    $buttonClasses
                );
            }
        }

        if ($this->settingsService->getBool('showRelatedProducts')) {
            $relatedCount = $this->topfeedHelperServiceV2->getRelatedProductsCount($productId, $parentId);
            if ($relatedCount !== 0) {
                $event->getPage()->setMenuItem(
                    $this->translator->trans('topdata-topfeed.menu.related') . " ($relatedCount)",
                    '#',
                    [
                        'path' => $this->router->generate(
                            'frontend.topdata_top_feed.products',
                            [
                                'productId' => $productId,
                                'linkType'  => LinkTypeEnum::RELATED->value
                            ]
                        ),
                    ],
                    $buttonClasses
                );
            }
        }

        if ($this->settingsService->getBool('showSimilarProducts')) {
            $similarProductsCount = $this->topfeedHelperServiceV2->getSimilarProductsCount($productId, $parentId);
            if ($similarProductsCount !== 0) {
                $event->getPage()->setMenuItem(
                    $this->translator->trans('topdata-topfeed.menu.similar') . " ($similarProductsCount)",
                    '#',
                    [
                        'path' => $this->router->generate(
                            'frontend.topdata_top_feed.products',
                            [
                                'productId' => $productId,
                                'linkType'  => LinkTypeEnum::SIMILAR->value
                            ]
                        ),
                    ],
                    $buttonClasses
                );
            }
        }
    }
}
