<?php declare(strict_types=1);

namespace Topdata\TopdataTopFeedSW6\Storefront\Page\LinkedProductsPopup;

use Shopware\Core\Content\Product\Aggregate\ProductVisibility\ProductVisibilityDefinition;
use Shopware\Core\Content\Product\SalesChannel\ProductAvailableFilter;
use Shopware\Core\Content\Product\SalesChannel\SalesChannelProductEntity;
use Shopware\Core\Framework\Adapter\Translation\Translator;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\EntityCollection;
use Shopware\Core\System\SalesChannel\Entity\SalesChannelRepository;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Page\GenericPageLoader;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\Request;
use Topdata\TopdataTopFeedSW6\Content\Product\Events\AssociatedProductsCriteriaEvent;

/**
 * Class LinkedProductsPopupLoader
 * 
 * This class is responsible for loading and preparing data for the LinkedProductsPopupPage.
 * It handles the retrieval of product information, associated products, and prepares
 * the page data for display in the popup.
 */
class LinkedProductsPopupLoader
{
    public function __construct(
        private readonly GenericPageLoader        $genericPageLoader,
        private readonly SalesChannelRepository   $productRepository,
        private readonly EventDispatcherInterface $eventDispatcher,
        private readonly Translator               $translator,
        private readonly ProductIdRetriever       $productIdRetriever,
    )
    {
    }

    /**
     * ==== MAIN ====
     * 10/2024 renamed from load() to buildLinkedProductsPopupPage()
     *
     * it creates a LinkedProductsPopupPage object and fills it with data
     *
     * @param Request $request The current HTTP request
     * @param SalesChannelContext $salesChannelContext The current sales channel context
     * @param string $productId The ID of the product to load
     * @param LinkTypeEnum $pageTypeEnum The type of linked products to retrieve
     * @return LinkedProductsPopupPage The prepared LinkedProductsPopup page object
     */
    public function buildLinkedProductsPopupPage(Request $request, SalesChannelContext $salesChannelContext, string $productId, LinkTypeEnum $pageTypeEnum): LinkedProductsPopupPage
    {
        $page = LinkedProductsPopupPage::createFrom($this->genericPageLoader->load($request, $salesChannelContext));

        $product = $this->_getProduct($productId, $salesChannelContext);
        if (!$product) {
            $page->pageTitle = $this->translator->trans('topdata-topfeed.product.notFound');
            return $page;
        }

        $page->product = $product;
        $page->pageTitle = $this->_getPageTitle($pageTypeEnum, $product);

        $productIds = $this->productIdRetriever->getLinkedProductIdsByLinkType($pageTypeEnum, $productId, $product->getParentId());
        $page->products = $productIds ? $this->_loadProducts($productIds, $salesChannelContext) : null;

        $this->eventDispatcher->dispatch(
            new LinkedProductsPopupLoadedEvent($page, $salesChannelContext, $request)
        );

        return $page;
    }

    /**
     * Retrieves a single product by its ID.
     *
     * @param string $productId The ID of the product to retrieve
     * @param SalesChannelContext $salesChannelContext The current sales channel context
     * @return SalesChannelProductEntity|null The retrieved product entity or null if not found
     */
    private function _getProduct(string $productId, SalesChannelContext $salesChannelContext): ?SalesChannelProductEntity
    {
        return $this->productRepository
            ->search(new Criteria([$productId]), $salesChannelContext)
            ->get($productId);
    }

    /**
     * Generates the page title based on the page type and product.
     *
     * @param LinkTypeEnum $pageType The type of linked products page
     * @param SalesChannelProductEntity $product The main product entity
     * @return string The generated page title
     */
    private function _getPageTitle(LinkTypeEnum $pageType, SalesChannelProductEntity $product): string
    {
        $titleKey = 'topdata-topfeed.product.' . $pageType->value;
        return sprintf(
            '%s %s "%s"',
            $this->translator->trans($titleKey),
            $this->translator->trans('topdata-topfeed.for'),
            $product->getName()
        );
    }

    /**
     * Loads multiple products by their IDs.
     *
     * This method retrieves multiple products, applies filters, and adds necessary
     * associations for display in the popup.
     *
     * @param array $productIds An array of product IDs to load
     * @param SalesChannelContext $salesChannelContext The current sales channel context
     * @return EntityCollection<SalesChannelProductEntity> A collection of loaded product entities
     */
    private function _loadProducts(array $productIds, SalesChannelContext $salesChannelContext): EntityCollection
    {
        $criteria = (new Criteria($productIds))
            ->addFilter(new ProductAvailableFilter($salesChannelContext->getSalesChannel()->getId(), ProductVisibilityDefinition::VISIBILITY_LINK))
            ->addAssociation('prices')
            ->addAssociation('cover')
            ->addAssociation('manufacturer');

        $this->eventDispatcher->dispatch(
            new AssociatedProductsCriteriaEvent($criteria, $salesChannelContext)
        );

        return $this->productRepository->search($criteria, $salesChannelContext)->getEntities();
    }
}
