<?php declare(strict_types=1);

namespace Topdata\TopdataTopFeedSW6\Service;

use Doctrine\DBAL\Connection;
use Shopware\Core\Content\Product\ProductCollection;
use Shopware\Core\Content\Product\ProductEntity;
use Shopware\Core\Content\Product\SalesChannel\SalesChannelProductCollection;
use Shopware\Core\Content\Product\SalesChannel\SalesChannelProductEntity;
use Shopware\Core\Content\Product\SalesChannel\Sorting\ProductSortingEntity;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Sorting\FieldSorting;
use Shopware\Core\Framework\Struct\ArrayEntity;
use Shopware\Core\Framework\Uuid\Uuid;
use Shopware\Core\System\SalesChannel\Entity\SalesChannelRepository;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Page\Product\ProductPage;
use Shopware\Storefront\Page\Product\ProductPageLoadedEvent;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Topdata\TopdataConnectorSW6\Constants\MergedPluginConfigKeyConstants;
use Topdata\TopdataTopFeedSW6\Component\Collection;
use Topdata\TopdataTopFeedSW6\Content\Product\Events\AssociatedProductsCriteriaEvent;

/**
 * Service class to extend the product page with additional information.
 * 10/2024 created (extracted from ProductPageLoadedEventSubscriber)
 */
class ProductPageExtender
{


    private array $colorVariantIds;
    private array $capacityVariantIds;
    // ----
    private SalesChannelProductEntity $product;
    private ProductPage $page;
    private SalesChannelContext $salesChannelContext;


    public function __construct(
        private readonly SettingsService          $settingsService,
        private readonly TopfeedHelperServiceV1   $topfeedHelperServiceV1,
        private readonly SalesChannelRepository   $productRepository,
        private readonly EntityRepository         $productSortingRepository,
        private readonly EventDispatcherInterface $eventDispatcher,
        private readonly PageExtensionService     $pageExtensionService,
        private readonly Connection               $connection,
    )
    {
    }

    /**
     * Initializes the ProductPageExtender with the current product page and sales channel context.
     *
     * @param ProductPage $page The current product page.
     * @param SalesChannelContext $salesChannelContext The current sales channel context.
     */
    public function init(ProductPage $page, SalesChannelContext $salesChannelContext): void
    {
        $this->page = $page;
        $this->salesChannelContext = $salesChannelContext;
        $this->product = $page->getProduct();
        if ($this->product->getParentId() && $this->settingsService->getBool('variantsHideForVariated')) {
            $this->colorVariantIds = [];
            $this->capacityVariantIds = [];
        } else {
            $this->colorVariantIds = $this->topfeedHelperServiceV1->getColorVariantProductIds($this->product->getId(), $this->product->getParentId());
            $this->capacityVariantIds = $this->topfeedHelperServiceV1->getCapacityVariantProductIds($this->product->getId(), $this->product->getParentId());
        }
        // ---- page extension service for easily extend the product page
        $this->pageExtensionService->init('TopdataTopFeedSW6', $page);
    }


    /**
     * Adds associated products to the product page.
     *
     */
    public function addAssociatedProducts(): void
    {
        $currentProductId = $this->product->getId();
        $parentProductId = $this->product->getParentId();

        // ---- Retrieve associated product IDs based on plugin settings
        $associatedProductIds = [
            'alternate_products' => $this->settingsService->getBool(MergedPluginConfigKeyConstants::DISPLAY_OPTION_showAlternateProductsTab) ?
                $this->topfeedHelperServiceV1->getAlternateProductIds($currentProductId, $parentProductId) : false,

            'bundled_products' => $this->settingsService->getBool(MergedPluginConfigKeyConstants::DISPLAY_OPTION_showBundledProductsTab) ?
                $this->topfeedHelperServiceV1->getBundledProductIds($currentProductId, $parentProductId) : false,

            'bundles' => $this->settingsService->getBool(MergedPluginConfigKeyConstants::DISPLAY_OPTION_showBundlesTab) ?
                $this->topfeedHelperServiceV1->getBundleIds($currentProductId, $parentProductId) : false,

            'capacity_variant_products' => $this->settingsService->getBool(MergedPluginConfigKeyConstants::DISPLAY_OPTION_showCapacityVariantProductsTab) ?
                $this->capacityVariantIds : false,

            'color_variant_products' => $this->settingsService->getBool(MergedPluginConfigKeyConstants::DISPLAY_OPTION_showColorVariantProductsTab) ?
                $this->colorVariantIds : false,

            'variant_products' => $this->settingsService->getBool(MergedPluginConfigKeyConstants::DISPLAY_OPTION_showVariantProductsTab) ?
                $this->topfeedHelperServiceV1->getVariantProductIds($currentProductId, $parentProductId) : false,

            'related_products' => $this->settingsService->getBool(MergedPluginConfigKeyConstants::DISPLAY_OPTION_showRelatedProductsTab) ?
                $this->topfeedHelperServiceV1->getRelatedProductIds($currentProductId, $parentProductId) : false,

            'similar_products' => $this->settingsService->getBool(MergedPluginConfigKeyConstants::DISPLAY_OPTION_showSimilarProductsTab) ?
                $this->topfeedHelperServiceV1->getSimilarProductIds($currentProductId, $parentProductId) : false,
        ];

        // ---- Merge all product IDs into a single array
        $allProductIds = [];
        foreach ($associatedProductIds as $ids) {
            if ($ids) { // $ids can be an array or false
                $allProductIds = array_merge($allProductIds, $ids);
            }
        }
        $allProductIds = array_unique($allProductIds);

        // ---- If there are no products, we can stop here
        if (empty($allProductIds)) {
            $this->page->addExtension('associated_products', new Collection());
            return;
        }

        // ---- Fetch the associated products, now correctly sorted
        $associatedProductsSorted = $this->_getSortedProductsCollection($allProductIds);

        // ---- [NEW LOGIC] Organize the sorted products into their respective categories
        $struct = new Collection();
        $categorizedProducts = [];

        // Initialize empty collections for each active category
        foreach ($associatedProductIds as $key => $ids) {
            if ($ids !== false) {
                $categorizedProducts[$key] = new ProductCollection();
            }
        }

        // Iterate through the PRE-SORTED collection and distribute products
        // This preserves the sort order within each category.
        foreach ($associatedProductsSorted as $product) {
            $productId = $product->getId();
            foreach ($categorizedProducts as $key => $collection) {
                // Use the original ID list to check which category this product belongs to
                if (in_array($productId, $associatedProductIds[$key])) {
                    $collection->add($product);
                }
            }
        }

        // Add the now-sorted and populated collections to the final struct
        foreach ($categorizedProducts as $key => $collection) {
            if ($collection->count() > 0) {
                $struct->set($key, $collection);
            }
        }

        $this->page->addExtension('associated_products', $struct);
    }

    /**
     * Adds product variants (colors and capacities) to the product page.
     *
     */
    public function addProductVariants(): void
    {
        $struct = [];
        $struct['colors_type'] = $this->settingsService->getString('colorVariants');
        $struct['colors'] = [];
        // ---- Handle color variants
        if ($struct['colors_type'] !== 'none') {
            $productIds = $this->colorVariantIds;
            if ($productIds !== []) {
                $products = $this->productRepository->search(
                    (new Criteria($productIds))
                        ->addAssociation('prices')
                        ->addAssociation('cover')
                        ->addAssociation('properties.group'),
                    $this->salesChannelContext
                )->getEntities();
                foreach ($products as $product) {
                    /** @var SalesChannelProductEntity $product */

                    $prodProperties = $product->getProperties();
                    $colors = [];
                    $capacities = [];
                    // ---- Extract color and capacity properties
                    foreach ($prodProperties as $property) {

                        $propertyGroupName = $property->getGroup()->getName();
                        if (($translation = $property->getGroup()->getTranslated()) && isset($translation['name'])) {
                            $propertyGroupName = $translation['name'];
                        }

                        $propertyName = $property->getName();
                        if (($translation = $property->getTranslated()) && isset($translation['name'])) {
                            $propertyName = $translation['name'];
                        }

                        if (in_array($propertyGroupName, $this->_colorNames())) {


                            $colors[] = $this->_filterColor($propertyName);
                        }
                        if (in_array($propertyGroupName, $this->_capacityNames())) {
                            $capacities[] = $this->_filterColor($propertyName);
                        }
                    }

                    $productName = $this->_getProductName($product ?? null);

                    $struct['colors'][] = [
                        'id'        => $product->getId(),
                        'color'     => implode(', ', $colors),
                        'capacity'  => implode(', ', $capacities),
                        'name'      => $productName,
                        'cover'     => $product->getCover(),
                        'current'   => false,
                        'delta'     => $this->_stringDelta($productName, $this->product->getName()),
                        'product'   => $product,
                        'shortName' => implode(', ', $colors)
                    ];
                }

                $colors = [];
                $capacities = [];
                // ---- Extract color and capacity properties from the current product
                foreach ($this->product->getProperties() as $property) {

                    $propertyGroupName = $property->getGroup()->getName();
                    if (($translation = $property->getGroup()->getTranslated()) && isset($translation['name'])) {
                        $propertyGroupName = $translation['name'];
                    }

                    $propertyName = $property->getName();
                    if (($translation = $property->getTranslated()) && isset($translation['name'])) {
                        $propertyName = $translation['name'];
                    }


                    if (in_array($propertyGroupName, $this->_colorNames())) {
                        $colors[] = $this->_filterColor($propertyName);
                    }
                    if (in_array($propertyGroupName, $this->_capacityNames())) {
                        $capacities[] = $this->_filterColor($propertyName);
                    }
                }


                $productName = $this->_getProductName($product ?? null);

                $current = [
                    'id'        => $this->product->getId(),
                    'color'     => implode(', ', $colors),
                    'capacity'  => implode(', ', $capacities),
                    'name'      => $productName,
                    'cover'     => $this->product->getCover(),
                    'current'   => true,
                    'delta'     => 0,
                    'product'   => $this->product,
                    'shortName' => implode(', ', $colors)
                ];

                // ---- Delete other capacities
                foreach ($struct['colors'] as $key => $item) {
                    if ($item['capacity'] !== $current['capacity']) {
                        unset($struct['colors'][$key]);
                    }
                }

                // ---- Unique colors (try to find same product version)
                $uniq = [];
                foreach ($struct['colors'] as $key => $item) {
                    $found = false;
                    foreach ($uniq as $k => $v) {
                        if ($v['color'] === $item['color']) {
                            if ($v['delta'] > $item['delta']) {
                                $uniq[$k] = $item;
                            }
                            $found = true;
                            break;
                        }
                    }
                    if (!$found) {
                        $uniq[] = $item;
                    }
                }
                $struct['colors'] = $uniq;

                if ($struct['colors'] !== []) {
                    $struct['colors'][] = $current;

                    usort($struct['colors'], fn($a, $b): int => $a['color'] <=> $b['color']);
                }
            }
        }

        $struct['capacities_type'] = $this->settingsService->getString('capacityVariants');
        $struct['capacities'] = [];
        // ---- Handle capacity variants
        if ($struct['capacities_type'] !== 'none') {
            $productIds = $this->capacityVariantIds;
            if ($productIds !== []) {
                $products = $this
                    ->productRepository
                    ->search(
                        (new Criteria($productIds))
                            ->addAssociation('cover')
                            ->addAssociation('properties.group'),
                        $this->salesChannelContext
                    )
                    ->getEntities();
                foreach ($products as $product) {
                    /** @var SalesChannelProductEntity $product */

                    $prodProperties = $product->getProperties();
                    $colors = [];
                    // ---- Extract capacity properties
                    foreach ($prodProperties as $property) {

                        $propertyGroupName = $property->getGroup()->getName();
                        if (($translation = $property->getGroup()->getTranslated()) && isset($translation['name'])) {
                            $propertyGroupName = $translation['name'];
                        }

                        $propertyName = $property->getName();
                        if (($translation = $property->getTranslated()) && isset($translation['name'])) {
                            $propertyName = $translation['name'];
                        }

                        if (in_array($propertyGroupName, $this->_capacityNames())) {
                            $colors[] = $this->_filterColor($propertyName);
                        }
                    }

                    $productName = $this->_getProductName($product ?? null);

                    $struct['capacities'][] = [
                        'id'        => $product->getId(),
                        'capacity'  => implode(', ', $colors),
                        'name'      => $productName,
                        'cover'     => $product->getCover(),
                        'current'   => false,
                        'product'   => $product,
                        'shortName' => implode(', ', $colors)
                    ];
                }

                $colors = [];
                // ---- Extract capacity properties from the current product
                foreach ($this->product->getProperties() as $property) {

                    $propertyGroupName = $property->getGroup()->getName();
                    if (($translation = $property->getGroup()->getTranslated()) && isset($translation['name'])) {
                        $propertyGroupName = $translation['name'];
                    }

                    $propertyName = $property->getName();
                    if (($translation = $property->getTranslated()) && isset($translation['name'])) {
                        $propertyName = $translation['name'];
                    }

                    if (in_array($propertyGroupName, $this->_capacityNames())) {
                        $colors[] = $propertyName;
                    }
                }

                $productName = $this->_getProductName($product ?? null);


                $struct['capacities'][] = [
                    'id'        => $this->product->getId(),
                    'capacity'  => implode(', ', $colors),
                    'name'      => $productName,
                    'cover'     => $this->product->getCover(),
                    'current'   => true,
                    'product'   => $this->product,
                    'shortName' => implode(', ', $colors)
                ];

                usort($struct['capacities'], fn($a, $b): int => $a['capacity'] <=> $b['capacity']);
            }
        }

        $this->page->addExtension('product_variants', new ArrayEntity($struct));
    }


    /**
     * Adds 'feed_product_devices' extension to the product page if the product or its parent has associated devices.
     * DONE: the injected variable should be called TopdataTopFeedSW6_productHasDevices
     *
     */
    public function addProductHasDevices(): void
    {
        // Check if product devices tab should be shown
        if (!$this->settingsService->getBool('showProductDevicesTab')) {
            return;
        }

        // Check for devices associated with the current product
        $deviceExists = $this->_checkDeviceExists($this->product->getId());

        // If no devices found and product has a parent, check the parent product
        if (!$deviceExists && $this->product->getParentId()) {
            $deviceExists = $this->_checkDeviceExists($this->product->getParentId());
        }

        // If devices found, add the extension to the page - it is just used as a boolean flag in twig..
        $this->pageExtensionService->updateExtension([
            'productHasDevices' => $deviceExists
        ]);
    }

    /**
     * Adds information about other devices for product variants to the product page.
     *
     */
    public function addProductVariantOtherDevices(): void
    {
        if ($this->colorVariantIds) {
            $productIds = $this->topfeedHelperServiceV1->getProductIdsWithOtherDevices(
                $this->product->getId(),
                $this->colorVariantIds
            );
            $this->page->addExtension('productColorsOtherDevices', new ArrayEntity($productIds));
        }

        if ($this->capacityVariantIds) {
            $productIds = $this->topfeedHelperServiceV1->getProductIdsWithOtherDevices(
                $this->product->getId(),
                $this->capacityVariantIds
            );
            $this->page->addExtension('productCapacitiesOtherDevices', new ArrayEntity($productIds));
        }
    }

    /**
     * Returns an array of color-related property names.
     *
     * @return array
     */
    private function _colorNames(): array
    {
        return [
            'Farbe',
            'Color',
        ];
    }

    /**
     * IT DOES NOTHING. REMOVE THIS? OR FIX IT?
     *
     * Filters the color string.
     *
     * @param string $stringWithColor
     * @return string
     */
    private function _filterColor(string $stringWithColor): string
    {
        return $stringWithColor;
    }


    /**
     * Retrieves a collection of products based on given product IDs
     *
     * @param string[] $productIds numeric array of product ids
     * @return ProductCollection|SalesChannelProductCollection|array
     */
    private function _getSortedProductsCollection(array $productIds): ProductCollection
    {
        if (!$productIds) {
            return new ProductCollection();
        }

        $criteria = new Criteria($productIds);
        $this->eventDispatcher->dispatch(
            new AssociatedProductsCriteriaEvent($criteria, $this->salesChannelContext)
        );

        $criteria->addAssociation('prices');
        $criteria->addAssociation('manufacturer');
        $this->_addSorting($criteria);

        $ret = $this->productRepository->search($criteria, $this->salesChannelContext)->getEntities();

        // dd($ret->map(static fn(SalesChannelProductEntity $product) => $product->getName()), $criteria->getSorting());

        return $ret;
    }

    /**
     * Calculates the difference between two strings.
     *
     * @param string $str1
     * @param string $str2
     * @return int
     */
    private function _stringDelta($str1, $str2): int
    {
        if (!is_string($str1) || !is_string($str2)) {
            return 0;
        }
        $str1 = trim($str1);
        $str2 = trim($str2);
        if ($str1 === $str2) {
            return 0;
        }
        $strlen1 = strlen($str1);
        $strlen2 = strlen($str2);
        if ($strlen1 === 0) {
            return $strlen2 ?: 0;
        }

        if ($strlen2 === 0) {
            return $strlen1 ?: 0;
        }

        $deltaStart = 0;
        $deltaEnd = 0;
        // ---- Find the starting difference
        for ($i = 0; $i < $strlen1; $i++) {
            if ($i === $strlen1 - 1 && $strlen2 > $strlen1) {
                return $strlen2 - $strlen1;
            }

            if ($i === $strlen2 - 1 && $strlen1 > $strlen2) {
                return $strlen1 - $strlen2;
            }

            if ($str1[$i] !== $str2[$i]) {
                $deltaStart = $i;
                break;
            }
        }

        // ---- Find the ending difference
        while (true) {
            $strlen1--;
            $strlen2--;
            if ($strlen1 < 0 && $strlen2 >= 0) {
                return $strlen2 - $strlen1;
            }
            if ($strlen2 < 0 && $strlen1 >= 0) {
                return $strlen1 - $strlen2;
            }
            if ($strlen2 < 0 && $strlen1 < 0) {
                return 0;
            }
            if ($str1[$strlen1] !== $str2[$strlen2]) {
                $deltaEnd = max($strlen1, $strlen2);
                break;
            }
        }

        return ($deltaStart > $deltaEnd) ? ($deltaStart + 1 - $deltaEnd) : ($deltaEnd + 1 - $deltaStart);
    }

    /**
     * Returns an array of capacity-related property names.
     *
     * @return array
     */
    private function _capacityNames(): array
    {
        return [
            'Kapazität (Zusatz)',
            'Kapazität',
            'Capacity',
        ];
    }

    /**
     * Checks if a device exists for a given product ID.
     *
     * @param string $productId
     * @return bool
     */
    private function _checkDeviceExists(string $productId): bool
    {
        $query = 'SELECT device_id FROM topdata_device_to_product WHERE product_id = 0x' . $productId . ' LIMIT 1';
        $result = $this->connection->executeQuery($query)->fetchAllAssociative();
        return count($result) > 0;
    }

    /**
     * 07/2025 created as a hotfix: if product is null, we use an empty string
     */
    public function _getProductName($product): string
    {
        if ($product) {
            $productName = $product->getName();
            if (($translation = $product->getTranslated()) && isset($translation['name'])) {
                $productName = $translation['name'];
            }
        } else {
            $productName = '';
        }

        return $productName;
    }

    /**
     * Adds sorting criteria to the given Criteria object based on the plugin settings.
     *
     * @param Criteria $criteria The Criteria object to add sorting to.
     */
    public function _addSorting(Criteria $criteria): void
    {
        $sortingId = $this->settingsService->getString('crossSellingSorting');
        if (Uuid::isValid($sortingId)) {
            $sortingCriteria = new Criteria([$sortingId]);
            /** @var \Shopware\Core\Content\Product\SalesChannel\Sorting\ProductSortingEntity|null $productSorting */
            $productSorting = $this->productSortingRepository->search($sortingCriteria, $this->salesChannelContext->getContext())->first();
            if ($productSorting) {
                $fields = $productSorting->getFields();
                foreach ($fields as $field) {
                    $direction = strtolower($field['order']) === 'asc' ? FieldSorting::ASCENDING : FieldSorting::DESCENDING;
                    $criteria->addSorting(new FieldSorting($field['field'], $direction));
                }
            }
        }
        // Fallback if no valid sorting was applied from the configuration
        if (empty($criteria->getSorting())) {
            $criteria->addSorting(new FieldSorting('product.name', FieldSorting::ASCENDING));
        }
    }


}