/**
 * @module TopdataTopFeedLinkedProducts
 * @description This module defines the TopdataTopFeedLinkedProducts plugin, which handles the loading and displaying of linked products in a modal.
 */

import Plugin from 'src/plugin-system/plugin.class';
import DomAccess from 'src/helper/dom-access.helper';
import HttpClient from 'src/service/http-client.service';
import PseudoModalUtil from 'src/utility/modal-extension/pseudo-modal.util';

/**
 * @class TopdataTopFeedLinkedProducts
 * @extends Plugin
 * @description A plugin to handle linked products functionality.
 */
export default class TopdataTopFeedLinkedProducts extends Plugin {
    /**
     * @static
     * @property {Object} options - Default options for the plugin.
     * @property {string} options.text - Specifies the text that is prompted to the user.
     * @property {string} options.itemSelector - Selector of the item.
     */
    static options = {
        text: 'Seems like there\'s nothing more to see here.',
        itemSelector: '.topdata-top-feed-linked-products',
    };

    /**
     * @method init
     * @description Initializes the plugin, sets up the HttpClient, and adds a click event listener to the specified element.
     */
    init() {
        this._client = new HttpClient();
        let clickElement = DomAccess.querySelector(document, this.options.itemSelector);
        clickElement.addEventListener('click', this.loadData.bind(this));
    }

    /**
     * @method loadData
     * @description Handles the click event, sends an HTTP GET request to fetch data, and displays it in a modal.
     * @param {Event} event - The click event.
     */
    loadData(event) {
        event.preventDefault();
        let path = DomAccess.querySelector(document, this.options.itemSelector).dataset.path;

        this._client.get(path, (responseText, request) => {
            if (request.status >= 400) {
                console.log(responseText);
            }
            try {
                const response = JSON.parse(responseText);
                if (response.success === true) {
                    container.innerHTML = response.html;
                    const content = `
                    <div class="js-pseudo-modal-template">
                        <div class="js-pseudo-modal-template-title-element">` + response.title + `</div>
                        <div class="js-pseudo-modal-template-content-element">` + response.body + `</div>
                    </div>
                    `;
                    this.openModal(content)

                } else {
                    console.log(response);
                }
            } catch (error) {
                console.log(error);
            }
        });
    }

    /**
     * @method openModal
     * @description Creates and opens a modal with the provided content.
     * @param {string} content - The HTML content to be displayed in the modal.
     */
    openModal(content) {
        this.modal = new PseudoModalUtil(content);
        this.modal.open();
    }
}

/*
$(document).on('click',SELECTOR+'linked-products', function() {
    let button = $(this);
//    let productid = button.data('productid');
//    let linktype = button.data('linktype');
    let path = button.data('path');

    let container = $(LINKED_PRODUCTS_MODAL).first();
    let containerTitle = container.find('.modal-title');
    let containerBody = container.find('.modal-body');
    showLinkedProductsContentLoading(true);
    if (typeof linkedProductsPopupHttpClient === 'undefined') {
        linkedProductsPopupHttpClient = new HttpClient(window.accessKey, window.contextToken);
    }
    linkedProductsPopupHttpClient.abort();
    linkedProductsPopupHttpClient.get(path, (response) => {
        response = JSON.parse(response);
//        console.log(response);
        if(response.success === true) {
            containerTitle.html(response.title);
            containerBody.html(response.body);
        }
        else {
            containerTitle.html('error');
            containerBody.html('');
//            console.log(response);
        }
        showLinkedProductsContentLoading(false);
    });
    if($(this).closest(SELECTOR +'modal-content').length === 0) {
        $(LINKED_PRODUCTS_MODAL_TRIGGER).click();
    }
    return false;
});
*/