<?php declare(strict_types=1);

namespace Topdata\TopdataTopFeedSW6\Service;

use Doctrine\DBAL\Connection;
use Shopware\Core\Framework\Uuid\Uuid;

/**
 * TODO: make it a service with connection injected
 * TODO: split into multiple services, one for each relationship type [alternate, bundled, related, similar, capacity_variant, color_variant, variant]
 *       --> strategy pattern
 *
 *
 * Helper class for managing product relationships and counts.
 *
 * This class provides utility methods for retrieving and counting
 * various types of product relationships such as alternate products,
 * bundled products, related products, and variants.
 *
 * 10/2024 made it a service (Helper --> TopfeedHelperService)
 */
class TopfeedHelperService
{

    public function __construct(
        private readonly Connection $connection,
    )
    {
    }

    /**
     * Get query parameters for a specific product relationship type.
     *
     * @param string $key The type of product relationship
     * @return array|false An array of query parameters or false if the key is invalid
     */
    private static function _getQueryParams(string $key): array|false
    {
        if (in_array($key, ['alternate', 'bundled', 'related', 'similar', 'capacity_variant', 'color_variant', 'variant'])) {
            return [
                'table'  => 'topdata_product_to_' . $key,
                'field1' => $key . '_product_id',
                'field2' => 'product_id'
            ];
        }

        if ($key === 'bundles') {
            return [
                'table'  => 'topdata_product_to_bundled',
                'field1' => 'product_id',
                'field2' => 'bundled_product_id'
            ];
        }

        return false;
    }

    /**
     * Get linked product IDs for a specific product and relationship type.
     *
     * @param string $productId The ID of the product to get linked products for
     * @param string $link The type of product relationship
     * @return array An array of linked product IDs
     */
    private function _getLinkedIds(string $productId, string $link): array
    {
        $xids = [];
        if (!Uuid::isValid($productId)) {
            return $xids;
        }

        $queryParams = self::_getQueryParams($link);
        if (!$queryParams) {
            return $xids;
        }

        $productIds = $this->connection->executeQuery('
    SELECT LOWER(HEX(' . $queryParams['field1'] . ')) as id
     FROM ' . $queryParams['table'] . '
     WHERE 0x' . $productId . ' = ' . $queryParams['field2'] . '
            ')->fetchAllAssociative();
        foreach ($productIds as $id) {
            $xids[] = $id['id'];
        }
        return $xids;
    }

    /**
     * Get alternate product IDs for a given product.
     *
     * @param string $productId The ID of the product to get alternate products for
     * @param string|null $parentProductId The ID of the parent product, if applicable
     * @return array An array of alternate product IDs
     */
    public function getAlternateProductIds(string $productId, $parentProductId = null): array
    {
        $xids = $this->_getLinkedIds($productId, 'alternate');

        if (!count($xids) && $parentProductId) {
            return $this->_getLinkedIds($parentProductId, 'alternate');
        }

        return $xids;
    }


    /**
     * Get the count of alternate products for a given product.
     *
     * @param string $productId The ID of the product to count alternate products for
     * @param string|null $parentProductId The ID of the parent product, if applicable
     * @return int The count of alternate products
     */
    public function getAlternateProductsCount(string $productId, $parentProductId = null): int
    {
        if (!Uuid::isValid($productId)) {
            return 0;
        }

        $return = (int)$this->connection->executeQuery('
SELECT COUNT(*) as cnt
 FROM topdata_product_to_alternate
 WHERE 0x' . $productId . ' = product_id
     LIMIT 1
            ')->fetchOne();

        if (!$return && $parentProductId && Uuid::isValid($parentProductId)) {
            return (int)$this->connection->executeQuery('
    SELECT COUNT(*) as cnt
     FROM topdata_product_to_alternate
     WHERE 0x' . $parentProductId . ' = product_id
         LIMIT 1
                ')->fetchOne();
        }
        return $return;
    }


    /**
     * Get bundled product IDs for a given product.
     *
     * @param string $productId The ID of the product to get bundled products for
     * @param string|null $parentProductId The ID of the parent product, if applicable
     * @return array An array of bundled product IDs
     */
    public function getBundledProductIds(string $productId, $parentProductId = null): array
    {
        $xids = $this->_getLinkedIds($productId, 'bundled');

        if (!count($xids) && $parentProductId) {
            return $this->_getLinkedIds($parentProductId, 'bundled');
        }

        return $xids;
    }


    /**
     * Get the count of bundled products for a given product.
     *
     * @param string $productId The ID of the product to count bundled products for
     * @param string|null $parentProductId The ID of the parent product, if applicable
     * @return int The count of bundled products
     */
    public function getBundledProductsCount(string $productId, $parentProductId = null): int
    {
        if (!Uuid::isValid($productId)) {
            return 0;
        }

        $return = (int)$this->connection->executeQuery('
SELECT COUNT(*) as cnt
 FROM topdata_product_to_bundled
 WHERE 0x' . $productId . ' = product_id
     LIMIT 1
            ')->fetchOne();

        if (!$return && $parentProductId && Uuid::isValid($parentProductId)) {
            return (int)$this->connection->executeQuery('
    SELECT COUNT(*) as cnt
     FROM topdata_product_to_bundled
     WHERE 0x' . $parentProductId . ' = product_id
         LIMIT 1
                ')->fetchOne();
        }
        return $return;
    }


    /**
     * Get related product IDs for a given product.
     *
     * @param string $productId The ID of the product to get related products for
     * @param string|null $parentProductId The ID of the parent product, if applicable
     * @return array An array of related product IDs
     */
    public function getRelatedProductIds(string $productId, $parentProductId = null): array
    {
        $xids = $this->_getLinkedIds($productId, 'related');

        if (!count($xids) && $parentProductId) {
            return $this->_getLinkedIds($parentProductId, 'related');
        }

        return $xids;
    }


    /**
     * Get the count of related products for a given product.
     *
     * @param string $productId The ID of the product to count related products for
     * @param string|null $parentProductId The ID of the parent product, if applicable
     * @return int The count of related products
     */
    public function getRelatedProductsCount(string $productId, $parentProductId = null): int
    {
        if (!Uuid::isValid($productId)) {
            return 0;
        }

        $return = (int)$this->connection->executeQuery('
SELECT COUNT(*) as cnt
 FROM topdata_product_to_related
 WHERE 0x' . $productId . ' = product_id
     LIMIT 1
            ')->fetchOne();

        if (!$return && $parentProductId && Uuid::isValid($parentProductId)) {
            return (int)$this->connection->executeQuery('
    SELECT COUNT(*) as cnt
     FROM topdata_product_to_related
     WHERE 0x' . $parentProductId . ' = product_id
         LIMIT 1
                ')->fetchOne();
        }
        return $return;
    }


    /**
     * Get bundle IDs for a given product.
     *
     * @param string $productId The ID of the product to get bundles for
     * @param string|null $parentProductId The ID of the parent product, if applicable
     * @return array An array of bundle IDs
     */
    public function getBundleIds(string $productId, $parentProductId = null): array
    {
        $xids = $this->_getLinkedIds($productId, 'bundles');

        if (!count($xids) && $parentProductId) {
            return $this->_getLinkedIds($parentProductId, 'bundles');
        }

        return $xids;
    }


    /**
     * Get the count of bundles for a given product.
     *
     * @param string $productId The ID of the product to count bundles for
     * @param string|null $parentProductId The ID of the parent product, if applicable
     * @return int The count of bundles
     */
    public function getBundlesCount(string $productId, $parentProductId = null): int
    {
        if (!Uuid::isValid($productId)) {
            return 0;
        }

        $return = (int)$this->connection->executeQuery('
SELECT COUNT(*) as cnt
 FROM topdata_product_to_bundled
 WHERE 0x' . $productId . ' = bundled_product_id
     LIMIT 1
            ')->fetchOne();

        if (!$return && $parentProductId && Uuid::isValid($parentProductId)) {
            return (int)$this->connection->executeQuery('
    SELECT COUNT(*) as cnt
     FROM topdata_product_to_bundled
     WHERE 0x' . $parentProductId . ' = bundled_product_id
         LIMIT 1
                ')->fetchOne();
        }
        return $return;
    }


    /**
     * Get similar product IDs for a given product.
     *
     * @param string $productId The ID of the product to get similar products for
     * @param string|null $parentProductId The ID of the parent product, if applicable
     * @return array An array of similar product IDs
     */
    public function getSimilarProductIds(string $productId, $parentProductId = null): array
    {
        $xids = $this->_getLinkedIds($productId, 'similar');

        if (!count($xids) && $parentProductId) {
            return $this->_getLinkedIds($parentProductId, 'similar');
        }

        return $xids;
    }


    /**
     * Get the count of similar products for a given product.
     *
     * @param string $productId The ID of the product to count similar products for
     * @param string|null $parentProductId The ID of the parent product, if applicable
     * @return int The count of similar products
     */
    public function getSimilarProductsCount(string $productId, $parentProductId = null): int
    {
        if (!Uuid::isValid($productId)) {
            return 0;
        }

        $return = (int)$this->connection->executeQuery('
SELECT COUNT(*) as cnt
 FROM topdata_product_to_similar
 WHERE 0x' . $productId . ' = product_id
     LIMIT 1
            ')->fetchOne();

        if (!$return && $parentProductId && Uuid::isValid($parentProductId)) {
            return (int)$this->connection->executeQuery('
    SELECT COUNT(*) as cnt
     FROM topdata_product_to_similar
     WHERE 0x' . $parentProductId . ' = product_id
         LIMIT 1
                ')->fetchOne();
        }
        return $return;
    }


    /**
     * Get color variant product IDs for a given product.
     *
     * @param string $productId The ID of the product to get color variants for
     * @param string|null $parentProductId The ID of the parent product, if applicable
     * @return array An array of color variant product IDs
     */
    public function getColorVariantProductIds(string $productId, $parentProductId = null): array
    {
        $xids = $this->_getLinkedIds($productId, 'color_variant');

        if (!count($xids) && $parentProductId) {
            return $this->_getLinkedIds($parentProductId, 'color_variant');
        }

        return $xids;
    }


    /**
     * Get the count of color variant products for a given product.
     *
     * @param string $productId The ID of the product to count color variants for
     * @param string|null $parentProductId The ID of the parent product, if applicable
     * @return int The count of color variant products
     */
    public function getColorVariantProductsCount(string $productId, $parentProductId = null): int
    {
        if (!Uuid::isValid($productId)) {
            return 0;
        }

        $return = (int)$this->connection->executeQuery('
SELECT COUNT(*) as cnt
 FROM topdata_product_to_color_variant
 WHERE 0x' . $productId . ' = product_id
     LIMIT 1
            ')->fetchOne();

        if (!$return && $parentProductId && Uuid::isValid($parentProductId)) {
            return (int)$this->connection->executeQuery('
    SELECT COUNT(*) as cnt
     FROM topdata_product_to_color_variant
     WHERE 0x' . $parentProductId . ' = product_id
         LIMIT 1
                ')->fetchOne();
        }
        return $return;
    }


    /**
     * Get capacity variant product IDs for a given product.
     *
     * @param string $productId The ID of the product to get capacity variants for
     * @param string|null $parentProductId The ID of the parent product, if applicable
     * @return array An array of capacity variant product IDs
     */
    public function getCapacityVariantProductIds(string $productId, $parentProductId = null): array
    {
        $xids = $this->_getLinkedIds($productId, 'capacity_variant');

        if (!count($xids) && $parentProductId) {
            return $this->_getLinkedIds($parentProductId, 'capacity_variant');
        }

        return $xids;
    }


    /**
     * Get the count of capacity variant products for a given product.
     *
     * @param string $productId The ID of the product to count capacity variants for
     * @param string|null $parentProductId The ID of the parent product, if applicable
     * @return int The count of capacity variant products
     */
    public function getCapacityVariantProductsCount(string $productId, $parentProductId = null): int
    {
        if (!Uuid::isValid($productId)) {
            return 0;
        }

        $return = (int)$this->connection->executeQuery('
SELECT COUNT(*) as cnt
 FROM topdata_product_to_capacity_variant
 WHERE 0x' . $productId . ' = product_id
     LIMIT 1
            ')->fetchOne();

        if (!$return && $parentProductId && Uuid::isValid($parentProductId)) {
            return (int)$this->connection->executeQuery('
    SELECT COUNT(*) as cnt
     FROM topdata_product_to_capacity_variant
     WHERE 0x' . $parentProductId . ' = product_id
         LIMIT 1
                ')->fetchOne();
        }
        return $return;
    }


    /**
     * Get variant product IDs for a given product.
     *
     * @param string $productId The ID of the product to get variants for
     * @param string|null $parentProductId The ID of the parent product, if applicable
     * @return array An array of variant product IDs
     */
    public function getVariantProductIds(string $productId, $parentProductId = null): array
    {
        $xids = $this->_getLinkedIds($productId, 'variant');

        if (!count($xids) && $parentProductId) {
            $xids = $this->_getLinkedIds($parentProductId, 'variant');
        }

        $xids = [];
        if (!Uuid::isValid($productId)) {
            return $xids;
        }

        return $xids;
    }


    /**
     * Get the count of variant products for a given product.
     *
     * @param string $productId The ID of the product to count variants for
     * @param string|null $parentProductId The ID of the parent product, if applicable
     * @return int The count of variant products
     */
    public function getVariantProductsCount(string $productId, $parentProductId = null): int
    {
        if (!Uuid::isValid($productId)) {
            return 0;
        }

        $return = (int)$this->connection->executeQuery('
SELECT COUNT(*) as cnt
 FROM topdata_product_to_variant
 WHERE 0x' . $productId . ' = product_id
     LIMIT 1
            ')->fetchOne();

        if (!$return && $parentProductId && Uuid::isValid($parentProductId)) {
            return (int)$this->connection->executeQuery('
    SELECT COUNT(*) as cnt
     FROM topdata_product_to_variant
     WHERE 0x' . $parentProductId . ' = product_id
         LIMIT 1
                ')->fetchOne();
        }
        return $return;
    }

    /**
     * Get product IDs with other devices for a given product and its linked products.
     *
     * @param string $productId The ID of the main product
     * @param array $linkedIds An array of linked product IDs
     * @return array An array of product IDs with different device associations
     */
    public function getProductIdsWithOtherDevices(string $productId, array $linkedIds): array
    {
        $xids = [];
        $linkedIds[] = $productId;
        $linkedIds = '0x' . implode(',0x', $linkedIds);
        $datas = $this->connection->executeQuery('SELECT LOWER(HEX(product_id)) as product_id, LOWER(HEX(device_id)) as device_id FROM `topdata_device_to_product` WHERE product_id IN (' . $linkedIds . ') order by device_id')->fetchAllAssociative();
        $productIds = [];
        foreach ($datas as $data)
            $productIds[$data['product_id']][] = $data['device_id'];
        if (!isset($productIds[$productId])) {
            return [];
        }
        $org = $productIds[$productId];
        unset($productIds[$productId]);
        if ($productIds !== []) {
            foreach ($productIds as $id => $devices) {
                if (implode('|', $devices) !== implode('|', $org)) {
                    $xids[] = $id;
                }
            }
        }

        return $xids;
    }
}
