<?php declare(strict_types=1);

namespace Topdata\TopdataTopFeedSW6\Service;

use Doctrine\DBAL\Connection;
use Shopware\Core\Content\Product\ProductCollection;
use Shopware\Core\Content\Product\ProductEntity;
use Shopware\Core\Content\Product\SalesChannel\SalesChannelProductEntity;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Sorting\FieldSorting;
use Shopware\Core\Framework\Struct\ArrayEntity;
use Shopware\Core\Framework\Uuid\Uuid;
use Shopware\Core\System\SalesChannel\Entity\SalesChannelRepository;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Page\Product\ProductPage;
use Shopware\Storefront\Page\Product\ProductPageLoadedEvent;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Topdata\TopdataTopFeedSW6\Component\Collection;
use Topdata\TopdataTopFeedSW6\Content\Product\Events\AssociatedProductsCriteriaEvent;

/**
 * 10/2024 created (extracted from ProductPageLoadedEventSubscriber)
 */
class ProductPageExtender
{


    private array $colorVariantIds;
    private array $capacityVariantIds;
    // ----
    private SalesChannelProductEntity $product;
    private ProductPage $page;
    private SalesChannelContext $salesChannelContext;


    public function __construct(
        private readonly SettingsService          $settingsService,
        private readonly TopfeedHelperService     $topfeedHelperService,
        private readonly SalesChannelRepository   $productRepository,
        private readonly EventDispatcherInterface $eventDispatcher,
        private readonly PageExtensionService     $pageExtensionService,
        private readonly Connection               $connection,
    )
    {
    }


    public function init(ProductPage $page, SalesChannelContext $salesChannelContext): void
    {
        $this->page = $page;
        $this->salesChannelContext = $salesChannelContext;
        $this->product = $page->getProduct();
        if ($this->product->getParentId() && $this->settingsService->getBool('variantsHideForVariated')) {
            $this->colorVariantIds = [];
            $this->capacityVariantIds = [];
        } else {
            $this->colorVariantIds = $this->topfeedHelperService->getColorVariantProductIds($this->product->getId(), $this->product->getParentId());
            $this->capacityVariantIds = $this->topfeedHelperService->getCapacityVariantProductIds($this->product->getId(), $this->product->getParentId());
        }
        // ---- page extension service for easily extend the product page
        $this->pageExtensionService->init('TopdataTopFeedSW6', $page);
    }


    /**
     * Adds associated products to the product page.
     *
     */
    public function addAssociatedProducts(): void
    {
        $currentProductId = $this->product->getId();
        $parentProductId = $this->product->getParentId();
        $struct = new Collection();

        $associatedProductIds = [
            'alternate_products' => $this->settingsService->getBool('showAlternateProductsTab') ?
                $this->topfeedHelperService->getAlternateProductIds($currentProductId, $parentProductId) : false,

            'bundled_products' => $this->settingsService->getBool('showBundledProductsTab') ?
                $this->topfeedHelperService->getBundledProductIds($currentProductId, $parentProductId) : false,

            'bundles' => $this->settingsService->getBool('showBundlesTab') ?
                $this->topfeedHelperService->getBundleIds($currentProductId, $parentProductId) : false,

            'capacity_variant_products' => $this->settingsService->getBool('showCapacityVariantProductsTab') ?
                $this->capacityVariantIds : false,

            'color_variant_products' => $this->settingsService->getBool('showColorVariantProductsTab') ?
                $this->colorVariantIds : false,

            'variant_products' => $this->settingsService->getBool('showVariantProductsTab') ?
                $this->topfeedHelperService->getVariantProductIds($currentProductId, $parentProductId) : false,

            'related_products' => $this->settingsService->getBool('showRelatedProductsTab') ?
                $this->topfeedHelperService->getRelatedProductIds($currentProductId, $parentProductId) : false,

            'similar_products' => $this->settingsService->getBool('showSimilarProductsTab') ?
                $this->topfeedHelperService->getSimilarProductIds($currentProductId, $parentProductId) : false,
        ];

        $productIds = [];

        foreach ($associatedProductIds as $el) {
            if ($el) {
                $productIds = array_merge($productIds, $el);
            }
        }

        $productIds = array_unique($productIds);

        if ($productIds !== []) {
            $associatedProducts = $this->_getProductsCollection($productIds);
        }

        foreach ($associatedProductIds as $key => $el) {
            if ($el) {
                $prods = [];
                foreach ($el as $id) {
                    if ($associatedProducts->get($id)) {
                        $prods[] = $associatedProducts->get($id);
                    }
                }
                if ($prods !== []) {
                    $struct->set($key, new ProductCollection($prods));
                }

            }
        }

        $this->page->addExtension('associated_products', $struct);
    }


    /**
     * Adds product variants (colors and capacities) to the product page.
     *
     */
    public function addProductVariants(): void
    {
        $struct = [];
        $struct['colors_type'] = $this->settingsService->getString('colorVariants');
        $struct['colors'] = [];
        if ($struct['colors_type'] !== 'none') {
            $productIds = $this->colorVariantIds;
            if ($productIds !== []) {
                $products = $this->productRepository->search(
                    (new Criteria($productIds))
                        ->addAssociation('prices')
                        ->addAssociation('cover')
                        ->addAssociation('properties.group'),
                    $this->salesChannelContext
                )->getEntities();
                foreach ($products as $product) {
                    /** @var SalesChannelProductEntity $product */

                    $prodProperties = $product->getProperties();
                    $colors = [];
                    $capacities = [];
                    foreach ($prodProperties as $property) {

                        $propertyGroupName = $property->getGroup()->getName();
                        if (($translation = $property->getGroup()->getTranslated()) && isset($translation['name'])) {
                            $propertyGroupName = $translation['name'];
                        }

                        $propertyName = $property->getName();
                        if (($translation = $property->getTranslated()) && isset($translation['name'])) {
                            $propertyName = $translation['name'];
                        }

                        if (in_array($propertyGroupName, $this->_colorNames())) {


                            $colors[] = $this->_filterColor($propertyName);
                        }
                        if (in_array($propertyGroupName, $this->_capacityNames())) {
                            $capacities[] = $this->_filterColor($propertyName);
                        }
                    }

                    $productName = $product->getName();
                    if (($translation = $product->getTranslated()) && isset($translation['name'])) {
                        $productName = $translation['name'];
                    }

                    $struct['colors'][] = [
                        'id'        => $product->getId(),
                        'color'     => implode(', ', $colors),
                        'capacity'  => implode(', ', $capacities),
                        'name'      => $productName,
                        'cover'     => $product->getCover(),
                        'current'   => false,
                        'delta'     => $this->_stringDelta($productName, $this->product->getName()),
                        'product'   => $product,
                        'shortName' => implode(', ', $colors)
                    ];
                }

                $colors = [];
                $capacities = [];
                foreach ($this->product->getProperties() as $property) {

                    $propertyGroupName = $property->getGroup()->getName();
                    if (($translation = $property->getGroup()->getTranslated()) && isset($translation['name'])) {
                        $propertyGroupName = $translation['name'];
                    }

                    $propertyName = $property->getName();
                    if (($translation = $property->getTranslated()) && isset($translation['name'])) {
                        $propertyName = $translation['name'];
                    }


                    if (in_array($propertyGroupName, $this->_colorNames())) {
                        $colors[] = $this->_filterColor($propertyName);
                    }
                    if (in_array($propertyGroupName, $this->_capacityNames())) {
                        $capacities[] = $this->_filterColor($propertyName);
                    }
                }


                $productName = $product->getName();
                if (($translation = $product->getTranslated()) && isset($translation['name'])) {
                    $productName = $translation['name'];
                }

                $current = [
                    'id'        => $this->product->getId(),
                    'color'     => implode(', ', $colors),
                    'capacity'  => implode(', ', $capacities),
                    'name'      => $productName,
                    'cover'     => $this->product->getCover(),
                    'current'   => true,
                    'delta'     => 0,
                    'product'   => $this->product,
                    'shortName' => implode(', ', $colors)
                ];

                //delete other capacities
                foreach ($struct['colors'] as $key => $item) {
                    if ($item['capacity'] !== $current['capacity']) {
                        unset($struct['colors'][$key]);
                    }
                }

                //unique colors (try to find same product version)
                $uniq = [];
                foreach ($struct['colors'] as $key => $item) {
                    $found = false;
                    foreach ($uniq as $k => $v) {
                        if ($v['color'] === $item['color']) {
                            if ($v['delta'] > $item['delta']) {
                                $uniq[$k] = $item;
                            }
                            $found = true;
                            break;
                        }
                    }
                    if (!$found) {
                        $uniq[] = $item;
                    }
                }
                $struct['colors'] = $uniq;

                if ($struct['colors'] !== []) {
                    $struct['colors'][] = $current;

                    usort($struct['colors'], fn($a, $b): int => $a['color'] <=> $b['color']);
                }
            }
        }

        $struct['capacities_type'] = $this->settingsService->getString('capacityVariants');
        $struct['capacities'] = [];
        if ($struct['capacities_type'] !== 'none') {
            $productIds = $this->capacityVariantIds;
            if ($productIds !== []) {
                $products = $this
                    ->productRepository
                    ->search(
                        (new Criteria($productIds))
                            ->addAssociation('cover')
                            ->addAssociation('properties.group'),
                        $this->salesChannelContext
                    )
                    ->getEntities();
                foreach ($products as $product) {
                    /** @var SalesChannelProductEntity $product */

                    $prodProperties = $product->getProperties();
                    $colors = [];
                    foreach ($prodProperties as $property) {

                        $propertyGroupName = $property->getGroup()->getName();
                        if (($translation = $property->getGroup()->getTranslated()) && isset($translation['name'])) {
                            $propertyGroupName = $translation['name'];
                        }

                        $propertyName = $property->getName();
                        if (($translation = $property->getTranslated()) && isset($translation['name'])) {
                            $propertyName = $translation['name'];
                        }

                        if (in_array($propertyGroupName, $this->_capacityNames())) {
                            $colors[] = $this->_filterColor($propertyName);
                        }
                    }

                    $productName = $product->getName();
                    if (($translation = $product->getTranslated()) && isset($translation['name'])) {
                        $productName = $translation['name'];
                    }

                    $struct['capacities'][] = [
                        'id'        => $product->getId(),
                        'capacity'  => implode(', ', $colors),
                        'name'      => $productName,
                        'cover'     => $product->getCover(),
                        'current'   => false,
                        'product'   => $product,
                        'shortName' => implode(', ', $colors)
                    ];
                }

                $colors = [];
                foreach ($this->product->getProperties() as $property) {

                    $propertyGroupName = $property->getGroup()->getName();
                    if (($translation = $property->getGroup()->getTranslated()) && isset($translation['name'])) {
                        $propertyGroupName = $translation['name'];
                    }

                    $propertyName = $property->getName();
                    if (($translation = $property->getTranslated()) && isset($translation['name'])) {
                        $propertyName = $translation['name'];
                    }

                    if (in_array($propertyGroupName, $this->_capacityNames())) {
                        $colors[] = $propertyName;
                    }
                }


                // hotfix: if product is null, we use an empty string
                if ($product) {
                    $productName = $product->getName();
                    if (($translation = $product->getTranslated()) && isset($translation['name'])) {
                        $productName = $translation['name'];
                    }
                } else {
                    $productName  = '';
                }


                $struct['capacities'][] = [
                    'id'        => $this->product->getId(),
                    'capacity'  => implode(', ', $colors),
                    'name'      => $productName,
                    'cover'     => $this->product->getCover(),
                    'current'   => true,
                    'product'   => $this->product,
                    'shortName' => implode(', ', $colors)
                ];

                usort($struct['capacities'], fn($a, $b): int => $a['capacity'] <=> $b['capacity']);
            }
        }

        $this->page->addExtension('product_variants', new ArrayEntity($struct));
    }


    /**
     * Adds 'feed_product_devices' extension to the product page if the product or its parent has associated devices.
     * DONE: the injected variable should be called TopdataTopFeedSW6_productHasDevices
     *
     */
    public function addProductHasDevices(): void
    {
        // Check if product devices tab should be shown
        if (!$this->settingsService->getBool('showProductDevicesTab')) {
            return;
        }

        // Check for devices associated with the current product
        $deviceExists = $this->_checkDeviceExists($this->product->getId());

        // If no devices found and product has a parent, check the parent product
        if (!$deviceExists && $this->product->getParentId()) {
            $deviceExists = $this->_checkDeviceExists($this->product->getParentId());
        }

        // If devices found, add the extension to the page - it is just used as a boolean flag in twig..
        $this->pageExtensionService->updateExtension([
            'productHasDevices' => $deviceExists
        ]);
    }

    /**
     * Adds information about other devices for product variants to the product page.
     *
     */
    public function addProductVariantOtherDevices(): void
    {
        if ($this->colorVariantIds) {
            $productIds = $this->topfeedHelperService->getProductIdsWithOtherDevices(
                $this->product->getId(),
                $this->colorVariantIds
            );
            $this->page->addExtension('productColorsOtherDevices', new ArrayEntity($productIds));
        }

        if ($this->capacityVariantIds) {
            $productIds = $this->topfeedHelperService->getProductIdsWithOtherDevices(
                $this->product->getId(),
                $this->capacityVariantIds
            );
            $this->page->addExtension('productCapacitiesOtherDevices', new ArrayEntity($productIds));
        }
    }

    /**
     * Returns an array of color-related property names.
     *
     * @return array
     */
    private function _colorNames(): array
    {
        return [
            'Farbe',
            'Color',
        ];
    }

    /**
     * IT DOES NOTHING. REMOVE THIS? OR FIX IT?
     *
     * Filters the color string.
     *
     * @param string $stringWithColor
     * @return string
     */
    private function _filterColor(string $stringWithColor): string
    {
        return $stringWithColor;
    }


    /**
     * Retrieves a collection of products based on given product IDs
     *
     * @param array $productIds
     * @param SalesChannelContext $salesChannelContext
     * @return ProductCollection|array
     */
    private function _getProductsCollection($productIds)
    {
        if (!$productIds) {
            return [];
        }
        $criteria = new Criteria($productIds);
        $this->eventDispatcher->dispatch(
            new AssociatedProductsCriteriaEvent($criteria, $this->salesChannelContext)
        );

//            $criteria->addAssociation('properties.group');
//            $criteria->addSorting(new FieldSorting('properties.group.name', FieldSorting::DESCENDING));
//            $criteria->addAssociation('manufacturer');
        $criteria->addSorting(new FieldSorting('manufacturer.name', FieldSorting::ASCENDING));
        $criteria->addSorting(new FieldSorting('name', FieldSorting::ASCENDING));

        return $this
            ->productRepository
            ->search($criteria, $this->salesChannelContext)
            ->getEntities();

    }

    /**
     * Calculates the difference between two strings.
     *
     * @param string $str1
     * @param string $str2
     * @return int
     */
    private function _stringDelta($str1, $str2): int
    {
        if (!is_string($str1) || !is_string($str2)) {
            return 0;
        }
        $str1 = trim($str1);
        $str2 = trim($str2);
        if ($str1 === $str2) {
            return 0;
        }
        $strlen1 = strlen($str1);
        $strlen2 = strlen($str2);
        if ($strlen1 === 0) {
            return $strlen2 ?: 0;
        }

        if ($strlen2 === 0) {
            return $strlen1 ?: 0;
        }

        $deltaStart = 0;
        $deltaEnd = 0;
        for ($i = 0; $i < $strlen1; $i++) {
            if ($i === $strlen1 - 1 && $strlen2 > $strlen1) {
                return $strlen2 - $strlen1;
            }

            if ($i === $strlen2 - 1 && $strlen1 > $strlen2) {
                return $strlen1 - $strlen2;
            }

            if ($str1[$i] !== $str2[$i]) {
                $deltaStart = $i;
                break;
            }
        }

        while (true) {
            $strlen1--;
            $strlen2--;
            if ($strlen1 < 0 && $strlen2 >= 0) {
                return $strlen2 - $strlen1;
            }
            if ($strlen2 < 0 && $strlen1 >= 0) {
                return $strlen1 - $strlen2;
            }
            if ($strlen2 < 0 && $strlen1 < 0) {
                return 0;
            }
            if ($str1[$strlen1] !== $str2[$strlen2]) {
                $deltaEnd = max($strlen1, $strlen2);
                break;
            }
        }

        return ($deltaStart > $deltaEnd) ? ($deltaStart + 1 - $deltaEnd) : ($deltaEnd + 1 - $deltaStart);
    }

    /**
     * Returns an array of capacity-related property names.
     *
     * @return array
     */
    private function _capacityNames(): array
    {
        return [
            'Kapazität (Zusatz)',
            'Kapazität',
            'Capacity',
        ];
    }

    /**
     * Checks if a device exists for a given product ID.
     *
     * @param string $productId
     * @return bool
     */
    private function _checkDeviceExists(string $productId): bool
    {
        $query = 'SELECT device_id FROM topdata_device_to_product WHERE product_id = 0x' . $productId . ' LIMIT 1';
        $result = $this->connection->executeQuery($query)->fetchAllAssociative();
        return count($result) > 0;
    }


}
