<?php declare(strict_types=1);

namespace Topdata\TopdataTopFeedSW6\Service;

use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\System\SalesChannel\Entity\SalesChannelRepository;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Topdata\TopdataTopFeedSW6\Storefront\Page\LinkedProductsPopup\LinkTypeEnum;
use Topdata\TopdataTopFeedSW6\Storefront\Page\LinkedProductsPopup\ProductIdRetriever;


/**
 * 11/2024 created (extracted from TopFeedApiController)
 */
class LinkedProductService
{


    public function __construct(
        private readonly ProductIdRetriever     $productIdRetriever,
        private readonly SalesChannelRepository $productRepository,
        private readonly SettingsService        $settingsService, // TODO: replace this with TopConfigRegistry
    )
    {
    }

    /**
     * 11/2024 moved from TopFeedApiController to here
     *
     * Retrieves linked products based on the specified link type and product ID.
     *
     * This private method is used internally to fetch different types of linked products
     * such as variants, alternate products, bundled products, etc.
     *
     * @param LinkTypeEnum $linkType The type of link (e.g., variants, alternate, bundled)
     * @param string $productId The ID of the main product
     * @param SalesChannelContext $salesChannelContext The current sales channel context
     * @param bool $limit Whether to limit the number of returned products
     * @return array An array containing the linked products and a flag indicating if there are more
     */
    public function getLinkedProducts(LinkTypeEnum $linkType, string $productId, SalesChannelContext $salesChannelContext, bool $limit = true): array
    {
        $productIds = $this->productIdRetriever->getProductIdsUsingSettings($linkType, $productId);

        if ($productIds === []) {
            return ['hasMore' => false, 'entities' => null];
        }

        $productsCount = count($productIds);
        if ($limit) {
            $limit = $this->settingsService->getInt('listVariantsLimit');
            if ($limit > 0) {
                $productIds = array_slice($productIds, 0, $limit);
            }
        }
        $hasMore = ($productsCount > count($productIds));

        return [
            'hasMore'  => $hasMore,
            'entities' => $this->productRepository->search(
                (new Criteria($productIds))->addAssociations([
                    'prices',
                    'properties.group'
                ]),
                $salesChannelContext
            )->getEntities()
        ];
    }

}
