import Plugin from 'src/plugin-system/plugin.class';
import DomAccess from 'src/helper/dom-access.helper';
import HttpClient from 'src/service/http-client.service';

/**
 * TopdataTopFeedLinkedProductsList Plugin
 *
 * This plugin handles the linked products list for the Topdata Top Feed.
 * It initializes the HTTP client, collects product IDs from the DOM,
 * sends them to the server, and updates the DOM with the server response.
 */
export default class TopdataTopFeedLinkedProductsList extends Plugin {
    static options = {
        /**
         * Selector of Item
         *
         * @type {string}
         */
        itemSelector: 'form[name="TopFeedListLinkedProducts"]',

        /**
         * Selector of ListItem
         *
         * @type {string}
         */
        itemListSelector: '[data-listLinkedProducts]',
    };

    /**
     * Initializes the plugin.
     * Sets up the HTTP client, collects product IDs, and sends them to the server.
     */
    init() {
        this._client = new HttpClient();
        let form = DomAccess.querySelector(document, this.options.itemSelector);

        let path = form.action;
        let data = { ids: [] };
        let listLinkedProducts = DomAccess.querySelectorAll(document, this.options.itemListSelector);

        listLinkedProducts.forEach((linkedProduct) => {
            let id = linkedProduct.dataset.productid;
            if (id) {
                data.ids.push(id);
            }
        });

        if (data.ids.length < 1) {
            return;
        }

        this._client.post(path, JSON.stringify(data), (responseText, request) => {
            if (request.status >= 400) {
                console.log(responseText);
            }
            try {
                const response = JSON.parse(responseText);
                if (response.success === true) {
                    response.products.forEach((k) => {
                        DomAccess.querySelector(document, '[data-listLinkedProducts][data-productid="' + k['id'] + '"]').innerHTML = k['html'];
                    });

                    let clickElements = document.querySelectorAll('.topdata-top-feed-show-more-variants');
                    if (clickElements.length > 0) {
                        clickElements.forEach((clickElement) => {
                            clickElement.addEventListener('click', this.loadShowMoreData.bind(this));
                        });
                    }
                } else {
                    console.log(response);
                }
            } catch (error) {
                console.log(error);
            }
        });
    }

    /**
     * Event handler for loading more data.
     *
     * @param {Event} event - The event object.
     */
    loadShowMoreData(event) {
        event.preventDefault();
        let button = event.currentTarget;
        let path = button.dataset.path;

        let container = DomAccess.querySelector(event.currentTarget.parentNode, '.topdata-top-feed-variants-tiles');

        this._client.get(path, (responseText, request) => {
            if (request.status >= 400) {
                console.log(responseText);
            }
            try {
                const response = JSON.parse(responseText);
                if (response.success === true) {
                    container.innerHTML = response.html;
                } else {
                    console.log(response);
                }
            } catch (error) {
                console.log(error);
            }
        });
    }
}