<?php declare(strict_types=1);

namespace Topdata\TopdataTopFeedSW6\Controller;

use Doctrine\DBAL\Connection;
use Shopware\Core\Content\Product\ProductCollection;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\NotFilter;
use Shopware\Core\Framework\Uuid\Uuid;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Controller\StorefrontController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Annotation\Route;
use Topdata\TopdataTopFeedSW6\Service\TopfeedHelperService;
use Topdata\TopdataTopFeedSW6\Service\SettingsService;
use Topdata\TopdataTopFeedSW6\Storefront\Page\CompatibleDevicesWidget\CompatibleDevicesWidgetLoader;
use Topdata\TopdataTopFeedSW6\Storefront\Page\LinkedProductsPopup\LinkedProductsPopupLoader;
use Topdata\TopdataTopFeedSW6\Storefront\Page\LinkedProductsPopup\LinkTypeEnum;
use Topdata\TopdataTopFeedSW6\Storefront\Page\LinkedProductsPopup\ProductIdRetriever;

#[Route(defaults: ['_routeScope' => ['storefront']])]
class TopFeedController extends StorefrontController
{


    public function __construct(
        private readonly LinkedProductsPopupLoader     $linkedProductsPopupLoader,
        private readonly SettingsService               $settingsService,
        private readonly CompatibleDevicesWidgetLoader $compatibleDevicesWidgetLoader,
        private readonly ProductIdRetriever            $productIdRetriever,
        private readonly TopfeedHelperService          $topfeedHelperService,
    )
    {
    }

    /**
     * TODO: the linkType and productId should be passed as parameters
     *
     * Retrieves linked products for a given product and link type.
     *
     * This method is used to fetch and render linked products in a popup.
     * It's typically called via AJAX to dynamically load related product information.
     *
     * @param Request $request The incoming request object
     * @param SalesChannelContext $salesChannelContext The current sales channel context
     * @return JsonResponse A JSON response containing the rendered HTML for the linked products
     */
    #[Route('/topdata-top-feed/products/{linkType}/{productId}',
        name: 'frontend.topdata_top_feed.products',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET'])
    ]
    public function productsLinked(Request $request, SalesChannelContext $salesChannelContext): JsonResponse
    {
        $productId = $request->get('productId');
        $linkType = $request->get('linkType');
        if (!Uuid::isValid($productId)) {
            return new JsonResponse(['success' => false, 'text' => 'Product id invalid']);
        }

        $result = ['success' => true];
        $page = $this->linkedProductsPopupLoader->buildLinkedProductsPopupPage(
            $request,
            $salesChannelContext,
            $productId,
            $linkType
        );

        $body = $this->renderStorefront('@TopdataTopFeedSW6/storefront/page/linked-products-popup/index.html.twig', [
            'page' => $page
        ])->getContent();
        $result['body'] = $body;
        $result['title'] = $page->pageTitle;

        $response = new JsonResponse($result);
        $response->headers->set("X-Robots-Tag", "noindex, nofollow");

        return $response;
    }


    /**
     * TODO: move this into a service
     *
     * Retrieves linked products based on the specified link type and product ID.
     *
     * This private method is used internally to fetch different types of linked products
     * such as variants, alternate products, bundled products, etc.
     *
     * @param LinkTypeEnum $linkType The type of link (e.g., variants, alternate, bundled)
     * @param string $productId The ID of the main product
     * @param SalesChannelContext $salesChannelContext The current sales channel context
     * @param bool $limit Whether to limit the number of returned products
     * @return array An array containing the linked products and a flag indicating if there are more
     */
    private function _getLinkedProducts(LinkTypeEnum $linkType, string $productId, SalesChannelContext $salesChannelContext, bool $limit = true): array
    {
        $productIds = $this->productIdRetriever->getProductIdsUsingSettings($linkType, $productId);

        if ($productIds === []) {
            return ['hasMore' => false, 'entities' => null];
        }

        $productsCount = count($productIds);
        if ($limit) {
            $limit = $this->settingsService->getInt('listVariantsLimit');
            if ($limit > 0) {
                $productIds = array_slice($productIds, 0, $limit);
            }
        }
        $hasMore = ($productsCount > count($productIds));

        $productRepository = $this->container->get('sales_channel.product.repository');
        return [
            'hasMore'  => $hasMore,
            'entities' => $productRepository->search(
                (new Criteria($productIds))->addAssociations([
                    'prices',
                    'properties.group'
                ]),
                $salesChannelContext
            )->getEntities()
        ];
    }

    /**
     * TODO: linkType and $productId should be passed as parameters
     *
     * Retrieves all linked products for a given product.
     *
     * This method is used to fetch and render all linked products without pagination.
     * It's typically used when displaying a full list of variants or related products.
     *
     * @param Request $request The incoming request object
     * @param SalesChannelContext $salesChannelContext The current sales channel context
     * @return JsonResponse A JSON response containing the rendered HTML for all linked products
     */
    #[Route('/topdata-top-feed/list-linked-products-all',
        name: 'frontend.topdata_top_feed.list_linked_products_all',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET']
    )]
    public function listLinkedProductsAll(Request $request, SalesChannelContext $salesChannelContext): JsonResponse
    {
        $productId = $request->get('productId');
        $linkType = $request->get('linkType');
        $linkTypeEnum = LinkTypeEnum::from($linkType);
        $data = '';
        $products = [];
        $products['variants'] = $this->_getLinkedProducts(
            $linkTypeEnum,
            $productId,
            $salesChannelContext,
            false
        );

        foreach ($products as $key => $value) {
            if (!$value['entities']) {
                unset($products[$key]);
                continue;
            }

            if (
                $linkTypeEnum === LinkTypeEnum::COLOR_VARIANTS
                || $linkTypeEnum === LinkTypeEnum::CAPACITY_VARIANTS
            ) {
                $linkedProductIds = $value['entities']->getIds();
                if (count($linkedProductIds) === 0) {
                    continue;
                }

                $products[$key]['otherDevices'] = $this->helperService->getProductIdsWithOtherDevices(
                    $this->container->get(Connection::class),
                    $productId,
                    $linkedProductIds
                );
            }
        }

        if ($products !== []) {
            $data = $this->renderStorefront('@TopdataTopFeedSW6/storefront/page/list-linked-products/all-variants.html.twig', [
                'products' => $products
            ])->getContent();
        }
        $response = new JsonResponse(['success' => true, 'html' => $data]);
        $response->headers->set("X-Robots-Tag", "noindex, nofollow");
        return $response;
    }


    /**
     * Retrieves compatible devices for a given product.
     *
     * This method is used to fetch and render a list of devices compatible with the specified product.
     * It's typically called via AJAX to dynamically load compatibility information.
     *
     * @param string $productid The ID of the product to find compatible devices for
     * @param Request $request The incoming request object
     * @param SalesChannelContext $salesChannelContext The current sales channel context
     * @return JsonResponse A JSON response containing the rendered HTML for compatible devices
     */
    #[Route('/topdata-top-feed/compatible-devices/{productid}',
        name: 'frontend.topdata_top_feed.compatible_devices',
        defaults: ['productid' => '', 'XmlHttpRequest' => true],
        methods: ['GET', 'POST']
    )]
    public function compatibleDevices(string $productid, Request $request, SalesChannelContext $salesChannelContext): JsonResponse
    {
        if (!$productid || !Uuid::isValid($productid)) {
            return new JsonResponse(['success' => false, 'html' => 'Something went wrong']);
        }

        $result = ['success' => true];
        $page = $this->compatibleDevicesWidgetLoader->load($request, $salesChannelContext, $productid);
        $result['html'] = $this->renderStorefront('@TopdataTopFeedSW6/storefront/page/product-detail/feed-devices-content.html.twig', [
            'page' => $page
        ])->getContent();

        $response = new JsonResponse($result);
        $response->headers->set("X-Robots-Tag", "noindex, nofollow");
        return $response;
    }


    /**
     * TODO: move some of this to a service
     *
     * Retrieves an optimized list of linked products for multiple products.
     *
     * This method is used to fetch and render linked products for multiple products in an optimized manner.
     * It's designed to handle bulk requests efficiently, reducing database queries and improving performance.
     *
     * @param Request $request The incoming request object
     * @param SalesChannelContext $salesChannelContext The current sales channel context
     * @return JsonResponse A JSON response containing the rendered HTML for linked products of multiple products
     */
    #[Route('/topdata-top-feed/list-linked-products',
        name: 'frontend.topdata_top_feed.list_linked_products',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET', 'POST']
    )]
    public function listLinkedProductsOptimized(Request $request, SalesChannelContext $salesChannelContext): JsonResponse
    {
        $limit = $this->settingsService->getInt('listVariantsLimit');
        $productIds = $request->get('ids');
        $data = [];
        $linkedProducts = [];
        $allLinkedProdIds = [];

        if ($this->settingsService->getBool('listHideForVariated') && is_array($productIds)) {
            $realIds = [];
            foreach ($productIds as $id) {
                if (Uuid::isValid($id)) {
                    $realIds[] = $id;
                }
            }
            $realIds = array_unique($realIds);
            $criteria = new Criteria($realIds);
            $criteria->addFilter(
                new NotFilter(NotFilter::CONNECTION_OR, [new EqualsFilter('parentId', null)])
            );
            $productVariations = $this
                ->container
                ->get('sales_channel.product.repository')
                ->search(
                    $criteria,
                    $salesChannelContext
                )
                ->getEntities();
        } else {
            $productVariations = new ProductCollection();
        }

        foreach ($productIds as $id) {
            $products = [];

            foreach ($this->settingsService->getLinkedSettings() as $key => $enabled) {
                $linkTypeEnum = LinkTypeEnum::from($key);
                if (($linkTypeEnum === LinkTypeEnum::COLOR_VARIANTS
                        ||
                        $linkTypeEnum === LinkTypeEnum::CAPACITY_VARIANTS
                    )
                    && $productVariations->get($id)
                ) {
                    continue;
                }
                if ($enabled) {
                    $ids = $this->productIdRetriever->getLinkedProductIdsByLinkType($linkTypeEnum, $id);
                    if (count($ids) === 0) {
                        continue;
                    }
                    $hasMore = $limit && (count($ids) > $limit);
                    if ($limit !== 0) {
                        $ids = array_slice($ids, 0, $limit);
                    }
                    $products[$key] = [
                        'ids'          => $ids,
                        'entities'     => [],
                        'otherDevices' => [],
                        'hasMore'      => $hasMore
                    ];
                    $allLinkedProdIds = array_merge($allLinkedProdIds, $ids);
                }
            }

            foreach ($products as $key => $value) {
                if (
                    $key === LinkTypeEnum::COLOR_VARIANTS->value
                    || $key === LinkTypeEnum::CAPACITY_VARIANTS->value
                ) {
                    $products[$key]['otherDevices'] = $this->topfeedHelperService->getProductIdsWithOtherDevices(
                        $id,
                        $value['ids']
                    );
                }
            }

            if ($products !== []) {
                $linkedProducts[$id] = $products;
            }
        }

        $allLinkedProdIds = array_unique($allLinkedProdIds);

        if ($allLinkedProdIds !== []) {
            $allLinkedProducts = $this
                ->container
                ->get('sales_channel.product.repository')
                ->search(
                    (new Criteria($allLinkedProdIds))->addAssociations(['prices', 'properties.group', 'manufacturer']),
                    $salesChannelContext
                )
                ->getEntities();
        } else {
            $allLinkedProducts = new ProductCollection();
        }


        foreach ($linkedProducts as $id => $linkedInfo) {
            foreach ($linkedInfo as $linkType => $info) {
                $prods = [];
                foreach ($info['ids'] as $lid) {
                    if ($allLinkedProducts->get($lid)) {
                        $prods[] = $allLinkedProducts->get($lid);
                    }
                }
                if ($prods !== []) {
                    $linkedProducts[$id][$linkType]['entities'] = new ProductCollection($prods);
                }
            }

        }

        foreach ($linkedProducts as $id => $linkedInfo) {
            $data[] = [
                'id'   => $id,
                'html' => $this->renderStorefront('@TopdataTopFeedSW6/storefront/page/list-linked-products/index.html.twig', [
                    'products'        => $linkedInfo,
                    'parentProductId' => $id
                ])->getContent()
            ];
        }

        return new JsonResponse(['success' => true, 'products' => $data]);
    }


}
