<?php


namespace Topdata\TopdataProductBoxesSW6\Subscriber;

use Shopware\Core\Content\Media\Aggregate\MediaFolder\MediaFolderDefinition;
use Shopware\Core\Content\Product\Events\ProductListingResultEvent;
use Shopware\Core\Content\Product\Events\ProductSearchResultEvent;
use Shopware\Core\Content\Product\ProductEntity;
use Shopware\Core\Content\Product\SalesChannel\SalesChannelProductEntity;
use Shopware\Core\Content\ProductStream\ProductStreamDefinition;
use Shopware\Core\Content\Property\PropertyGroupEntity;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Event\EntitySearchedEvent;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\System\SalesChannel\Entity\SalesChannelEntitySearchResultLoadedEvent;
use Shopware\Core\System\SalesChannel\Event\SalesChannelProcessCriteriaEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Topdata\TopdataConnectorSW6\Core\Content\Product\ProductDefinition;
use Topdata\TopdataProductBoxesSW6\Manager\PropertyGroupManager;


/**
 * 08/2023 created
 */
class MyEventSubscriber implements EventSubscriberInterface
{

    const ENTITY_VAR_NAME__BOX_PROPERTIES = 'TopdataProductBoxesSW6_boxProperties';

    public function __construct(
        private readonly PropertyGroupManager $propertyGroupManager,
        private readonly EntityRepository     $propertyGroupRepository
    )
    {
    }


    public static function getSubscribedEvents(): array
    {
        return [
            ProductListingResultEvent::class             => 'onProductListingResult',
            ProductSearchResultEvent::class              => 'onProductSearchResult',
            'sales_channel.product.process.criteria'     => 'onProductCriteria',
            'sales_channel.product.search.result.loaded' => 'onSalesChannelEntitySearchResultLoaded',

        ];
    }


    public function onProductCriteria(SalesChannelProcessCriteriaEvent $event): void
    {
        $event->getCriteria()->addAssociation('properties');
    }


    public function onProductListingResult(ProductListingResultEvent $event)
    {
        $categoryId = $event->getResult()->getCurrentFilter('navigationId');
        if (!$categoryId) {
            return;
        }

        $this->_addCategorySpecificPropertiesToEntities([$categoryId], $event->getResult()->getElements(), $event->getContext());

    }


    public function onProductSearchResult(ProductSearchResultEvent $event)
    {
        /** @var SalesChannelProductEntity $element */
        foreach ($event->getResult()->getElements() as $element) {
            $this->_addCategorySpecificPropertiesToEntities($element->getCategoryIds(), [$element], $event->getContext());
        }
    }


    /**
     * private helper method
     */
    private function _getProperties(array $mapGroupIdToGroupName, SalesChannelProductEntity $salesChannelProductEntity, ?array $propertyGroupIds)
    {
        $ret = [];
        foreach ($propertyGroupIds as $propertyGroupId) {
            $filtered = $salesChannelProductEntity->getProperties()->filterByGroupId($propertyGroupId);

            $values = $filtered->map(fn($property) => $property->getName());

            $ret[] = [
                'propertyGroupId' => $propertyGroupId,
                'name'            => $mapGroupIdToGroupName[$propertyGroupId],
                'value'           => empty($values) ? null : implode(', ', $values),
            ];
        }

        return $ret;
    }

    /**
     * private helper
     */
    private function _addCategorySpecificPropertiesToEntities(?array $categoryIds, array $salesChannelProductEntities, Context $context): void
    {
        if (empty($categoryIds)) {
            return;
        }
        // ---- make sure to not add properties twice
        $filtered = array_filter($salesChannelProductEntities, fn(SalesChannelProductEntity $salesChannelProductEntity) => !property_exists($salesChannelProductEntity, self::ENTITY_VAR_NAME__BOX_PROPERTIES));
        if (empty($filtered)) {
            return;
        }

        // ---- fetch propertyGroup Ids
        $propertyGroupIds = [];
        foreach ($categoryIds as $categoryId) {
            $x = $this->propertyGroupManager->getPropertyGroupIdsByCategoryId($categoryId, $context);
            if ($x) {
                array_push($propertyGroupIds, ...$x);
            }
        }
        $propertyGroupIds = array_unique($propertyGroupIds);
        // TODO: we could implement some sorting here by priority (if more than one categoryId is given)
        if (empty($propertyGroupIds)) {
            return;
        }

        // ----- fetch propertyGroup Names
        $criteria = new Criteria($propertyGroupIds);
        $groups = $this->propertyGroupRepository->search($criteria, $context);
        $mapGroupIdToGroupName = $groups->map(fn(PropertyGroupEntity $propertyGroup) => $propertyGroup->getName());

        /** @var SalesChannelProductEntity $salesChannelProductEntity */
        foreach ($filtered as $salesChannelProductEntity) {
            $salesChannelProductEntity->{self::ENTITY_VAR_NAME__BOX_PROPERTIES} = $this->_getProperties($mapGroupIdToGroupName, $salesChannelProductEntity, $propertyGroupIds);
        }
    }


    public function onSalesChannelEntitySearchResultLoaded(SalesChannelEntitySearchResultLoadedEvent $event)
    {
        /** @var SalesChannelProductEntity $element */
        foreach ($event->getResult()->getElements() as $element) {
            $this->_addCategorySpecificPropertiesToEntities($element->getCategoryIds(), [$element], $event->getContext());
        }

    }

}
