<?php


namespace Topdata\TopdataProductBoxesSW6\Manager;


use Shopware\Core\Content\Category\CategoryEntity;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Topdata\TopdataConnectorSW6\Core\Content\Category\TopdataCategoryExtension\TopdataCategoryExtensionEntity;
use Topdata\TopdataProductBoxesSW6\Entity\Extension\TopdataPbCategoryExtension\TopdataPbCategoryExtensionEntity;
use Topdata\TopdataProductBoxesSW6\Entity\TopdataPbCategoryExtensionToPropertyGroup\TopdataPbCategoryExtensionToPropertyGroupCollection;

/**
 * 08/2023 created
 */
class PropertyGroupManager
{

    public function __construct(
        private readonly EntityRepository $categoryRepository,
        private readonly EntityRepository $topdataPbCategoryExtensionRepository,
        private readonly EntityRepository $topdataPbCategoryExtensionToPropertyGroupRepository,
    )
    {
    }


    /**
     * recursive method
     *
     * called when in a category page, eg "http://myshop.de/Bueroartikel/Versandtaschen/"
     * 08/2023 created
     *
     * @param string $categoryId
     * @return ?string[] ids of property groups, sorted already,
     *                  or null if no property groups should be shown in the product boxes of this category
     */
    public function getPropertyGroupIdsByCategoryId(string $categoryId, Context $context): ?array
    {
        assert(!empty($categoryId));

        // ---- traverse the category tree upwards until we find a category with a topdataPbCategoryExtension
        $criteria = new Criteria([$categoryId]);
        $criteria->addAssociation('topdataPbCategoryExtension.propertyGroupXrefs');

        /** @var CategoryEntity $category */
        $category = $this->categoryRepository->search($criteria, $context)->first();

        /** @var TopdataPbCategoryExtensionEntity $ext */
        $ext = $category->getExtension('topdataPbCategoryExtension');
        if ($ext) {
            if (!$ext->getIsEnabled()) {
                return null;
            }

            // ---- parent recursive
            if ($ext->getUseParentCategory()) {
                if (!$category->getParentId()) {
                    return null;
                }

                return $this->getPropertyGroupIdsByCategoryId($category->getParentId(), $context);
            }

            // ---- extract and return propertyGroupIds
            $xrefs = $ext->getPropertyGroupXrefs();
            $xrefs->sortByPosition();
            $productGroupIds = $xrefs->map(fn($xref) => $xref->getPropertyGroupId());

            return empty($productGroupIds) ? null : array_values($productGroupIds);
        } else {
            // ---- no ext - try parent
            if (!$category->getParentId()) {
                return null;
            }

            return $this->getPropertyGroupIdsByCategoryId($category->getParentId(), $context);
        }
    }


    /**
     * 08/2023 created
     *
     * @param string $topdataPbCategoryExtensionId
     * @param array $propertyGroupsFormData
     * @param Context $context
     * @return void
     */
    public function updatePropertyGroupsOfTopdataPbCategoryExtension(string $topdataPbCategoryExtensionId, array $propertyGroupsFormData, Context $context): void
    {

        // ---- delete old
        $this->_deleteOldPropertyGroupXrefs($topdataPbCategoryExtensionId, $context);

        // ---- create new propertyGroupXrefs
        $payload = $this->_buildNewPropertyGroupXrefsPayload($propertyGroupsFormData, $topdataPbCategoryExtensionId);
        $this->topdataPbCategoryExtensionToPropertyGroupRepository->create($payload, $context);
    }


    /**
     * deletes existing topdataPbCategoryExtension items and their relations to colors
     *
     * 08/2023 created
     *
     * @param string $topdataPbCategoryExtensionId
     * @param Context $context
     * @return void
     */
    private function _deleteOldPropertyGroupXrefs(string $topdataPbCategoryExtensionId, Context $context): void
    {
        $criteria = new Criteria();
        $criteria->addFilter(new EqualsFilter('id', $topdataPbCategoryExtensionId));
        $criteria->addAssociation('propertyGroupXrefs');
        /** @var TopdataPbCategoryExtensionEntity $topdataPbCategoryExtensionEntity */
        $topdataPbCategoryExtensionEntity = $this->topdataPbCategoryExtensionRepository->search($criteria, $context)->first();
        $oldPropertyGroupXrefsIds = array_values($topdataPbCategoryExtensionEntity->getPropertyGroupXrefs()->getIds());

        // ---- delete old topdataPbCategoryExtension item to color relations
        $this->topdataPbCategoryExtensionToPropertyGroupRepository->delete(array_map(fn($id) => ['id' => $id], $oldPropertyGroupXrefsIds), $context);
    }

    /**
     * 08/2023 created
     *
     * @param array $propertyGroupsFormData
     * @param string $topdataPbCategoryExtensionId
     * @return array
     */
    private function _buildNewPropertyGroupXrefsPayload(array $propertyGroupsFormData, string $topdataPbCategoryExtensionId): array
    {
        $position = 1;
        return array_map(function ($x) use ($topdataPbCategoryExtensionId, &$position) {
            return [
                'topdataPbCategoryExtensionId' => $topdataPbCategoryExtensionId,
                'propertyGroupId'              => $x['id'],
                'position'                     => $position++,
            ];
        }, $propertyGroupsFormData);
    }

}
