<?php


namespace Topdata\TopdataPrivateShopSW6\Subscriber;

use Shopware\Core\Framework\Validation\BuildValidationEvent;
use Shopware\Core\System\SystemConfig\SystemConfigService;
use Shopware\Storefront\Framework\Routing\StorefrontResponse;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Topdata\TopdataPrivateShopSW6\Constants\PrivateShopModeConstants;
use Shopware\Core\Framework\Validation\Exception\ConstraintViolationException;
use Symfony\Component\Validator\ConstraintViolation;
use Symfony\Component\Validator\ConstraintViolationList;

/**
 * Listens for storefront response events and redirects to the login page based on the private shop configuration.
 * Also handles customer registration validation.
 *
 * 08/2023 created
 */
class MyEventSubscriber implements EventSubscriberInterface
{
    private SystemConfigService $systemConfigService;
    private UrlGeneratorInterface $router;

    public function __construct(
        SystemConfigService   $systemConfigService,
        UrlGeneratorInterface $router
    )
    {
        $this->systemConfigService = $systemConfigService;
        $this->router = $router;
    }

    /**
     * Returns an array of event names this subscriber wants to listen to.
     *
     * @return array The event names to listen to.
     */
    public static function getSubscribedEvents(): array
    {
        return [
            ResponseEvent::class                   => 'onKernelResponse',
            'framework.validation.customer.create' => 'onCustomerCreate',
        ];
    }

    /**
     * Handles the kernel response event to redirect to the login page if the private shop mode is enabled
     * and the user is not logged in.
     *
     * 09/2023 created
     */
    public function onKernelResponse(ResponseEvent $event): void
    {
        $privateShopMode = $this->systemConfigService->get('TopdataPrivateShopSW6.config.privateShopMode');

        // ---- Check if the private shop mode is enabled. If not, return early.
        if (!in_array($privateShopMode, [PrivateShopModeConstants::HIDE_PRODUCTS, PrivateShopModeConstants::HIDE_PRODUCTS_AND_CATEGORIES])) {
            return;
        }

        $response = $event->getResponse();

        // ---- Check if the response is a StorefrontResponse. If not, return early.
        if (!$response instanceof StorefrontResponse) {
            return;
        }

        // ---- Check if the response is a RedirectResponse. If so, return early.
        if ($response instanceof RedirectResponse) {
            return;
        }

        // ---- Check if a customer is logged in. If so, return early.
        $customer = $response->getContext()?->getCustomer();
        if ($customer) {
            return;
        }

        $routeName = $event->getRequest()->attributes->get('_route');
        $routeParams = $event->getRequest()->attributes->get('_route_params');

        // ---- Most strict private shop mode: hide products and categories
        if (PrivateShopModeConstants::hideCategories($privateShopMode) && in_array($routeName, [
                // protected routes:
                'frontend.navigation.page' // category page
            ])) {
            $event->setResponse($this->_getRedirectToLoginResponse($routeName, $routeParams));
            return;
        }

        // ---- 2nd strict private shop mode: hide products (but not categories)
        if (PrivateShopModeConstants::hideProducts($privateShopMode) && in_array($routeName, [
                // protected routes:
                'frontend.detail.page', // product page
                'frontend.search.page', // search result page
            ])) {
            $event->setResponse($this->_getRedirectToLoginResponse($routeName, $routeParams));
            return;
        }
    }

    /**
     * Generates a redirect response to the login page, including the current route and parameters as redirect parameters.
     *
     * 09/2023 created
     *
     * @param string $routeName The name of the route to redirect to after login.
     * @param array $routeParams The parameters of the route to redirect to after login.
     * @return RedirectResponse The redirect response to the login page.
     */
    private function _getRedirectToLoginResponse(string $routeName, $routeParams): RedirectResponse
    {
        $loginPage = $this->router->generate('frontend.account.login.page', [
            'redirectTo'         => $routeName,
            'redirectParameters' => \json_encode($routeParams),
        ], UrlGeneratorInterface::ABSOLUTE_URL);

        return new RedirectResponse($loginPage);
    }

    /**
     * Handles the customer create validation event to disable customer registration if configured.
     *
     * 12/2023 created
     *
     * @param BuildValidationEvent $event The validation event.
     */
    public function onCustomerCreate(BuildValidationEvent $event): void
    {
        $disableRegistration = $this->systemConfigService->get('TopdataPrivateShopSW6.config.disableRegistration');
        if ($disableRegistration) {
            $violation = new ConstraintViolation(
                'Customer registration is disabled by plugin configuration.',
                '',
                [],
                '',
                '/',
                null,
                null,
                'REGISTRATION_DISABLED'
            );

            $violationList = new ConstraintViolationList([$violation]);
            throw new ConstraintViolationException($violationList, $event->getData()->all());
        }
    }
}