<?php declare(strict_types=1);

namespace Topdata\TopdataLinkedOemRemSW6\Twig;

use Psr\Log\LoggerInterface;
use Shopware\Core\Checkout\Cart\Cart;
use Shopware\Core\Checkout\Cart\LineItem\LineItem;
use Shopware\Core\Content\Product\SalesChannel\SalesChannelProductEntity;
use Shopware\Core\System\Currency\CurrencyFormatter;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Core\System\SystemConfig\SystemConfigService;
use Topdata\TopdataLinkedOemRemSW6\Constants\SavingsDisplayModeConstants;
use Topdata\TopdataLinkedOemRemSW6\DTO\SavingsOverOemDTO;
use Topdata\TopdataLinkedOemRemSW6\Service\OemRemProductFinderService;
use Twig\Extension\AbstractExtension;
use Twig\TwigFunction;

/**
 * This Twig extension provides functions to determine and display matching REM products for OEM products in the cart,
 * as well as to decide whether to show savings badges or alternative product popups based on configured thresholds.
 */
class MyTwigExtension extends AbstractExtension
{
    public function __construct(
        private readonly OemRemProductFinderService $productFinderService,
        private readonly SystemConfigService        $systemConfigService,
        private readonly LoggerInterface            $logger,
        private readonly CurrencyFormatter          $currencyFormatter
    )
    {
    }

    /**
     * Returns a list of Twig functions provided by this extension.
     *
     * @return array<TwigFunction>
     */
    public function getFunctions(): array
    {
        return [
            new TwigFunction('TopdataLinkedOemRemSW6_getMatchingRemProductsForOemProductsInCart', [$this, 'getMatchingRemProductsForOemProductsInCart'], ['needs_context' => true]),
            new TwigFunction('TopdataLinkedOemRemSW6_shouldShowSavingsBadge', [$this, 'TopdataLinkedOemRemSW6_shouldShowSavingsBadge'], ['needs_context' => true]),
            new TwigFunction('TopdataLinkedOemRemSW6_topdataShouldShowAlternativePopup', [$this, 'TopdataLinkedOemRemSW6_topdataShouldShowAlternativePopup'], ['needs_context' => true]),
            new TwigFunction('TopdataLinkedOemRemSW6_getFormattedSavingsString', [$this, 'getFormattedSavingsString'], ['needs_context' => true]),
        ];
    }

    /**
     * NEW: Generates the fully formatted HTML string for the savings badge.
     *
     * @param array $twigContext
     * @param SalesChannelProductEntity $product
     * @return string
     */
    public function getFormattedSavingsString(array $twigContext, SalesChannelProductEntity $product): string
    {
        /** @var SalesChannelContext $salesChannelContext */
        $salesChannelContext = $twigContext['context'];
        $salesChannelId = $salesChannelContext->getSalesChannelId();

        /** @var SavingsOverOemDTO|null $savings */
        $savings = $product->getExtension('TopdataLinkedOemRemSW6')?->get('savingsOverOemDTO');

        if (!$savings) {
            return '';
        }

        // Determine which saving values to use for display
        $savingValue = $savings->getPriceSavingAbsolute();
        $savingPercent = $savings->getPriceSavingPercentAbsolute();
        $isNormalized = false;
        if ($savings->wasNormalized && $savings->normalizedPriceSavingAbsolute > 0) {
            $savingValue = $savings->normalizedPriceSavingAbsolute;
            $savingPercent = $savings->normalizedPriceSavingPercent;
            $isNormalized = true;
        }

        $savingsDisplayMode = $this->systemConfigService->getString(
            'TopdataLinkedOemRemSW6.config.savingsDisplayMode',
            $salesChannelId
        ) ?: SavingsDisplayModeConstants::MONEY_NOT_ROUNDED;

        $formattedString = '';

        switch ($savingsDisplayMode) {
            case SavingsDisplayModeConstants::MONEY_NOT_ROUNDED:
            case SavingsDisplayModeConstants::MONEY_ROUNDED_0:
                $decimals = ($savingsDisplayMode === SavingsDisplayModeConstants::MONEY_ROUNDED_0) ? 0 : 2;
                $formattedString = $this->currencyFormatter->formatCurrencyByLanguage(
                    $savingValue,
                    $salesChannelContext->getCurrency()->getIsoCode(),
                    $salesChannelContext->getContext()->getLanguageId(),
                    $salesChannelContext->getContext(),
                    $decimals
                );
                break;

            case SavingsDisplayModeConstants::PERCENT_ROUNDED_0:
            case SavingsDisplayModeConstants::PERCENT_ROUNDED_1:
                $decimals = ($savingsDisplayMode === SavingsDisplayModeConstants::PERCENT_ROUNDED_0) ? 0 : 1;
                $formattedString = number_format($savingPercent, $decimals, ',', '.') . '%';
                break;
        }

        if ($isNormalized) {
            $formattedString .= '<span class="topdata-savings-normalizer">*</span>';
        }

        return $formattedString;
    }


    /**
     * Decides if the savings badge should be shown for a REM product.
     * This is based on the 'Savings Display Threshold' config.
     *
     * @param array $twigContext
     * @param SalesChannelProductEntity $product
     * @return bool
     */
    public function TopdataLinkedOemRemSW6_shouldShowSavingsBadge(array $twigContext, SalesChannelProductEntity $product): bool
    {
        /** @var SavingsOverOemDTO|null $dto */
        $dto = $product->getExtension('TopdataLinkedOemRemSW6')?->get('savingsOverOemDTO');

        return $this->_isThresholdMet($twigContext, $dto, 'savingsThreshold');
    }

    /**
     * Decides if the alternative popup should be triggered for an OEM product.
     * This is based on the 'Popup Savings Threshold' config.
     *
     * @param array $twigContext
     * @param SalesChannelProductEntity $product
     * @return bool
     */
    public function TopdataLinkedOemRemSW6_topdataShouldShowAlternativePopup(array $twigContext, SalesChannelProductEntity $product): bool
    {
        /** @var SavingsOverOemDTO|null $dto */
        $dto = $product->getExtension('topdata_oem_rem_data')?->get('bestAlternativeDto');

        return $this->_isThresholdMet($twigContext, $dto, 'popupSavingsThreshold');
    }

    /**
     * Generic helper to check if a saving DTO meets a configured threshold.
     *
     * @param array $twigContext
     * @param SavingsOverOemDTO|null $dto
     * @param string $configPrefix
     * @return bool
     */
    private function _isThresholdMet(array $twigContext, ?SavingsOverOemDTO $dto, string $configPrefix): bool
    {
        if (!$dto) {
            return false;
        }

        // ---- Get Sales Channel Context and ID
        /** @var SalesChannelContext $salesChannelContext */
        $salesChannelContext = $twigContext['context'];
        $salesChannelId = $salesChannelContext->getSalesChannelId();

        // ---- Retrieve Threshold Values from System Configuration
        $thresholdValue = $this->systemConfigService->get('TopdataLinkedOemRemSW6.config.' . $configPrefix . 'Value', $salesChannelId) ?? 0.0;
        $thresholdType = $this->systemConfigService->getString('TopdataLinkedOemRemSW6.config.' . $configPrefix . 'Type', $salesChannelId) ?: 'absolute';

        // ---- Determine Saving Values (Absolute and Percentage)
        $savingValue = $dto->getPriceSavingAbsolute();
        $savingPercent = $dto->getPriceSavingPercentAbsolute();
        if ($dto->wasNormalized && $dto->normalizedPriceSavingAbsolute > 0) {
            $savingValue = $dto->normalizedPriceSavingAbsolute;
            $savingPercent = $dto->normalizedPriceSavingPercent;
        }

        // ---- Debug logging using Shopware's logger
        $this->logger->debug('Topdata Threshold Debug', [
            'thresholdValue' => $thresholdValue,
            'thresholdType'  => $thresholdType,
            'savingValue'    => $savingValue,
            'configPrefix'   => $configPrefix,
            'salesChannelId' => $salesChannelId,
            'meetsThreshold' => ($thresholdType === 'absolute') ? ($savingValue >= $thresholdValue) : ($savingPercent >= $thresholdValue)
        ]);
        if ($savingValue <= 0) {
            return false;
        }

        // ---- Check if Threshold is Met Based on Type
        if ($thresholdType === 'absolute') {
            return $savingValue >= $thresholdValue;
        }

        if ($thresholdType === 'percentage') {
            return $savingPercent >= $thresholdValue;
        }

        return false;
    }

    /**
     * Retrieves matching REM products for OEM products in the cart.
     *
     * @param array $twigContext
     * @param Cart $cart
     * @return array
     */
    public function getMatchingRemProductsForOemProductsInCart(array $twigContext, Cart $cart): array
    {
        // ---- Extract Product IDs from Cart
        $productIds = $cart->getLineItems()->filterType(LineItem::PRODUCT_LINE_ITEM_TYPE)->getReferenceIds();
        /** @var SalesChannelContext $salesChannelContext */
        $salesChannelContext = $twigContext['context'];

        // ---- Find Matching REM Products
        return $this->productFinderService->findMatchingRemProductsFlat($productIds, $salesChannelContext);
    }
}