<?php declare(strict_types=1);

namespace Topdata\TopdataLinkedOemRemSW6\Subscriber;

use Shopware\Core\Content\Product\SalesChannel\SalesChannelProductEntity;
use Shopware\Core\Defaults;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\Struct\ArrayEntity;
use Shopware\Core\System\SalesChannel\Entity\SalesChannelRepository;
use Shopware\Storefront\Page\Product\ProductPageLoadedEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Topdata\TopdataLinkedOemRemSW6\Service\OemRemSavingsService;

class CheckForRemAlternativesSubscriber implements EventSubscriberInterface
{
    public function __construct(
        private readonly SalesChannelRepository $productRepository,
        private readonly OemRemSavingsService   $oemRemSavingsService,
    ) {
    }

    public static function getSubscribedEvents(): array
    {
        return [
            ProductPageLoadedEvent::class => 'onProductPageLoaded',
        ];
    }

    /**
     * Enhances the OEM product on its detail page with data about its best REM alternative.
     */
    public function onProductPageLoaded(ProductPageLoadedEvent $event): void
    {
        $product = $event->getPage()->getProduct();

        // Ensure properties are loaded on the product for accurate savings calculation
        $criteria = new Criteria([$product->getId()]);
        $criteria->addAssociation('properties.group');
        $loadedProduct = $this->productRepository->search($criteria, $event->getSalesChannelContext())->first();

        if ($loadedProduct) {
            // Find the single best alternative
            $bestRemDTOs = $this->oemRemSavingsService->findBestRemAlternatives(
                $loadedProduct,
                $event->getSalesChannelContext(),
                $event->getSalesChannelContext()->getCurrencyId(),
                1 // We only need the best one
            );
            
            $bestDto = !empty($bestRemDTOs) ? $bestRemDTOs[0] : null;

            // Add the DTO to the extension, even if it's null.
            $product->addExtension('topdata_oem_rem_data', new ArrayEntity(['bestAlternativeDto' => $bestDto]));
        }
    }
}
