<?php declare(strict_types=1);

namespace Topdata\TopdataLinkedOemRemSW6\Service;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Exception;
use Shopware\Core\Content\Product\SalesChannel\SalesChannelProductEntity;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\EntitySearchResult;
use Shopware\Core\System\SalesChannel\Entity\SalesChannelRepository;
use Shopware\Core\System\SalesChannel\SalesChannelContext;

/**
 * Service for retrieving SalesChannelProductEntity objects for OEM or REM products
 * based on the links stored in the topdata_lor_oem_to_rem table.
 *
 * 05/2025 created as part of OemRemService refactoring
 */
class OemRemProductFinderService
{
    public function __construct(
        private readonly SalesChannelRepository $productRepository,
        private readonly Connection $connection
    ) {
    }

    /**
     * Builds a mapping from OEM product IDs to their corresponding REM product IDs.
     *
     * Example output:
     * [ '0012345678' => [ '12345678', '12345679' ] ]
     * Where '0012345678' is an OEM product ID and the array contains all matching REM product IDs.
     *
     * @param array $oemProductIds Array of OEM product IDs to find matches for
     * @return array A map of OEM product IDs to arrays of REM product IDs
     * @throws \Doctrine\DBAL\Exception If the database query fails
     */
    private function _buildMapOemProductIdToRemProductIds(array $oemProductIds): array
    {
        if (empty($oemProductIds)) {
            return [];
        }

        // ---- find matching REM product ids
        $SQL = "    
                    SELECT 
                        LOWER(HEX(rem_product_id)) AS remProductId, 
                        LOWER(HEX(oem_product_id)) AS oemProductId 
                    FROM topdata_lor_oem_to_rem 
                    WHERE oem_product_id IN (:oemProductIds)";
        $rows = $this->connection->fetchAllAssociative($SQL, ['oemProductIds' => array_map('hex2bin', $oemProductIds)], ['oemProductIds' => Connection::PARAM_STR_ARRAY]);

        if (empty($rows)) {
            return [];
        }

        // ---- build Lookup map
        $mapOemProductIdToRemProductIds = [];
        foreach ($rows as $row) {
            if (!isset($mapOemProductIdToRemProductIds[$row['oemProductId']])) {
                $mapOemProductIdToRemProductIds[$row['oemProductId']] = [];
            }
            $mapOemProductIdToRemProductIds[$row['oemProductId']][] = $row['remProductId'];
        }

        return $mapOemProductIdToRemProductIds;
    }

    /**
     * Builds a mapping from REM product IDs to their corresponding OEM product IDs.
     *
     * Example output:
     * [ '0012345678' => [ '12345678', '12345679' ] ]
     * Where '0012345678' is a REM product ID and the array contains all matching OEM product IDs.
     *
     * @param array $remProductIds Array of REM product IDs to find matches for
     * @return array A map of REM product IDs to arrays of OEM product IDs
     * @throws \Doctrine\DBAL\Exception If the database query fails
     */
    private function _buildMapRemProductIdToOemProductIds(array $remProductIds): array
    {
        if (empty($remProductIds)) {
            return [];
        }

        // ---- find matching OEM product ids
        $SQL = "    
                    SELECT 
                        LOWER(HEX(oem_product_id)) AS oemProductId, 
                        LOWER(HEX(rem_product_id)) AS remProductId 
                    FROM topdata_lor_oem_to_rem 
                    WHERE rem_product_id IN (:remProductIds)";
        $rows = $this->connection->fetchAllAssociative($SQL, ['remProductIds' => array_map('hex2bin', $remProductIds)], ['remProductIds' => Connection::PARAM_STR_ARRAY]);
        if (empty($rows)) {
            return [];
        }

        // ---- build Lookup map
        $mapRemProductIdToOemProductIds = [];
        foreach ($rows as $row) {
            if (!isset($mapRemProductIdToOemProductIds[$row['remProductId']])) {
                $mapRemProductIdToOemProductIds[$row['remProductId']] = [];
            }
            $mapRemProductIdToOemProductIds[$row['remProductId']][] = $row['oemProductId'];
        }

        return $mapRemProductIdToOemProductIds;
    }

    /**
     * Finds REM products that match given OEM products.
     * Used to display REM alternatives when OEM products are in the cart.
     *
     * DOES query the DB
     *
     * @param string[] $oemProductIds Array of OEM product IDs to find REM alternatives for
     * @param SalesChannelContext $salesChannelContext Current sales channel context
     * @return SalesChannelProductEntity[][] A map where keys are OEM product IDs and values are arrays of matching REM product entities
     * @throws Exception If the database query fails
     */
    public function findMatchingRemProductsMap(array $oemProductIds, SalesChannelContext $salesChannelContext): array
    {
        $mapOemProductIdToRemProductIds = $this->_buildMapOemProductIdToRemProductIds($oemProductIds);

        // ---- fetch the products
        $remProductIds = [];
        foreach($mapOemProductIdToRemProductIds as $ids){
            $remProductIds = array_merge($remProductIds, $ids);
        }
        $remProductIds = array_unique($remProductIds);


        if (empty($remProductIds)) {
            return [];
        }
        $criteria = new Criteria($remProductIds);
        $associatedProducts = $this->productRepository->search($criteria, $salesChannelContext)->getEntities();


        // ---- build the map
        $mapFinal = [];
        foreach ($mapOemProductIdToRemProductIds as $oemProductId => $singleOemRemProductIds) {
            $mapFinal[$oemProductId] = [];
            foreach($singleOemRemProductIds as $remProductId){
                $product = $associatedProducts->get($remProductId);
                if($product){
                    $mapFinal[$oemProductId][] = $product;
                }
            }
        }

        return $mapFinal;
    }

    /**
     * Finds all matching REM products for given OEM products and returns them as a flat array.
     * This is a convenience method that flattens the map returned by findMatchingRemProductsMap.
     *
     * DOES query the DB
     *
     * @param string[] $oemProductIds Array of OEM product IDs to find REM alternatives for
     * @param SalesChannelContext $salesChannelContext Current sales channel context
     * @return SalesChannelProductEntity[] Flat array of all matching REM product entities
     */
    public function findMatchingRemProductsFlat(array $oemProductIds, SalesChannelContext $salesChannelContext): array
    {
        $map = $this->findMatchingRemProductsMap($oemProductIds, $salesChannelContext);
        $flatArray = [];
        foreach ($map as $products) {
            $flatArray = array_merge($flatArray, $products);
        }
        // Remove duplicates if a REM product is linked to multiple OEM products in the input
        $uniqueProducts = [];
        foreach ($flatArray as $product) {
            $uniqueProducts[$product->getId()] = $product;
        }
        return array_values($uniqueProducts);
    }

    /**
     * Finds OEM products that match given REM products.
     * Used to display OEM alternatives when REM products are in the cart.
     *
     * DOES query the DB
     *
     * @param string[] $remProductIds Array of REM product IDs to find OEM alternatives for
     * @param SalesChannelContext $salesChannelContext Current sales channel context
     * @return array A map where keys are REM product IDs and values are arrays of matching OEM product entities
     * @throws Exception If the database query fails
     */
    public function findMatchingOemProductsMap(array $remProductIds, SalesChannelContext $salesChannelContext): array
    {
        $mapRemProductIdToOemProductIds = $this->_buildMapRemProductIdToOemProductIds($remProductIds);

        $oemProductIds = [];
        foreach($mapRemProductIdToOemProductIds as $ids){
            $oemProductIds = array_merge($oemProductIds, $ids);
        }
        $oemProductIds = array_unique($oemProductIds);

        if (empty($oemProductIds)) {
            return [];
        }
        $criteria = new Criteria($oemProductIds);
        // *** THIS IS THE FIX ***
        // Ensure necessary associations are loaded so the DTO can read the group name.
        $criteria->addAssociation('properties.group');
        $associatedProducts = $this->productRepository->search($criteria, $salesChannelContext)->getEntities();


        $mapFinal = [];
        foreach ($mapRemProductIdToOemProductIds as $remProductId => $singleRemOemProductIds) {
            $mapFinal[$remProductId] = [];
            foreach($singleRemOemProductIds as $oemProductId){
                $product = $associatedProducts->get($oemProductId);
                if($product){
                    $mapFinal[$remProductId][] = $product;
                }
            }
        }

        return $mapFinal;
    }

    /**
     * Finds all matching OEM products for given REM products and returns them as a flat array.
     * This is a convenience method that flattens the map returned by findMatchingOemProductsMap.
     *
     * @param string[] $remProductIds Array of REM product IDs to find OEM alternatives for
     * @param SalesChannelContext $salesChannelContext Current sales channel context
     * @return array Flat array of all matching OEM product entities
     */
    public function findMatchingOemProductsFlat(array $remProductIds, SalesChannelContext $salesChannelContext): array
    {
        $map = $this->findMatchingOemProductsMap($remProductIds, $salesChannelContext);
        $flatArray = [];
        foreach ($map as $products) {
            $flatArray = array_merge($flatArray, $products);
        }
        $uniqueProducts = [];
        foreach ($flatArray as $product) {
            $uniqueProducts[$product->getId()] = $product;
        }
        return array_values($uniqueProducts);
    }
}