<?php

namespace Topdata\TopdataLinkedOemRemSW6\DTO;

use Shopware\Core\Checkout\Cart\Price\Struct\CalculatedPrice;
use Shopware\Core\Content\Product\SalesChannel\SalesChannelProductEntity;
use Shopware\Core\Content\Property\Aggregate\PropertyGroupOption\PropertyGroupOptionEntity;
use Topdata\TopdataLinkedOemRemSW6\Foundation\Util\CliLogger;
use Topdata\TopdataLinkedOemRemSW6\Traits\CreateFromDictTrait;

/**
 * @method static SavingsOverOemDTO createFromDict(array $dict)
 */
class SavingsOverOemDTO
{
    use CreateFromDictTrait;

    private SalesChannelProductEntity $oemProduct;
    private SalesChannelProductEntity $remProduct;
    private float $priceOem;
    private float $priceRem;
    private ?int $capacityOem = null;
    private ?int $capacityRem = null;
    private ?string $capacityUnit = null;
    private bool $haveComparableCapacity;

    public ?string $capacitySourceGroupIdOem = null;
    public ?string $capacitySourceGroupIdRem = null;
    public ?string $capacitySourceGroupNameOem = null;
    public ?string $capacitySourceOptionNameOem = null;
    public ?string $capacitySourceOptionIdOem = null;
    public ?string $capacitySourceGroupNameRem = null;
    public ?string $capacitySourceOptionNameRem = null;
    public ?string $capacitySourceOptionIdRem = null;

    private float $priceSavingAbsolute;
    private float $priceSavingPercentAbsolute;
    private string $currencyId;
    private ?float $pricePerUnitOem = null;
    private ?float $pricePerUnitRem = null;
    private ?float $priceSavingPerUnitAbsolute = null;
    private ?float $priceSavingPerUnitRelative = null;
    public ?float $normalizedPriceRem = null;
    public ?float $normalizedPriceSavingAbsolute = null;
    public ?float $normalizedPriceSavingPercent = null;
    public bool $wasNormalized = false;

    public static function createFromOemProductAndRemProduct(
        SalesChannelProductEntity $oemProduct,
        SalesChannelProductEntity $remProduct,
        $sw6IdCapacity,
        string                    $currencyId,
        string                    $strategy = \Topdata\TopdataLinkedOemRemSW6\Constants\SavingsCalculationStrategyConstants::STRATEGY_SIMPLE
    ): ?self
    {
        $oemPrice = self::getProductPrice($oemProduct);
        $remPrice = self::getProductPrice($remProduct);

        if ($oemPrice === null || $remPrice === null) {
            CliLogger::warning("Could not determine price for OEM ({$oemProduct->getId()}) or REM ({$remProduct->getId()}). Cannot create SavingsOverOemDTO.");
            return null;
        }

        $capacityGroupIds = [];
        if ($sw6IdCapacity) {
            $capacityGroupIds = is_array($sw6IdCapacity) ? $sw6IdCapacity : [$sw6IdCapacity];
        }

        $capacityOemData = self::_findAndExtractCapacityData($oemProduct, $capacityGroupIds);
        $capacityRemData = self::_findAndExtractCapacityData($remProduct, $capacityGroupIds);

        $dict = [
            'oemProduct'                 => $oemProduct,
            'remProduct'                 => $remProduct,
            'priceOem'                   => $oemPrice,
            'priceRem'                   => $remPrice,
            'currencyId'                 => $currencyId,
            'priceSavingAbsolute'        => $oemPrice - $remPrice,
            'priceSavingPercentAbsolute' => ($oemPrice > 0) ? (($oemPrice - $remPrice) / $oemPrice) * 100 : 0,
            'capacityOem'                => $capacityOemData['value'],
            'capacityRem'                => $capacityRemData['value'],
            'capacityUnit'               => $capacityOemData['unit'] ?? $capacityRemData['unit'],
            'capacitySourceGroupIdOem'    => $capacityOemData['groupId'],
            'capacitySourceGroupNameOem'  => $capacityOemData['groupName'],
            'capacitySourceOptionNameOem' => $capacityOemData['optionName'],
            'capacitySourceOptionIdOem'   => $capacityOemData['optionId'],
            'capacitySourceGroupIdRem'    => $capacityRemData['groupId'],
            'capacitySourceGroupNameRem'  => $capacityRemData['groupName'],
            'capacitySourceOptionNameRem' => $capacityRemData['optionName'],
            'capacitySourceOptionIdRem'   => $capacityRemData['optionId'],
            'haveComparableCapacity'     => false,
            'pricePerUnitOem'            => null,
            'pricePerUnitRem'            => null,
            'priceSavingPerUnitAbsolute' => null,
            'priceSavingPerUnitRelative' => null,
            'normalizedPriceRem'         => null,
            'normalizedPriceSavingAbsolute' => null,
            'normalizedPriceSavingPercent' => null,
            'wasNormalized'              => false,
        ];

        if ($dict['capacityOem'] !== null && $dict['capacityRem'] !== null) {
            $dict['haveComparableCapacity'] = !($capacityOemData['unit'] !== null && $capacityRemData['unit'] !== null && $capacityOemData['unit'] !== $capacityRemData['unit']);
        }

        if ($dict['haveComparableCapacity']) {
            if ($dict['capacityOem'] > 0) $dict['pricePerUnitOem'] = $dict['priceOem'] / $dict['capacityOem'];
            if ($dict['capacityRem'] > 0) $dict['pricePerUnitRem'] = $dict['priceRem'] / $dict['capacityRem'];

            if ($strategy === \Topdata\TopdataLinkedOemRemSW6\Constants\SavingsCalculationStrategyConstants::STRATEGY_NORMALIZED && $dict['pricePerUnitRem'] !== null && $dict['capacityOem'] > 0) {
                $dict['normalizedPriceRem'] = $dict['pricePerUnitRem'] * $dict['capacityOem'];
                $dict['normalizedPriceSavingAbsolute'] = $dict['priceOem'] - $dict['normalizedPriceRem'];
                if ($dict['priceOem'] > 0) {
                    $dict['normalizedPriceSavingPercent'] = ($dict['normalizedPriceSavingAbsolute'] / $dict['priceOem']) * 100;
                }
                $dict['wasNormalized'] = true;
            }
        }

        return self::createFromDict($dict);
    }

    private static function getProductPrice(SalesChannelProductEntity $product): ?float
    {
        $calculatedPrice = $product->getCalculatedPrice();
        if ($calculatedPrice instanceof CalculatedPrice) {
            return $calculatedPrice->getTotalPrice();
        }
        CliLogger::warning("Product {$product->getId()} does not have a CalculatedPrice.");
        return null;
    }

    /**
     * Finds and extracts capacity data from a product based on a list of property group IDs.
     *
     * @return array{value: ?int, unit: ?string, groupId: ?string, groupName: ?string, optionId: ?string, optionName: ?string}
     */
    private static function _findAndExtractCapacityData(SalesChannelProductEntity $product, array $capacityGroupIds): array
    {
        $defaultReturn = ['value' => null, 'unit' => null, 'groupId' => null, 'groupName' => null, 'optionId' => null, 'optionName' => null];

        if (empty($capacityGroupIds) || $product->getProperties() === null) {
            return $defaultReturn;
        }

        foreach ($capacityGroupIds as $groupId) {
            /** @var PropertyGroupOptionEntity|null $propertyOption */
            $propertyOption = $product->getProperties()->filterByGroupId($groupId)->first();

            if ($propertyOption) {
                $optionName = $propertyOption->getTranslated()['name'] ?? $propertyOption->getName();
                if ($optionName) {
                    $parsedData = self::_parseCapacityFromPropertyOption($optionName);
                    if ($parsedData['value'] !== null) {
                        $group = $propertyOption->getGroup();
                        $groupName = $group ? ($group->getTranslated()['name'] ?? $group->getName()) : null;

                        return [
                            'value' => $parsedData['value'],
                            'unit' => $parsedData['unit'],
                            'groupId' => $groupId,
                            'groupName' => $groupName,
                            'optionId' => $propertyOption->getId(),
                            'optionName' => $optionName,
                        ];
                    }
                }
            }
        }
        return $defaultReturn;
    }

    private static function _parseCapacityFromPropertyOption(string $name): array
    {
        $value = null;
        if (preg_match('/(\d+[\.,]?\d*)/', $name, $matches)) {
            $numericPart = str_replace(['.', ','], ['', '.'], $matches[1]);
            $value = (int)filter_var($numericPart, FILTER_SANITIZE_NUMBER_INT);
        }
        $unit = self::_normalizeUnit($name);
        return ['value' => $value > 0 ? $value : null, 'unit' => $unit];
    }

    private static function _normalizeUnit(string $name): ?string
    {
        $nameLower = strtolower($name);
        $units = [
            'pages' => ['blatt', 'seiten', 'pages', 'sheet'],
            'ml' => ['ml'],
            'g' => ['g'],
            'kg' => ['kg'],
            'l' => ['l'],
        ];
        foreach ($units as $canonical => $aliases) {
            foreach ($aliases as $alias) {
                if (str_contains($nameLower, $alias)) return $canonical;
            }
        }
        return null;
    }

    // Getters
    public function getOemProduct(): SalesChannelProductEntity { return $this->oemProduct; }
    public function getRemProduct(): SalesChannelProductEntity { return $this->remProduct; }
    public function getPriceOem(): float { return $this->priceOem; }
    public function getPriceRem(): float { return $this->priceRem; }
    public function getCapacityOem(): ?int { return $this->capacityOem; }
    public function getCapacityRem(): ?int { return $this->capacityRem; }
    public function getCapacityUnit(): ?string { return $this->capacityUnit; }
    public function haveComparableCapacity(): bool { return $this->haveComparableCapacity; }
    public function getPriceSavingAbsolute(): float { return $this->priceSavingAbsolute; }
    public function getPriceSavingPercentAbsolute(): float { return $this->priceSavingPercentAbsolute; }
    public function getPricePerUnitOem(): ?float { return $this->pricePerUnitOem; }
    public function getPricePerUnitRem(): ?float { return $this->pricePerUnitRem; }
    public function getPriceSavingPerUnitAbsolute(): ?float { return $this->priceSavingPerUnitAbsolute; }
    public function getPriceSavingPerUnitRelative(): ?float { return $this->priceSavingPerUnitRelative; }
    public function getcurrencyId(): string { return $this->currencyId; }
    public function isCheaperPerUnit(): bool { return $this->haveComparableCapacity && $this->pricePerUnitRem !== null && $this->pricePerUnitOem !== null && $this->pricePerUnitRem < $this->pricePerUnitOem; }
    public function getCapacitySourceGroupIdOem(): ?string { return $this->capacitySourceGroupIdOem; }
    public function getCapacitySourceGroupIdRem(): ?string { return $this->capacitySourceGroupIdRem; }
    public function getCapacitySourceGroupNameOem(): ?string { return $this->capacitySourceGroupNameOem; }
    public function getCapacitySourceOptionNameOem(): ?string { return $this->capacitySourceOptionNameOem; }
    public function getCapacitySourceOptionIdOem(): ?string { return $this->capacitySourceOptionIdOem; }
    public function getCapacitySourceGroupNameRem(): ?string { return $this->capacitySourceGroupNameRem; }
    public function getCapacitySourceOptionNameRem(): ?string { return $this->capacitySourceOptionNameRem; }
    public function getCapacitySourceOptionIdRem(): ?string { return $this->capacitySourceOptionIdRem; }
}