<?php declare(strict_types=1);

namespace Topdata\TopdataLinkedOemRemSW6\Command;

use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\System\SalesChannel\Entity\SalesChannelRepository;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Topdata\TopdataLinkedOemRemSW6\Foundation\Command\AbstractTopdataCommand;
use Topdata\TopdataLinkedOemRemSW6\Foundation\Util\CliLogger;
use Topdata\TopdataLinkedOemRemSW6\DTO\SavingsOverOemDTO;
use Topdata\TopdataLinkedOemRemSW6\Service\OemRemSavingsService;
use Topdata\TopdataLinkedOemRemSW6\Foundation\Service\SalesChannelSelectionService;

/**
 * Command to show detailed savings information for a specific REM product.
 */
#[AsCommand(
    name: 'topdata:linked-oem-rem:show-rem-details',
    description: 'Shows detailed savings information for a specific REM product.'
)]
class Command_ShowRemDetails extends AbstractTopdataCommand
{
    public function __construct(
        private readonly SalesChannelRepository       $productRepository,
        private readonly OemRemSavingsService         $savingsService,
        private readonly SalesChannelSelectionService $salesChannelSelectionService
    )
    {
        parent::__construct();
    }

    public function configure(): void
    {
        $this->addArgument('remProductNumber', InputArgument::REQUIRED, 'The product number of the REM product.');
        $this->addOption('sales-channel-id', 's', InputOption::VALUE_REQUIRED, 'The ID of the sales channel to use. If not provided, you will be prompted to select one.');
    }

    /**
     * Executes the command.
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        CliLogger::setCliStyle($this->cliStyle);
        $remProductNumber = $input->getArgument('remProductNumber');
        CliLogger::title(sprintf('Details for REM Product: %s', $remProductNumber));

        $salesChannelContext = $this->salesChannelSelectionService->getSalesChannelContext($input);
        $currencySymbol = $salesChannelContext->getCurrency()->getSymbol();

        $criteria = new Criteria();
        $criteria->addFilter(new EqualsFilter('productNumber', $remProductNumber));
        $criteria->addAssociation('properties.group');
        $remProduct = $this->productRepository->search($criteria, $salesChannelContext)->first();

        if (!$remProduct) {
            CliLogger::error(sprintf('REM product with number "%s" not found in the selected sales channel.', $remProductNumber));
            return Command::FAILURE;
        }

        $savingsDto = $this->savingsService->getSavingsOverOem($remProduct, $salesChannelContext, $salesChannelContext->getCurrencyId());

        if (!$savingsDto) {
            CliLogger::warning(sprintf('No linked OEM product or savings information found for REM product %s.', $remProductNumber));
            CliLogger::section('REM Product Information');
            $this->_displayProductInfo($remProduct, $currencySymbol);
            $this->done();
            return Command::SUCCESS;
        }

        $oemProduct = $savingsDto->getOemProduct();

        CliLogger::section('REM Product Information');
        $this->_displayProductInfo(
            $remProduct,
            $currencySymbol,
            $savingsDto
        );

        CliLogger::section('Compared OEM Product Information');
        $this->_displayProductInfo(
            $oemProduct,
            $currencySymbol,
            $savingsDto,
            false // isRem = false
        );

        CliLogger::section('Savings Calculation');
        $this->_displaySavingsInfo($savingsDto, $currencySymbol);

        $this->done();
        return Command::SUCCESS;
    }

    private function _displayProductInfo(
        \Shopware\Core\Content\Product\SalesChannel\SalesChannelProductEntity $product,
        string                                                                $currencySymbol,
        ?SavingsOverOemDTO                                                    $dto = null,
        bool                                                                  $isRem = true
    ): void {
        $cliStyle = CliLogger::getCliStyle();
        
        $capacityString = 'N/A';
        if ($dto) {
            $capacity = $isRem ? $dto->getCapacityRem() : $dto->getCapacityOem();
            $unit = $dto->getCapacityUnit();
            $sourceGroupId = $isRem ? $dto->getCapacitySourceGroupIdRem() : $dto->getCapacitySourceGroupIdOem();
            $sourceGroupName = $isRem ? $dto->getCapacitySourceGroupNameRem() : $dto->getCapacitySourceGroupNameOem();
            $sourceOptionName = $isRem ? $dto->getCapacitySourceOptionNameRem() : $dto->getCapacitySourceOptionNameOem();
            $sourceOptionId = $isRem ? $dto->getCapacitySourceOptionIdRem() : $dto->getCapacitySourceOptionIdOem();

            if ($capacity) {
                $capacityString = sprintf('%d %s', $capacity, $unit);
                if ($sourceGroupName && $sourceOptionName) {
                    $groupInfo = sprintf('Group: "%s" [%s]', $sourceGroupName, $sourceGroupId);
                    $optionInfo = sprintf('Option: "%s" [%s]', $sourceOptionName, $sourceOptionId);
                    $capacityString .= sprintf(' (from %s, %s)', $groupInfo, $optionInfo);
                }
            }
        }
        
        $cliStyle->definitionList(
            ['ID' => $product->getId()],
            ['Name' => $product->getTranslated()['name'] ?? $product->getName()],
            ['Price' => $this->_formatPrice($product->getCalculatedPrice()->getUnitPrice(), $currencySymbol)],
            ['Capacity' => $capacityString]
        );
    }

    private function _displaySavingsInfo(SavingsOverOemDTO $dto, string $currencySymbol): void
    {
        $cliStyle = CliLogger::getCliStyle();
        $cliStyle->definitionList(
            ['Simple Price Saving (Absolute)' => $this->_formatPrice($dto->getPriceSavingAbsolute(), $currencySymbol)],
            ['Simple Price Saving (Percent)' => round($dto->getPriceSavingPercentAbsolute(), 2) . '%'],
            ['---' => '---'],
            ['Was Normalized?' => $dto->wasNormalized ? 'Yes' : 'No'],
            ['Normalized REM Price' => $this->_formatPrice($dto->normalizedPriceRem, $currencySymbol)],
            ['Normalized Saving (Absolute)' => $this->_formatPrice($dto->normalizedPriceSavingAbsolute, $currencySymbol)],
            ['Normalized Saving (Percent)' => $dto->normalizedPriceSavingPercent !== null ? round($dto->normalizedPriceSavingPercent, 2) . '%' : 'N/A'],
            ['---' => '---'],
            ['Price per Unit (OEM)' => $this->_formatPrice($dto->getPricePerUnitOem(), $currencySymbol)],
            ['Price per Unit (REM)' => $this->_formatPrice($dto->getPricePerUnitRem(), $currencySymbol)]
        );
    }

    private function _formatPrice(?float $price, string $currencySymbol): string
    {
        if ($price === null) {
            return 'N/A';
        }
        return number_format($price, 4) . ' ' . $currencySymbol;
    }
}