<?php declare(strict_types=1);

namespace Topdata\TopdataLinkedOemRemSW6\Command;

use Doctrine\DBAL\Connection;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\System\SalesChannel\Entity\SalesChannelRepository;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Helper\Table;
use Symfony\Component\Console\Helper\TableStyle;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Topdata\TopdataLinkedOemRemSW6\Foundation\Command\AbstractTopdataCommand;
use Topdata\TopdataLinkedOemRemSW6\Foundation\Util\CliLogger;
use Topdata\TopdataLinkedOemRemSW6\Service\OemRemSavingsService;
use Topdata\TopdataLinkedOemRemSW6\Foundation\Service\SalesChannelSelectionService;

/**
 * Command to list all REM products with their calculated savings.
 */
#[AsCommand(
    name: 'topdata:linked-oem-rem:list-rem-savings',
    description: 'Lists all REM products and their calculated savings compared to their OEM counterparts.'
)]
class Command_ListRemSavings extends AbstractTopdataCommand
{
    private const DEFAULT_LIMIT             = 20;
    const         PLACEHOLDER_NOT_AVAILABLE = '---';

    public function __construct(
        private readonly Connection                   $connection,
        private readonly SalesChannelRepository       $productRepository,
        private readonly OemRemSavingsService         $savingsService,
        private readonly SalesChannelSelectionService $salesChannelSelectionService
    )
    {
        parent::__construct();
    }

    public function configure(): void
    {
        $this->addOption('page', 'p', InputOption::VALUE_REQUIRED, 'The page number for table display. Ignored for CSV export.', '1');
        $this->addOption('limit', 'l', InputOption::VALUE_REQUIRED, 'Items per page for table display. Ignored for CSV export.', (string)self::DEFAULT_LIMIT);
        $this->addOption('sales-channel-id', 's', InputOption::VALUE_OPTIONAL, 'The ID of the sales channel to use. If not provided, you will be prompted to select one.');
        $this->addOption(
            'csv',
            null,
            InputOption::VALUE_OPTIONAL,
            'Export all results to a CSV file. Optionally provide a filename (e.g., --csv=export.csv). If no filename is provided, a default one is generated.',
            false // Default value is false, meaning the option is not active unless specified.
        );
    }

    /**
     * Executes the command.
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $csvOptionValue = $input->getOption('csv');
        $isCsvExport = ($csvOptionValue !== false);

        if (!$isCsvExport) {
            CliLogger::title('List REM Products with Savings');
        }

        // --- 1. Fetch and Process Data ---
        $processedData = $this->_fetchAndProcessData($input, $output, $isCsvExport);
        if ($processedData === null) {
            $this->done();
            return Command::SUCCESS; // Graceful exit if no data found
        }

        // --- 2. Render Output based on format ---
        if ($isCsvExport) {
            $this->_writeCsv($processedData['headers'], $processedData['rows'], $csvOptionValue);
        } else {
            $this->_renderTable($processedData['headers'], $processedData['rows'], $output, $processedData['paginationInfo']);
        }

        $this->done();
        return Command::SUCCESS;
    }

    /**
     * Gathers all data from the database and processes it into a structured array.
     * @return array|null An array containing headers, rows, and pagination info, or null if no data.
     */
    private function _fetchAndProcessData(InputInterface $input, OutputInterface $output, bool $isCsvExport): ?array
    {
        $isVerbose = $output->getVerbosity() > OutputInterface::VERBOSITY_NORMAL;
        $salesChannelContext = $this->salesChannelSelectionService->getSalesChannelContext($input);
        $currencySymbol = $salesChannelContext->getCurrency()->getSymbol();
        $paginationInfo = '';

        if ($isCsvExport) {
            // For CSV export, fetch all product IDs, ignoring pagination.
            $remProductIds = $this->_fetchUniqueRemProductIds();
            if (empty($remProductIds)) {
                CliLogger::info('No linked REM products found in the database for CSV export.');
                return null;
            }
            CliLogger::info(sprintf("Found %d products to export to CSV.", count($remProductIds)));

        } else {
            // For table view, use pagination.
            $page = (int)$input->getOption('page');
            $limit = (int)$input->getOption('limit');
            $offset = ($page - 1) * $limit;

            $totalRemLinks = $this->_fetchTotalUniqueRemCount();
            if ($totalRemLinks === 0) {
                CliLogger::info('No linked REM products found in the database.');
                return null;
            }

            $totalPages = (int)ceil($totalRemLinks / $limit);
            if ($page > $totalPages && $page > 1) {
                CliLogger::warning(sprintf('Page %d is out of range. Total pages: %d.', $page, $totalPages));
                return null;
            }

            $remProductIds = $this->_fetchUniqueRemProductIds($limit, $offset);
            if (empty($remProductIds)) {
                CliLogger::info(sprintf('No linked REM products found on page %d.', $page));
                return null;
            }
            $paginationInfo = sprintf('Displaying page %d of %d (Total REMs: %d, %d per page)', $page, $totalPages, $totalRemLinks, $limit);
        }


        $criteria = new Criteria($remProductIds);
        $criteria->addAssociation('properties.group');
        $remProducts = $this->productRepository->search($criteria, $salesChannelContext);

        $dataRows = [];
        $index = 1;
        foreach ($remProducts->getElements() as $remProduct) {
            $savingsDto = $this->savingsService->getSavingsOverOem($remProduct, $salesChannelContext, $salesChannelContext->getCurrencyId());

            $simpleSavingValue = null;
            $normalizedSavingValue = null;
            $shouldHighlight = false;

            if ($savingsDto) {
                $simpleSavingValue = $savingsDto->getPriceSavingAbsolute();
                if ($savingsDto->wasNormalized) {
                    $normalizedSavingValue = $savingsDto->normalizedPriceSavingAbsolute;
                    if (abs($normalizedSavingValue - $simpleSavingValue) > 0.01) {
                        $shouldHighlight = true;
                    }
                }
            }

            $dataRows[] = [
                'data'      => [
                    '#'               => $index++,
                    'REM Number'      => $remProduct->getProductNumber(),
                    'REM Price'       => $this->_formatPrice($remProduct->getCalculatedPrice()->getUnitPrice(), $currencySymbol),
                    'Norm. REM Price' => $savingsDto && $savingsDto->wasNormalized ? $this->_formatPrice($savingsDto->normalizedPriceRem, $currencySymbol) : self::PLACEHOLDER_NOT_AVAILABLE,
                    'OEM Number'      => $savingsDto ? $savingsDto->getOemProduct()->getProductNumber() : self::PLACEHOLDER_NOT_AVAILABLE,
                    'OEM Price'       => $savingsDto ? $this->_formatPrice($savingsDto->getPriceOem(), $currencySymbol) : self::PLACEHOLDER_NOT_AVAILABLE,
                    'Simple Saving'   => $this->_formatPrice($simpleSavingValue, $currencySymbol),
                    'Norm. Saving'    => $this->_formatPrice($normalizedSavingValue, $currencySymbol),
                    'Norm. Saving %'  => $savingsDto ? ($savingsDto->wasNormalized ? ($savingsDto->normalizedPriceSavingPercent !== null ? round($savingsDto->normalizedPriceSavingPercent, 1) . '%' : self::PLACEHOLDER_NOT_AVAILABLE) : round($savingsDto->getPriceSavingPercentAbsolute(), 1) . '%') : self::PLACEHOLDER_NOT_AVAILABLE,
                    'Normalized?'     => $savingsDto && $savingsDto->wasNormalized ? 'Yes' : 'No',
                    'Cap. Source REM' => $savingsDto ? $this->_formatCapacitySource($savingsDto->getCapacitySourceGroupNameRem(), $savingsDto->getCapacitySourceOptionNameRem(), $savingsDto->getCapacitySourceGroupIdRem(), $savingsDto->getCapacitySourceOptionIdRem(), $isVerbose) : self::PLACEHOLDER_NOT_AVAILABLE,
                    'Cap. Source OEM' => $savingsDto ? $this->_formatCapacitySource($savingsDto->getCapacitySourceGroupNameOem(), $savingsDto->getCapacitySourceOptionNameOem(), $savingsDto->getCapacitySourceGroupIdOem(), $savingsDto->getCapacitySourceOptionIdOem(), $isVerbose) : self::PLACEHOLDER_NOT_AVAILABLE,
                ],
                'highlight' => $shouldHighlight,
            ];
        }

        return [
            'headers'        => array_keys($dataRows[0]['data'] ?? []),
            'rows'           => $dataRows,
            'paginationInfo' => $paginationInfo,
        ];
    }

    /**
     * Renders the data as a styled table in the console.
     */
    private function _renderTable(array $headers, array $rows, OutputInterface $output, string $paginationInfo): void
    {
        CliLogger::info($paginationInfo);

        $table = new Table($output);
        $table->setHeaders($headers);

        $tableRows = [];
        foreach ($rows as $row) {
            $rowData = $row['data'];
            if ($row['highlight']) {
                $rowData = array_map(fn($cell) => "<fg=yellow>$cell</>", $rowData);
            }
            $tableRows[] = $rowData;
        }
        $table->setRows($tableRows);

        $rightAlignedStyle = new TableStyle();
        $rightAlignedStyle->setPadType(STR_PAD_LEFT);

        $table->setColumnStyle(2, $rightAlignedStyle);
        $table->setColumnStyle(3, $rightAlignedStyle);
        $table->setColumnStyle(5, $rightAlignedStyle);
        $table->setColumnStyle(6, $rightAlignedStyle);
        $table->setColumnStyle(7, $rightAlignedStyle);
        $table->setColumnStyle(8, $rightAlignedStyle);

        $table->render();
    }

    /**
     * Writes the data to a CSV file.
     */
    private function _writeCsv(array $headers, array $rows, ?string $outputFile): void
    {
        // If the value is NULL, it means --csv was used without a filename.
        if ($outputFile === null || $outputFile === '') {
            $outputFile = 'rem_savings_export_' . date('Y-m-d_H-i-s') . '.csv';
        }

        $handle = fopen($outputFile, 'w');
        if ($handle === false) {
            CliLogger::error("Could not open file for writing: $outputFile");
            return;
        }

        // Write headers
        fputcsv($handle, $headers);

        // Write data rows
        foreach ($rows as $row) {
            // We only need the raw data for the CSV
            fputcsv($handle, $row['data']);
        }

        fclose($handle);
        CliLogger::success("Successfully exported data to: $outputFile");
    }

    private function _fetchTotalUniqueRemCount(): int
    {
        $sql = "SELECT COUNT(DISTINCT rem_product_id) FROM topdata_lor_oem_to_rem";
        return (int)$this->connection->fetchOne($sql);
    }

    private function _fetchUniqueRemProductIds(?int $limit = null, ?int $offset = null): array
    {
        $sql = "SELECT DISTINCT LOWER(HEX(rem_product_id)) as id FROM topdata_lor_oem_to_rem ORDER BY rem_product_number ASC";

        $params = [];
        $types = [];

        if ($limit !== null) {
            $sql .= " LIMIT :limit";
            $params['limit'] = $limit;
            $types['limit'] = \PDO::PARAM_INT;
        }
        if ($offset !== null) {
            $sql .= " OFFSET :offset";
            $params['offset'] = $offset;
            $types['offset'] = \PDO::PARAM_INT;
        }

        return $this->connection->fetchFirstColumn($sql, $params, $types);
    }

    private function _formatPrice(?float $price, string $currencySymbol): string
    {
        if ($price === null) {
            return self::PLACEHOLDER_NOT_AVAILABLE;
        }
        return number_format($price, 2) . ' ' . $currencySymbol;
    }

    private function _formatCapacitySource(?string $groupName, ?string $optionName, ?string $groupId, ?string $optionId, bool $isVerbose): string
    {
        if (!$groupName) return self::PLACEHOLDER_NOT_AVAILABLE;
        if ($isVerbose) {
            $groupPart = $groupId ? sprintf('%s [%s]', $groupName, $groupId) : $groupName;
            $optionPart = ($optionName && $optionId) ? sprintf('%s [%s]', $optionName, $optionId) : $optionName;
        } else {
            $groupPart = $groupName;
            $optionPart = $optionName;
        }
        $source = $groupPart;
        if ($optionName) {
            $source .= ' > ' . $optionPart;
        }
        return $source;
    }
}