<?php declare(strict_types=1);

namespace Topdata\TopdataLinkedOemRemSW6\Command;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Exception;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Topdata\TopdataLinkedOemRemSW6\Foundation\Command\AbstractTopdataCommand;

/**
 * Command to list OEM products with multiple REM alternatives or REM products with multiple OEM sources.
 */
#[AsCommand(
    name: 'topdata:linked-oem-rem:list-duplicates',
    description: 'Lists products with multiple links (e.g., OEM with 2+ REMs, or REM with 2+ OEMs).'
)]
class Command_ListDuplicateLinks extends AbstractTopdataCommand
{
    private const MODE_OEM      = 'oem';
    private const MODE_REM      = 'rem';
    private const DEFAULT_MODE  = self::MODE_OEM;
    private const MINIMUM_LINKS = 2; // Show products with at least this many links

    public function __construct(
        private readonly Connection $connection
    )
    {
        parent::__construct();
    }


    public function configure(): void
    {
        $this->addOption('mode', 'm', InputOption::VALUE_REQUIRED, sprintf('Specify the mode: "%s" to list OEM products with multiple REM alternatives, or "%s" to list REM products with multiple OEM sources.', self::MODE_OEM, self::MODE_REM), self::DEFAULT_MODE);
        $this->addOption('min-links', null, InputOption::VALUE_REQUIRED, 'Minimum number of linked products to display (e.g., 2 for OEM with 2 or more REMs).', (string)self::MINIMUM_LINKS);
    }

    /**
     * Executes the command.
     *
     * @param InputInterface $input
     * @param OutputInterface $output
     * @return int
     * @throws Exception
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $mode = strtolower($input->getOption('mode'));
        $minLinks = (int)$input->getOption('min-links');

        if (!in_array($mode, [self::MODE_OEM, self::MODE_REM], true)) {
            $this->cliStyle->error(sprintf('Invalid mode "%s". Allowed modes are "%s" or "%s".', $mode, self::MODE_OEM, self::MODE_REM));
            return Command::FAILURE;
        }

        $sourceProductColumn = '';
        $linkedProductColumn = '';
        $groupByColumn = '';
        $selectLinkedColumn = '';
        $tableHeaderSource = '';
        $tableHeaderLinked = '';
        $commandTitle = '';

        if ($mode === self::MODE_OEM) {
            $commandTitle = sprintf('OEM Products with %d or More REM Alternatives', $minLinks);
            $sourceProductColumn = 'oem_product_number';
            $linkedProductColumn = 'rem_product_number';
            $groupByColumn = 'oem_product_number';
            $selectLinkedColumn = 'rem_product_number';
            $tableHeaderSource = 'OEM Product Number';
            $tableHeaderLinked = sprintf('REM Alternatives (Count)', $minLinks);
        } else { // MODE_REM
            $commandTitle = sprintf('REM Products with %d or More OEM Sources', $minLinks);
            $sourceProductColumn = 'rem_product_number';
            $linkedProductColumn = 'oem_product_number';
            $groupByColumn = 'rem_product_number';
            $selectLinkedColumn = 'oem_product_number';
            $tableHeaderSource = 'REM Product Number';
            $tableHeaderLinked = sprintf('OEM Sources (Count)', $minLinks);
        }

        $this->cliStyle->title($commandTitle);

        $sql = <<<SQL
            SELECT
                {$groupByColumn} AS source_product,
                COUNT({$selectLinkedColumn}) AS link_count,
                GROUP_CONCAT({$selectLinkedColumn} SEPARATOR '\n') AS linked_products
            FROM
                topdata_lor_oem_to_rem
            GROUP BY
                {$groupByColumn}
            HAVING
                COUNT({$selectLinkedColumn}) >= :minLinks
            ORDER BY
                source_product ASC;
        SQL;

        $results = $this->connection->fetchAllAssociative($sql, ['minLinks' => $minLinks], ['minLinks' => \PDO::PARAM_INT]);

        if (empty($results)) {
            $this->cliStyle->info(sprintf('No products found with %d or more links in "%s" mode.', $minLinks, $mode));
            $this->done();
            return Command::SUCCESS;
        }

        $tableData = [];
        $index = 1;
        foreach ($results as $row) {
            $tableData[] = [
                $index++,
                $row['source_product'],
                sprintf("%s (%d)", $row['linked_products'], $row['link_count']),
            ];
        }

        $this->cliStyle->table(
            ['#', $tableHeaderSource, $tableHeaderLinked],
            $tableData
        );

        $this->cliStyle->success(sprintf('Found %d products matching the criteria.', count($results)));
        $this->done();
        return Command::SUCCESS;
    }
}