<?php

namespace Topdata\TopdataLinkedOemRemSW6\Service;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Exception;
use Shopware\Core\Content\Product\SalesChannel\SalesChannelProductEntity;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\EntitySearchResult;
use Shopware\Core\System\SalesChannel\Entity\SalesChannelRepository;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Core\System\SystemConfig\SystemConfigService;
use Topdata\TopdataLinkedOemRemSW6\Foundation\Util\CliLogger;
use Topdata\TopdataLinkedOemRemSW6\Constants\GlobalPluginConstants;
use Topdata\TopdataLinkedOemRemSW6\DTO\SavingsOverOemDTO;

/**
 * Service for managing relationships between OEM (Original Equipment Manufacturer) and REM (Remanufactured) products.
 * This service handles finding, linking, and comparing OEM and REM products based on their properties.
 * 
 * Naming convention:
 * - methods starting with find* DO query the DB
 * - methods starting with get* do NOT query the DB
 *
 * 11/2023 created
 */
class OemRemService
{

    public function __construct(
        private readonly SalesChannelRepository $productRepository,
        private readonly Connection             $connection,
        private readonly SystemConfigService    $systemConfigService
    )
    {
    }

    /**
     * Finds products that are linked between OEM and REM based on property values.
     * This method performs a complex database query to identify REM products that reference OEM products
     * and vice versa, using property groups for MPN and ReferenceOEM.
     * 
     * @return array Array of linked product pairs with their details
     */
    private function _findLinkedProducts(): array
    {
        CliLogger::info("Starting _findLinkedProducts...");

        // ---- Retrieve configured and constant IDs ----
        $propertyGroupIdMpn = $this->systemConfigService->get('TopdataLinkedOemRemSW6.config.propertyGroupIdMpn');
        $propertyGroupIdReferenceOem = $this->systemConfigService->get('TopdataLinkedOemRemSW6.config.propertyGroupIdReferenceOem');
        $languageIdGerman = GlobalPluginConstants::LANGUAGE_ID_GERMAN;

        // ---- Pre-flight checks for critical IDs ----
        $allCriticalIdsValid = true;

        // Check Property Group ID for MPN
        if (empty($propertyGroupIdMpn)) {
            CliLogger::error("SystemConfig 'TopdataLinkedOemRemSW6.config.propertyGroupIdMpn' is not set or is empty.");
            $allCriticalIdsValid = false;
        } else {
            $existsMpnGroup = $this->connection->fetchOne(
                "SELECT 1 FROM property_group WHERE id = UNHEX(:id)",
                ['id' => $propertyGroupIdMpn]
            );
            if (!$existsMpnGroup) {
                CliLogger::warning("Configured PropertyGroupID for MPN ('{$propertyGroupIdMpn}') not found in 'property_group' table.");
                $allCriticalIdsValid = false;
            } else {
                CliLogger::info("Configured PropertyGroupID for MPN ('{$propertyGroupIdMpn}') confirmed in DB.");
            }
        }

        // Check Property Group ID for Reference OEM
        if (empty($propertyGroupIdReferenceOem)) {
            CliLogger::error("SystemConfig 'TopdataLinkedOemRemSW6.config.propertyGroupIdReferenceOem' is not set or is empty.");
            $allCriticalIdsValid = false;
        } else {
            $existsRefOemGroup = $this->connection->fetchOne(
                "SELECT 1 FROM property_group WHERE id = UNHEX(:id)",
                ['id' => $propertyGroupIdReferenceOem]
            );
            if (!$existsRefOemGroup) {
                CliLogger::warning("Configured PropertyGroupID for ReferenceOEM ('{$propertyGroupIdReferenceOem}') not found in 'property_group' table.");
                $allCriticalIdsValid = false;
            } else {
                CliLogger::info("Configured PropertyGroupID for ReferenceOEM ('{$propertyGroupIdReferenceOem}') confirmed in DB.");
            }
        }

        // Check Language ID for German
        if (empty($languageIdGerman)) {
            // This should ideally not happen if the constant is defined
            CliLogger::error("GlobalPluginConstants::LANGUAGE_ID_GERMAN is empty. This is unexpected.");
            $allCriticalIdsValid = false;
        } else {
            $existsLanguage = $this->connection->fetchOne(
                "SELECT 1 FROM language WHERE id = UNHEX(:id)",
                ['id' => $languageIdGerman]
            );
            if (!$existsLanguage) {
                CliLogger::warning("Defined LanguageID for German ('{$languageIdGerman}') not found in 'language' table.");
                $allCriticalIdsValid = false;
            } else {
                CliLogger::info("Defined LanguageID for German ('{$languageIdGerman}') confirmed in DB.");
            }
        }

        if (!$allCriticalIdsValid) {
            CliLogger::error("One or more critical IDs (PropertyGroup, Language) are missing from config/constants or not found in DB. Aborting link finding process. Please check plugin configuration and data integrity.");
            return [];
        }

        // ---- step 1 find rem products which have a reference oem
        $SQL = "
                SELECT 
                        LOWER(HEX(pp1.product_id)) AS remProductId, 
                        -- p1.product_number          AS remProductNumber, 
                        LOWER(HEX(pp2.product_id)) AS oemProductId, 
                        -- p2.product_number          AS oemProductNumber, 
                        pgot1.name                 AS referenceOem
                
                         FROM product_property                  pp1
                         JOIN property_group_option             pgo1  ON pgo1.id = pp1.property_group_option_id AND pgo1.property_group_id = UNHEX(:propertyGroupIdReferenceOem)
                         JOIN property_group_option_translation pgot1 ON pgot1.property_group_option_id=pgo1.id AND pgot1.language_id=UNHEX(:languageIdGerman)
                         -- too slow: JOIN product                           p1    ON p1.id = pp1.product_id AND p1.active = 1
                    
                         JOIN property_group_option             pgo2  ON pgo2.property_group_id = UNHEX(:propertyGroupIdMpn)  
                         JOIN property_group_option_translation pgot2 ON pgot2.property_group_option_id=pgo2.id AND pgot2.language_id=UNHEX(:languageIdGerman) AND pgot2.name = pgot1.name
                         JOIN product_property                  pp2   ON pp2.property_group_option_id = pgo2.id
                         -- too slow: JOIN product                           p2    ON p2.id = pp2.product_id AND p2.active = 1
                
                ORDER BY referenceOem ASC
        ";

        $bind = [
            'propertyGroupIdMpn'          => $propertyGroupIdMpn,
            'propertyGroupIdReferenceOem' => $propertyGroupIdReferenceOem,
            'languageIdGerman'            => $languageIdGerman,
        ];
        $rows = $this->connection->fetchAllAssociative($SQL, $bind);

        if(count($rows) === 0) {
            CliLogger::warning("Main SQL query returned 0 rows for linked REM/OEM products.");
            CliLogger::info("This means no products could be linked based on the current property configuration and data.");
            CliLogger::info("Further debugging information for property options will follow.");
            CliLogger::info("Main SQL for linking: " . $SQL);
            CliLogger::dump("Bind parameters for main SQL: ", $bind);
            $this->_debugPropertyOptions($bind); // Call the deeper debug method
            return []; // Return early as no links were found
        }

        CliLogger::info("Found " . count($rows) . " potential REM/OEM product links (before checking product activity).");

        // ---- step 2 - fetch products from db and make loopup map
        $productIdsRem = array_column($rows, 'remProductId');
        $productIdsOem = array_column($rows, 'oemProductId');
        $productIds = array_unique(array_merge($productIdsRem, $productIdsOem));

        if (empty($productIds)) {
            CliLogger::info("No product IDs extracted from SQL results. This is unexpected if rows > 0.");
            return [];
        }

        $rows2 = $this->connection->fetchAllAssociative("SELECT LOWER(HEX(id)) AS id, product_number, active FROM product WHERE id IN (:ids)",
            ['ids' => array_map('hex2bin', $productIds)],
            ['ids' => Connection::PARAM_STR_ARRAY]
        );
        $mapProductDetails = array_column($rows2, null, 'id'); // Create a map with product ID as key
        CliLogger::info("Fetched details for " . count($mapProductDetails) . " products from DB for " . count($productIds) . " unique IDs.");


        // ---- step 3 - combine data from step1 and step2, filtering by active status
        $rows3 = [];
        foreach ($rows as $idx => $row) {
            $remProductDetail = $mapProductDetails[$row['remProductId']] ?? null;
            $oemProductDetail = $mapProductDetails[$row['oemProductId']] ?? null;

            if (!$remProductDetail) {
                CliLogger::warning("REM product details not found for ID {$row['remProductId']}. Skipping row: " . json_encode($row));
                continue;
            }
            if (!$oemProductDetail) {
                CliLogger::warning("OEM product details not found for ID {$row['oemProductId']}. Skipping row: " . json_encode($row));
                continue;
            }

            $remActive = (int)$remProductDetail['active'];
            $oemActive = (int)$oemProductDetail['active'];

            if (!$remActive || !$oemActive) {
                CliLogger::info("Skipping link: REM Product ({$row['remProductId']}) active: {$remActive}, OEM Product ({$row['oemProductId']}) active: {$oemActive}. Row: " . json_encode($row));
                continue;
            }

            $rows3[] = [
                'referenceOem'     => $row['referenceOem'],
                'remProductNumber' => $remProductDetail['product_number'] ?? '',
                'oemProductNumber' => $oemProductDetail['product_number'] ?? '',
                'remProductId'     => $row['remProductId'],
                'oemProductId'     => $row['oemProductId'],
            ];
        }
        CliLogger::info("Found " . count($rows3) . " active and validly linked REM/OEM product pairs.");
        return $rows3;
    }


    /**
     * Helper method to debug property options if the main query yields no results.
     * This is called if primary ID checks pass but the linking query still finds nothing.
     * Performs detailed diagnostics on property groups, options, and product assignments
     * to help identify why product linking might be failing.
     *
     * @param array $bindParams Parameters used in the main SQL (propertyGroupIds, languageId)
     */
    private function _debugPropertyOptions(array $bindParams): void
    {
        CliLogger::info("--- Deeper Debugging: Checking Property Options ---");

        // Check options for Reference OEM Property Group
        $sqlRefOemOptions = "
            SELECT LOWER(HEX(pgo.id)) as option_id, pgot.name 
            FROM property_group_option pgo
            JOIN property_group_option_translation pgot ON pgot.property_group_option_id = pgo.id
            WHERE pgo.property_group_id = UNHEX(:propertyGroupIdReferenceOem)
              AND pgot.language_id = UNHEX(:languageIdGerman)
            LIMIT 10
        ";
        $refOemOptions = $this->connection->fetchAllAssociative($sqlRefOemOptions, [
            'propertyGroupIdReferenceOem' => $bindParams['propertyGroupIdReferenceOem'],
            'languageIdGerman' => $bindParams['languageIdGerman'],
        ]);
        if (empty($refOemOptions)) {
            CliLogger::warning("No property options found for ReferenceOEM group (ID: {$bindParams['propertyGroupIdReferenceOem']}) with German language (ID: {$bindParams['languageIdGerman']}). This is a likely cause for no links.");
        } else {
            CliLogger::info(count($refOemOptions) . " property options found for ReferenceOEM group and German language. First few examples: " . json_encode($refOemOptions));
        }

        // Check options for MPN Property Group
        $sqlMpnOptions = "
            SELECT LOWER(HEX(pgo.id)) as option_id, pgot.name 
            FROM property_group_option pgo
            JOIN property_group_option_translation pgot ON pgot.property_group_option_id = pgo.id
            WHERE pgo.property_group_id = UNHEX(:propertyGroupIdMpn)
              AND pgot.language_id = UNHEX(:languageIdGerman)
            LIMIT 10
        ";
        $mpnOptions = $this->connection->fetchAllAssociative($sqlMpnOptions, [
            'propertyGroupIdMpn' => $bindParams['propertyGroupIdMpn'],
            'languageIdGerman' => $bindParams['languageIdGerman'],
        ]);
        if (empty($mpnOptions)) {
            CliLogger::warning("No property options found for MPN group (ID: {$bindParams['propertyGroupIdMpn']}) with German language (ID: {$bindParams['languageIdGerman']}). This is a likely cause for no links.");
        } else {
            CliLogger::info(count($mpnOptions) . " property options found for MPN group and German language. First few examples: " . json_encode($mpnOptions));
        }

        // Check for matching names if both have options (conceptual check based on a sample)
        if (!empty($refOemOptions) && !empty($mpnOptions)) {
            // This is a more complex check, let's just see if there *could* be matches.
            // For a full check, we'd need all options, not just a sample.
            $refOemSampleNames = array_column($refOemOptions, 'name');
            $mpnSampleNames = array_column($mpnOptions, 'name');
            $intersectingSampleNames = array_intersect($refOemSampleNames, $mpnSampleNames);
            if (empty($intersectingSampleNames)) {
                CliLogger::info("Based on a sample of 10 options from each group, no common 'name' values were found. The main query relies on 'pgot2.name = pgot1.name'. If no names match across all options, no links will be found.");
            } else {
                CliLogger::info("Common 'name' values found in samples between ReferenceOEM and MPN options. Examples: " . json_encode(array_values($intersectingSampleNames)));
            }
        }

        // Check if any products are actually assigned these properties
        $sqlProductPropertyCheckRefOem = "
            SELECT COUNT(DISTINCT pp.product_id)
            FROM product_property pp
            JOIN property_group_option pgo ON pgo.id = pp.property_group_option_id
            WHERE pgo.property_group_id = UNHEX(:propertyGroupIdReferenceOem)
        ";
        $countProductsWithRefOemProp = $this->connection->fetchOne($sqlProductPropertyCheckRefOem, ['propertyGroupIdReferenceOem' => $bindParams['propertyGroupIdReferenceOem']]);
        CliLogger::info("Number of distinct products associated with ANY option in the ReferenceOEM group: " . $countProductsWithRefOemProp);
        if ($countProductsWithRefOemProp == 0) {
            CliLogger::warning("No products are assigned properties from the ReferenceOEM group. This will result in no links.");
        }


        $sqlProductPropertyCheckMpn = "
            SELECT COUNT(DISTINCT pp.product_id)
            FROM product_property pp
            JOIN property_group_option pgo ON pgo.id = pp.property_group_option_id
            WHERE pgo.property_group_id = UNHEX(:propertyGroupIdMpn)
        ";
        $countProductsWithMpnProp = $this->connection->fetchOne($sqlProductPropertyCheckMpn, ['propertyGroupIdMpn' => $bindParams['propertyGroupIdMpn']]);
        CliLogger::info("Number of distinct products associated with ANY option in the MPN group: " . $countProductsWithMpnProp);
        if ($countProductsWithMpnProp == 0) {
            CliLogger::warning("No products are assigned properties from the MPN group. This will result in no links.");
        }

        CliLogger::info("--- End Deeper Debugging ---");
    }

    /**
     * Inserts or updates the OEM-to-REM product relationships in the database.
     * Handles the persistence of linked products in the topdata_lor_oem_to_rem table,
     * including cleanup of stale entries from previous runs.
     *
     * 11/2023 created
     *
     * @param array $rows Array of product relationship data to be persisted
     * @return void
     */
    private function _upsert(array $rows): void
    {
        if (empty($rows)) {
            CliLogger::info("_upsert called with empty rows. No data to insert/update.");
            // Clean up stale entries even if new data is empty
            $refreshRunId = date('YmdHis');
            $stmtDelete = $this->connection->prepare("DELETE FROM topdata_lor_oem_to_rem WHERE refresh_run_id != :refreshRunId OR refresh_run_id IS NULL");
            try {
                $numRowsDeleted = $stmtDelete->executeStatement(['refreshRunId' => $refreshRunId]);
                CliLogger::info("deleted $numRowsDeleted stale entries (called from empty upsert)");
            } catch (\Throwable $e) {
                CliLogger::error("Error deleting stale entries: " . $e->getMessage());
            }
            return;
        }

        $SQL = "INSERT INTO topdata_lor_oem_to_rem (refresh_run_id, reference_oem, oem_product_number, rem_product_number, oem_product_id, rem_product_id, created_at) VALUES
               (:refreshRunId, :referenceOem, :oemProductNumber, :remProductNumber, UNHEX(:oemProductId), UNHEX(:remProductId), NOW())
               ON DUPLICATE KEY UPDATE updated_at = NOW(), reference_oem = VALUES(reference_oem), oem_product_number = VALUES(oem_product_number), rem_product_number = VALUES(rem_product_number), refresh_run_id = VALUES(refresh_run_id)
               "; // Used VALUES() for ON DUPLICATE KEY UPDATE for robustness


        // Begin the transaction for bulk insertion
        $this->connection->beginTransaction();

        // Prepare the SQL statement
        $refreshRunId = date('YmdHis'); // we use for deleting stale entries

        $stmtUpsert = $this->connection->prepare($SQL);
        // Delete entries that are not part of the current refresh run.
        // It's safer to delete entries whose refresh_run_id IS NULL or not equal to the current one.
        $stmtDelete = $this->connection->prepare("DELETE FROM topdata_lor_oem_to_rem WHERE refresh_run_id != :refreshRunId OR refresh_run_id IS NULL");


        try {
            $insertedCount = 0;
            // Execute the prepared statement for each set of data
            foreach ($rows as $data) {
                $stmtUpsert->executeStatement(array_merge(
                    $data,
                    ['refreshRunId' => $refreshRunId],
                ));
                $insertedCount++;
            }
            CliLogger::info("Upserted $insertedCount rows into topdata_lor_oem_to_rem.");


            $numRowsDeleted = $stmtDelete->executeStatement(['refreshRunId' => $refreshRunId]);
            CliLogger::info("deleted $numRowsDeleted stale entries from topdata_lor_oem_to_rem.");

            // Commit the transaction
            $this->connection->commit();
        } catch (\Throwable $e) {
            // Rollback the transaction if an error occurs
            $this->connection->rollBack();
            CliLogger::error("Error during _upsert: " . $e->getMessage());
            CliLogger::error("SQL: " . $SQL);
            CliLogger::dump("Sample Data (first row if available):", $rows[0] ?? null);
            throw $e; // Re-throw the exception after logging
        }
    }


    /**
     * Refreshes all links between OEM and REM products in the database.
     * This is the main entry point for updating the product relationship table.
     * The process involves finding all linked products and then upserting them
     * into the topdata_lor_oem_to_rem table.
     *
     * 11/2023 created
     *
     * @return void
     * @throws \Throwable If any database operation fails
     */
    public function refreshLinks()
    {
        // ---- step 1 - find rem products which have a reference oem
        $rows = $this->_findLinkedProducts();
        CliLogger::info("refreshLinks: _findLinkedProducts returned " . count($rows) . " linked products.");

        // ---- step 2 - populate tbl topdata_lor_oem_to_rem
        // _upsert handles empty $rows internally now
        $this->_upsert($rows);
        CliLogger::info("refreshLinks completed.");
    }


    /**
     * Builds a mapping from OEM product IDs to their corresponding REM product IDs.
     * 
     * Example output:
     * [ '0012345678' => [ '12345678', '12345679' ] ]
     * Where '0012345678' is an OEM product ID and the array contains all matching REM product IDs.
     *
     * 11/2023 created
     *
     * @param array $oemProductIds Array of OEM product IDs to find matches for
     * @return array A map of OEM product IDs to arrays of REM product IDs
     * @throws \Doctrine\DBAL\Exception If the database query fails
     */
    private function _buildMapOemProductIdToRemProductIds(array $oemProductIds): array
    {
        if (empty($oemProductIds)) {
            return [];
        }

        // ---- find matching REM product ids
        $SQL = "    
                    SELECT 
                        LOWER(HEX(rem_product_id)) AS remProductId, 
                        LOWER(HEX(oem_product_id)) AS oemProductId 
                    FROM topdata_lor_oem_to_rem 
                    WHERE oem_product_id IN (:oemProductIds)";
        $rows = $this->connection->fetchAllAssociative($SQL, ['oemProductIds' => array_map('hex2bin', $oemProductIds)], ['oemProductIds' => Connection::PARAM_STR_ARRAY]);

        if (empty($rows)) {
            return [];
        }

        // ---- build Lookup map
        $mapOemProductIdToRemProductIds = [];
        foreach ($rows as $row) {
            if (!isset($mapOemProductIdToRemProductIds[$row['oemProductId']])) {
                $mapOemProductIdToRemProductIds[$row['oemProductId']] = [];
            }
            $mapOemProductIdToRemProductIds[$row['oemProductId']][] = $row['remProductId'];
        }

        return $mapOemProductIdToRemProductIds;
    }


    /**
     * Builds a mapping from REM product IDs to their corresponding OEM product IDs.
     * 
     * Example output:
     * [ '0012345678' => [ '12345678', '12345679' ] ]
     * Where '0012345678' is a REM product ID and the array contains all matching OEM product IDs.
     *
     * 11/2023 created
     *
     * @param array $remProductIds Array of REM product IDs to find matches for
     * @return array A map of REM product IDs to arrays of OEM product IDs
     * @throws \Doctrine\DBAL\Exception If the database query fails
     */
    private function _buildMapRemProductIdToOemProductIds(array $remProductIds): array
    {
        if (empty($remProductIds)) {
            return [];
        }

        // ---- find matching OEM product ids
        $SQL = "    
                    SELECT 
                        LOWER(HEX(oem_product_id)) AS oemProductId, 
                        LOWER(HEX(rem_product_id)) AS remProductId 
                    FROM topdata_lor_oem_to_rem 
                    WHERE rem_product_id IN (:remProductIds)";
        $rows = $this->connection->fetchAllAssociative($SQL, ['remProductIds' => array_map('hex2bin', $remProductIds)], ['remProductIds' => Connection::PARAM_STR_ARRAY]);
        if (empty($rows)) {
            return [];
        }

        // ---- build Lookup map
        $mapRemProductIdToOemProductIds = [];
        foreach ($rows as $row) {
            if (!isset($mapRemProductIdToOemProductIds[$row['remProductId']])) {
                $mapRemProductIdToOemProductIds[$row['remProductId']] = [];
            }
            $mapRemProductIdToOemProductIds[$row['remProductId']][] = $row['oemProductId'];
        }

        return $mapRemProductIdToOemProductIds;
    }


    /**
     * Finds REM products that match given OEM products.
     * Used to display REM alternatives when OEM products are in the cart.
     * 
     * DOES query the DB
     *
     * 11/2023 created
     *
     * @param string[] $oemProductIds Array of OEM product IDs to find REM alternatives for
     * @param SalesChannelContext $salesChannelContext Current sales channel context
     * @return SalesChannelProductEntity[][] A map where keys are OEM product IDs and values are arrays of matching REM product entities
     * @throws Exception If the database query fails
     */
    public function findMatchingRemProductsMap(array $oemProductIds, SalesChannelContext $salesChannelContext): array
    {
        $mapOemProductIdToRemProductIds = $this->_buildMapOemProductIdToRemProductIds($oemProductIds);

        // ---- fetch the products
        $remProductIds = [];
        foreach($mapOemProductIdToRemProductIds as $ids){
            $remProductIds = array_merge($remProductIds, $ids);
        }
        $remProductIds = array_unique($remProductIds);


        if (empty($remProductIds)) {
            return [];
        }
        $criteria = new Criteria($remProductIds);
        // Ensure necessary associations are loaded if SavingsOverOemDTO needs them
        // e.g., $criteria->addAssociation('properties.group');
        $associatedProducts = $this->productRepository->search($criteria, $salesChannelContext)->getEntities();


        // ---- build the map
        $mapFinal = [];
        foreach ($mapOemProductIdToRemProductIds as $oemProductId => $singleOemRemProductIds) {
            $mapFinal[$oemProductId] = [];
            foreach($singleOemRemProductIds as $remProductId){
                $product = $associatedProducts->get($remProductId);
                if($product){
                    $mapFinal[$oemProductId][] = $product;
                }
            }
        }

        return $mapFinal;
    }

    /**
     * Finds all matching REM products for given OEM products and returns them as a flat array.
     * This is a convenience method that flattens the map returned by findMatchingRemProductsMap.
     * 
     * DOES query the DB
     *
     * 11/2023 created
     *
     * @param string[] $oemProductIds Array of OEM product IDs to find REM alternatives for
     * @param SalesChannelContext $ctx Current sales channel context
     * @return SalesChannelProductEntity[] Flat array of all matching REM product entities
     */
    public function findMatchingRemProductsFlat(array $oemProductIds, SalesChannelContext $salesChannelContext): array
    {
        $map = $this->findMatchingRemProductsMap($oemProductIds, $salesChannelContext);
        $flatArray = [];
        foreach ($map as $products) {
            $flatArray = array_merge($flatArray, $products);
        }
        // Remove duplicates if a REM product is linked to multiple OEM products in the input
        $uniqueProducts = [];
        foreach ($flatArray as $product) {
            $uniqueProducts[$product->getId()] = $product;
        }
        return array_values($uniqueProducts);
    }


    /**
     * Finds OEM products that match given REM products.
     * Used to display OEM alternatives when REM products are in the cart.
     * 
     * DOES query the DB
     *
     * 11/2023 created
     *
     * @param string[] $remProductIds Array of REM product IDs to find OEM alternatives for
     * @param SalesChannelContext $salesChannelContext Current sales channel context
     * @return array A map where keys are REM product IDs and values are arrays of matching OEM product entities
     * @throws Exception If the database query fails
     */
    public function findMatchingOemProductsMap(array $remProductIds, SalesChannelContext $salesChannelContext): array
    {
        $mapRemProductIdToOemProductIds = $this->_buildMapRemProductIdToOemProductIds($remProductIds);

        $oemProductIds = [];
        foreach($mapRemProductIdToOemProductIds as $ids){
            $oemProductIds = array_merge($oemProductIds, $ids);
        }
        $oemProductIds = array_unique($oemProductIds);

        if (empty($oemProductIds)) {
            return [];
        }
        $criteria = new Criteria($oemProductIds);
        // Ensure necessary associations are loaded if SavingsOverOemDTO needs them
        // e.g., $criteria->addAssociation('properties.group');
        $associatedProducts = $this->productRepository->search($criteria, $salesChannelContext)->getEntities();


        $mapFinal = [];
        foreach ($mapRemProductIdToOemProductIds as $remProductId => $singleRemOemProductIds) {
            $mapFinal[$remProductId] = [];
            foreach($singleRemOemProductIds as $oemProductId){
                $product = $associatedProducts->get($oemProductId);
                if($product){
                    $mapFinal[$remProductId][] = $product;
                }
            }
        }

        return $mapFinal;
    }


    /**
     * Finds all matching OEM products for given REM products and returns them as a flat array.
     * This is a convenience method that flattens the map returned by findMatchingOemProductsMap.
     *
     * 11/2023 created
     *
     * @param string[] $remProductIds Array of REM product IDs to find OEM alternatives for
     * @param SalesChannelContext $salesChannelContext Current sales channel context
     * @return array Flat array of all matching OEM product entities
     */
    public function findMatchingOemProductsFlat(array $remProductIds, SalesChannelContext $salesChannelContext): array
    {
        $map = $this->findMatchingOemProductsMap($remProductIds, $salesChannelContext);
        $flatArray = [];
        foreach ($map as $products) {
            $flatArray = array_merge($flatArray, $products);
        }
        $uniqueProducts = [];
        foreach ($flatArray as $product) {
            $uniqueProducts[$product->getId()] = $product;
        }
        return array_values($uniqueProducts);
    }

    /**
     * Calculates the savings a customer would get by purchasing the REM product instead of its OEM equivalent.
     * Returns the savings information as a DTO, or null if no OEM product is linked or no savings can be calculated.
     *
     * 11/2023 created
     *
     * @param SalesChannelProductEntity $remProduct The REM product to calculate savings for
     * @param SalesChannelContext $salesChannelContext Current sales channel context
     * @return SavingsOverOemDTO|null DTO containing savings information, or null if not applicable
     */
    public function getSavingsOverOem(SalesChannelProductEntity $remProduct, SalesChannelContext $salesChannelContext): ?SavingsOverOemDTO
    {
        $oemProducts = $this->findMatchingOemProductsFlat([$remProduct->getId()], $salesChannelContext);
        if (empty($oemProducts)) {
            return null;
        }

        $ret = [];
        $sw6IdCapacity = $this->systemConfigService->get('TopdataLinkedOemRemSW6.config.propertyGroupIdCapacity');
        if(empty($sw6IdCapacity)){
            CliLogger::warning("SystemConfig 'TopdataLinkedOemRemSW6.config.propertyGroupIdCapacity' is not set. Cannot calculate savings accurately if capacity property is needed by DTO.");
        }

        foreach ($oemProducts as $oemProduct) {
            $dto = SavingsOverOemDTO::createFromOemProductAndRemProduct($oemProduct, $remProduct, $sw6IdCapacity);
            if ($dto) { // createFromOemProductAndRemProduct might return null
                $ret[] = $dto;
            }
        }
        if(empty($ret)){
            return null;
        }

        usort($ret, fn(SavingsOverOemDTO $item1, SavingsOverOemDTO $item2) => $item2->getPriceSavingAbsolute() <=> $item1->getPriceSavingAbsolute());

        return $ret[0] ?? null;
    }

    /**
     * Creates a map of savings information for multiple REM products compared to their OEM equivalents.
     * This method is optimized for bulk processing as it only queries OEM products once for all REM products.
     * 
     * Does NOT query the DB for REM products (relies on pre-fetched searchResults),
     * but DOES query the DB for corresponding OEM products.
     *
     * 11/2023 created
     *
     * @param EntitySearchResult $searchResults The pre-fetched REM products for which to find OEM savings
     * @param SalesChannelContext $salesChannelContext Current sales channel context
     * @return array Map of REM product IDs to their respective SavingsOverOemDTO objects
     * @throws Exception If the database query for OEM products fails
     */
    public function getSavingsOverOemMap(EntitySearchResult $searchResults, SalesChannelContext $salesChannelContext): array
    {
        if ($searchResults->getTotal() === 0) {
            return [];
        }
        // This map will contain OEM products, so it DOES query the DB via findMatchingOemProductsMap
        $mapRemIdToOemProducts = $this->findMatchingOemProductsMap($searchResults->getIds(), $salesChannelContext);

        $savingsMap = [];
        $sw6IdCapacity = $this->systemConfigService->get('TopdataLinkedOemRemSW6.config.propertyGroupIdCapacity');
        if(empty($sw6IdCapacity)){
            CliLogger::warning("SystemConfig 'TopdataLinkedOemRemSW6.config.propertyGroupIdCapacity' is not set. Cannot calculate savings accurately if capacity property is needed by DTO.");
        }

        foreach ($mapRemIdToOemProducts as $remProductId => $oemProducts) {
            if (empty($oemProducts)) {
                continue;
            }
            $remProduct = $searchResults->get($remProductId);
            if (!$remProduct) {
                continue;
            }

            $savingsForThisRem = [];
            foreach ($oemProducts as $oemProduct) {
                $dto = SavingsOverOemDTO::createFromOemProductAndRemProduct($oemProduct, $remProduct, $sw6IdCapacity);
                if ($dto) {
                    $savingsForThisRem[] = $dto;
                }
            }

            if (!empty($savingsForThisRem)) {
                usort($savingsForThisRem, fn(SavingsOverOemDTO $item1, SavingsOverOemDTO $item2) => $item2->getPriceSavingAbsolute() <=> $item1->getPriceSavingAbsolute());
                $savingsMap[$remProductId] = $savingsForThisRem[0];
            }
        }

        return $savingsMap;
    }
}