<?php declare(strict_types=1);

namespace Topdata\TopdataLinkedOemRemSW6\Service;

use Shopware\Core\Content\Product\SalesChannel\SalesChannelProductEntity;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;

// For loading associations
use Shopware\Core\Framework\DataAbstractionLayer\Search\EntitySearchResult;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Core\System\SystemConfig\SystemConfigService;
use Topdata\TopdataFoundationSW6\Util\CliLogger;
use Topdata\TopdataLinkedOemRemSW6\DTO\SavingsOverOemDTO;

/**
 * Service for calculating savings when comparing OEM and REM products.
 * Uses the OemRemProductFinderService to find related products and then
 * calculates savings using the SavingsOverOemDTO.
 *
 * 05/2025 created as part of OemRemService refactoring
 */
class OemRemSavingsService
{
    public function __construct(
        private readonly OemRemProductFinderService $productFinderService,
        private readonly SystemConfigService        $systemConfigService
        // Potentially Shopware\Core\System\SalesChannel\Entity\SalesChannelRepository if you need to re-fetch products with associations here.
        // private readonly SalesChannelRepository $productRepository
    )
    {
        // Load currency configuration settings
//        self::DEFAULT_CURRENCY = $this->systemConfigService->get('TopdataLinkedOemRemSW6.config.defaultCurrency') ?? 'CZK';
//        $this->currencyDisplayFormat = $this->systemConfigService->get('TopdataLinkedOemRemSW6.config.currencyDisplayFormat') ?? 'SYMBOL';
//        $this->currencyDecimalPlaces = $this->systemConfigService->get('TopdataLinkedOemRemSW6.config.currencyDecimalPlaces') ?? 2;
    }

//    private string $defaultCurrency;
//    private string $currencyDisplayFormat;
//    private int $currencyDecimalPlaces;

    /**
     * Finds the "best" REM alternative products for a given OEM product,
     * sorted by savings (prioritizing per-unit savings if capacities are comparable).
     *
     * @param SalesChannelProductEntity $oemProduct The OEM product.
     *                                            IMPORTANT: Ensure this $oemProduct has 'properties' association loaded
     *                                            if capacity data is to be read from its properties.
     * @param SalesChannelContext $salesChannelContext
     * @param int $limit Maximum number of alternatives to return.
     * @return SavingsOverOemDTO[] Sorted array of DTOs representing the best REM alternatives.
     */
    public function findBestRemAlternatives(
        SalesChannelProductEntity $oemProduct,
        SalesChannelContext       $salesChannelContext,
        string                    $currencyId,
        int                       $limit = 3
    ): array
    {
        // Step 1: Find all REM products linked to the given OEM product.
        // The productFinderService should ideally load 'properties.group' and 'properties.options'
        // or just 'properties' for the REM products if not already done.
        $remProducts = $this->productFinderService->findMatchingRemProductsFlat(
            [$oemProduct->getId()],
            $salesChannelContext
        );

        if (empty($remProducts)) {
            CliLogger::info("No REM products found linked to OEM product {$oemProduct->getId()} ({$oemProduct->getProductNumber()}).");
            return [];
        }

        $sw6IdCapacity = $this->systemConfigService->get('TopdataLinkedOemRemSW6.config.propertyGroupIdCapacity');
        if (empty($sw6IdCapacity)) {
            CliLogger::warning("SystemConfig 'TopdataLinkedOemRemSW6.config.propertyGroupIdCapacity' is not set. Per-unit comparison will be disabled.");
        }

        // Step 2: For each REM product, create a SavingsOverOemDTO.
        $savingsDTOs = [];
        foreach ($remProducts as $remProduct) {
            // Ensure $oemProduct and $remProduct have properties loaded for DTO creation.
            // If $oemProduct passed in might not have properties, you'd fetch it here:
            // $criteriaOem = new Criteria([$oemProduct->getId()]);
            // $criteriaOem->addAssociation('properties');
            // $loadedOemProduct = $this->productRepository->search($criteriaOem, $salesChannelContext)->first() ?? $oemProduct;

            // Similarly for $remProduct if findMatchingRemProductsFlat doesn't guarantee properties:
            // $criteriaRem = new Criteria([$remProduct->getId()]);
            // $criteriaRem->addAssociation('properties');
            // $loadedRemProduct = $this->productRepository->search($criteriaRem, $salesChannelContext)->first() ?? $remProduct;
            // $dto = SavingsOverOemDTO::createFromOemProductAndRemProduct($loadedOemProduct, $loadedRemProduct, $sw6IdCapacity);

            // Use the configured default currency if available, otherwise fall back to sales channel currency
            $dto = SavingsOverOemDTO::createFromOemProductAndRemProduct($oemProduct, $remProduct, $sw6IdCapacity, $currencyId);
            if ($dto) {
                // Only consider alternatives that are actually cheaper or offer some value
                if ($dto->getPriceSavingAbsolute() > 0 || ($dto->getPriceSavingPerUnitAbsolute() !== null && $dto->getPriceSavingPerUnitAbsolute() > 0)) {
                    $savingsDTOs[] = $dto;
                } elseif ($dto->getPriceRem() < $dto->getPriceOem()) { // Fallback: if REM is cheaper but DTO didn't count as "saving"
                    $savingsDTOs[] = $dto;
                }
            }
        }

        if (empty($savingsDTOs)) {
            CliLogger::info("No REM alternatives with positive savings found for OEM product {$oemProduct->getId()}.");
            return [];
        }

        // Step 3: Sort these DTOs.
        usort($savingsDTOs, function (SavingsOverOemDTO $a, SavingsOverOemDTO $b) {
            // Priority 1: Per-unit comparison if both are comparable and REM is cheaper per unit
            $aHasPerUnitSaving = $a->haveComparableCapacity() && $a->getPricePerUnitRem() !== null && $a->getPricePerUnitOem() !== null && $a->getPriceSavingPerUnitAbsolute() > 0;
            $bHasPerUnitSaving = $b->haveComparableCapacity() && $b->getPricePerUnitRem() !== null && $b->getPricePerUnitOem() !== null && $b->getPriceSavingPerUnitAbsolute() > 0;

            if ($aHasPerUnitSaving && $bHasPerUnitSaving) {
                // Both offer per-unit savings. Higher per-unit absolute saving is better.
                if ($a->getPriceSavingPerUnitAbsolute() !== $b->getPriceSavingPerUnitAbsolute()) {
                    return ($a->getPriceSavingPerUnitAbsolute() > $b->getPriceSavingPerUnitAbsolute()) ? -1 : 1;
                }
                // Secondary: Higher per-unit relative saving is better
                if ($a->getPriceSavingPerUnitRelative() !== null && $b->getPriceSavingPerUnitRelative() !== null && $a->getPriceSavingPerUnitRelative() !== $b->getPriceSavingPerUnitRelative()) {
                    return ($a->getPriceSavingPerUnitRelative() > $b->getPriceSavingPerUnitRelative()) ? -1 : 1;
                }
            } elseif ($aHasPerUnitSaving) {
                return -1; // A has per-unit saving, B doesn't (or B's per-unit isn't a saving)
            } elseif ($bHasPerUnitSaving) {
                return 1;  // B has per-unit saving, A doesn't
            }

            // Priority 2: Absolute price saving (if per-unit wasn't decisive or applicable)
            // Higher absolute saving is better.
            if ($a->getPriceSavingAbsolute() !== $b->getPriceSavingAbsolute()) {
                return ($a->getPriceSavingAbsolute() > $b->getPriceSavingAbsolute()) ? -1 : 1;
            }

            // Priority 3: Percentage price saving
            // Higher percentage saving is better.
            if ($a->getPriceSavingPercentAbsolute() !== $b->getPriceSavingPercentAbsolute()) {
                return ($a->getPriceSavingPercentAbsolute() > $b->getPriceSavingPercentAbsolute()) ? -1 : 1;
            }

            // Fallback: REM product with lower price is better
            if ($a->getPriceRem() !== $b->getPriceRem()) {
                return ($a->getPriceRem() < $b->getPriceRem()) ? -1 : 1;
            }

            // Final fallback for stable sort if all else is equal
            return strcmp($a->getRemProduct()->getId(), $b->getRemProduct()->getId());
        });

        // Step 4: Return a limited list.
        return array_slice($savingsDTOs, 0, $limit);
    }


    /**
     * Calculates the savings a customer would get by purchasing the REM product instead of its OEM equivalent.
     * Returns the savings information as a DTO, or null if no OEM product is linked or no savings can be calculated.
     *
     * @param SalesChannelProductEntity $remProduct The REM product to calculate savings for.
     *                                            IMPORTANT: Ensure this $remProduct has 'properties' association loaded.
     * @param SalesChannelContext $salesChannelContext Current sales channel context
     * @return SavingsOverOemDTO|null DTO containing savings information, or null if not applicable
     */
    public function getSavingsOverOem(SalesChannelProductEntity $remProduct, SalesChannelContext $salesChannelContext, string $currencyId): ?SavingsOverOemDTO
    {
        // IMPORTANT: $remProduct needs properties loaded for DTO creation
        // If not guaranteed by caller, load them here:
        // $criteriaRem = new Criteria([$remProduct->getId()]);
        // $criteriaRem->addAssociation('properties');
        // $loadedRemProduct = $this->productRepository->search($criteriaRem, $salesChannelContext)->first() ?? $remProduct;

        $oemProducts = $this->productFinderService->findMatchingOemProductsFlat([$remProduct->getId()], $salesChannelContext);
        if (empty($oemProducts)) {
            return null;
        }

        $bestSavingDto = null;
        $sw6IdCapacity = $this->systemConfigService->get('TopdataLinkedOemRemSW6.config.propertyGroupIdCapacity');
        if (empty($sw6IdCapacity)) {
            CliLogger::warning("SystemConfig 'TopdataLinkedOemRemSW6.config.propertyGroupIdCapacity' is not set. Cannot calculate savings accurately if capacity property is needed by DTO.");
        }

        foreach ($oemProducts as $oemProduct) {
            // IMPORTANT: $oemProduct from productFinderService needs properties. Ensure they are loaded.
            // Use the configured default currency if available, otherwise fall back to sales channel currency

            $currentDto = SavingsOverOemDTO::createFromOemProductAndRemProduct($oemProduct, $remProduct, $sw6IdCapacity, $currencyId);
            if (!$currentDto) {
                continue;
            }

            if ($bestSavingDto === null) {
                $bestSavingDto = $currentDto;
            } else {
                // Determine "better" DTO based on sorting logic (simplified here for brevity,
                // ideally use a shared comparison logic with findBestRemAlternatives)
                // Priority: Per-unit saving
                $currentHasPerUnitSaving = $currentDto->haveComparableCapacity() && $currentDto->getPriceSavingPerUnitAbsolute() > 0;
                $bestHasPerUnitSaving = $bestSavingDto->haveComparableCapacity() && $bestSavingDto->getPriceSavingPerUnitAbsolute() > 0;

                if ($currentHasPerUnitSaving && $bestHasPerUnitSaving) {
                    if ($currentDto->getPriceSavingPerUnitAbsolute() > $bestSavingDto->getPriceSavingPerUnitAbsolute()) {
                        $bestSavingDto = $currentDto;
                    }
                } elseif ($currentHasPerUnitSaving) { // Current has per-unit saving, best doesn't
                    $bestSavingDto = $currentDto;
                } elseif ($bestHasPerUnitSaving) { // Best has per-unit saving, current doesn't - keep best
                    // no change
                } else { // Neither has per-unit saving, compare absolute
                    if ($currentDto->getPriceSavingAbsolute() > $bestSavingDto->getPriceSavingAbsolute()) {
                        $bestSavingDto = $currentDto;
                    }
                }
            }
        }
        return $bestSavingDto;
    }

    /**
     * Creates a map of savings information for multiple REM products compared to their OEM equivalents.
     *
     * @param EntitySearchResult $searchResults The pre-fetched REM products (SalesChannelProductEntity collection)
     *                                          IMPORTANT: Ensure entities in $searchResults have 'properties' loaded.
     * @param SalesChannelContext $salesChannelContext Current sales channel context
     * @return array Map of REM product IDs to their respective SavingsOverOemDTO objects
     */
    public function getSavingsOverOemMap(EntitySearchResult $searchResults, SalesChannelContext $salesChannelContext, string $currencyId): array
    {
        if ($searchResults->getTotal() === 0) {
            return [];
        }
        // OEM products fetched by productFinderService need 'properties' loaded.
        $mapRemIdToOemProducts = $this->productFinderService->findMatchingOemProductsMap($searchResults->getIds(), $salesChannelContext);

        $savingsMap = [];
        $sw6IdCapacity = $this->systemConfigService->get('TopdataLinkedOemRemSW6.config.propertyGroupIdCapacity');
        if (empty($sw6IdCapacity)) {
            CliLogger::warning("SystemConfig 'TopdataLinkedOemRemSW6.config.propertyGroupIdCapacity' is not set. Cannot calculate savings accurately if capacity property is needed by DTO.");
        }

        foreach ($searchResults->getElements() as $remProduct) { // Iterate over actual entities
            // $remProduct needs 'properties' loaded
            $remProductId = $remProduct->getId();
            if (!isset($mapRemIdToOemProducts[$remProductId]) || empty($mapRemIdToOemProducts[$remProductId])) {
                continue;
            }

            $oemProductsForThisRem = $mapRemIdToOemProducts[$remProductId];
            $bestDtoForThisRem = null;

            foreach ($oemProductsForThisRem as $oemProduct) {

                $currentDto = SavingsOverOemDTO::createFromOemProductAndRemProduct($oemProduct, $remProduct, $sw6IdCapacity, $currencyId);
                if (!$currentDto) {
                    continue;
                }

                if ($bestDtoForThisRem === null) {
                    $bestDtoForThisRem = $currentDto;
                } else {
                    // Simplified comparison logic for "best" OEM for this REM.
                    // Prioritize per-unit saving.
                    $currentHasPerUnitSaving = $currentDto->haveComparableCapacity() && $currentDto->getPriceSavingPerUnitAbsolute() > 0;
                    $bestHasPerUnitSaving = $bestDtoForThisRem->haveComparableCapacity() && $bestDtoForThisRem->getPriceSavingPerUnitAbsolute() > 0;

                    if ($currentHasPerUnitSaving && $bestHasPerUnitSaving) {
                        if ($currentDto->getPriceSavingPerUnitAbsolute() > $bestDtoForThisRem->getPriceSavingPerUnitAbsolute()) {
                            $bestDtoForThisRem = $currentDto;
                        }
                    } elseif ($currentHasPerUnitSaving) {
                        $bestDtoForThisRem = $currentDto;
                    } elseif (!$bestHasPerUnitSaving) { // Neither has per-unit, compare absolute
                        if ($currentDto->getPriceSavingAbsolute() > $bestDtoForThisRem->getPriceSavingAbsolute()) {
                            $bestDtoForThisRem = $currentDto;
                        }
                    }
                }
            }
            if ($bestDtoForThisRem && ($bestDtoForThisRem->getPriceSavingAbsolute() > 0 || ($bestDtoForThisRem->getPriceSavingPerUnitAbsolute() !== null && $bestDtoForThisRem->getPriceSavingPerUnitAbsolute() > 0))) {
                $savingsMap[$remProductId] = $bestDtoForThisRem;
            }
        }
        return $savingsMap;
    }
}