<?php declare(strict_types=1);

namespace Topdata\TopdataLinkedOemRemSW6\Storefront\Controller;

use Shopware\Core\Checkout\Cart\LineItem\LineItem;
use Shopware\Core\Checkout\Cart\SalesChannel\CartService;
use Shopware\Core\Content\Product\SalesChannel\SalesChannelProductEntity;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\Uuid\Uuid;
use Shopware\Core\System\Currency\CurrencyFormatter;
use Shopware\Core\System\SalesChannel\Entity\SalesChannelRepository;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Core\System\SystemConfig\SystemConfigService;
use Shopware\Storefront\Controller\StorefrontController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Topdata\TopdataLinkedOemRemSW6\Service\OemRemSavingsService;

#[Route(defaults: ['_routeScope' => ['storefront']])]
class AlternativeProductController extends StorefrontController
{
    public function __construct(
        private readonly CartService            $cartService,
        private readonly SystemConfigService    $systemConfigService,
        private readonly OemRemSavingsService   $oemRemSavingsService,
        private readonly SalesChannelRepository $productRepository,
        private readonly CurrencyFormatter      $currencyFormatter,
    )
    {
    }

    /**
     * 07/2025 created
     */
    #[Route(
        path: "/topdata-linked-oem-rem/rem-alternatives-to-oem-product/{oemProductId}",
        name: "topdata-linked-oem-rem.get-rem-alternatives-to-oem-product",
        defaults: ["_routeScope" => ["storefront"], 'XmlHttpRequest' => true],
        methods: ["GET"],
    )]
    public function getOemAlternativeDetails(string $oemProductId, SalesChannelContext $salesChannelContext): JsonResponse
    {
        $salesChannelId = $salesChannelContext->getSalesChannel()->getId();
        $currencyId = $salesChannelContext->getCurrencyId();


        $oemCriteria = (new Criteria([$oemProductId]))
            ->addAssociation('cover.media')
            ->addAssociation('product_price')
            ->addAssociation('properties.group'); // For capacity check in DTO

        /** @var SalesChannelProductEntity|null $oemProduct */
        $oemProduct = $this->productRepository->search($oemCriteria, $salesChannelContext)->get($oemProductId);

        if (!$oemProduct) {
            return new JsonResponse(['success' => false, 'showPopup' => false, 'message' => 'OEM Product not found.']);
        }

        // Read config values
        $maxCount = $this->systemConfigService->getInt('TopdataLinkedOemRemSW6.config.oemAlternativeMaxCount', $salesChannelId) ?: 3;
        $sortOrder = $this->systemConfigService->getString('TopdataLinkedOemRemSW6.config.oemAlternativeSortOrder', $salesChannelId) ?: 'savings_desc';

        // Get multiple alternatives
        $bestRemDTOs = $this->oemRemSavingsService->findBestRemAlternatives($oemProduct, $salesChannelContext, $currencyId, $maxCount);

        // Apply additional sorting if needed
        if ($sortOrder === 'price_asc') {
            usort($bestRemDTOs, fn($a, $b) => $a->getPriceRem() <=> $b->getPriceRem());
        }

        if (empty($bestRemDTOs)) {
            return new JsonResponse([
                'success'   => false,
                'showPopup' => false,
                'message'   => 'No suitable REM alternatives found.'
            ]);
        }

        // Prepare alternatives data
        $alternativesData = [];
        foreach ($bestRemDTOs as $remDto) {
            /** @var SalesChannelProductEntity $remProductFromDto */
            $remProductFromDto = $remDto->getRemProduct();

            // We need to ensure the REM product from DTO also has its cover media loaded for the popup
            $remCriteria = (new Criteria([$remProductFromDto->getId()]))->addAssociation('cover.media');
            /** @var SalesChannelProductEntity|null $remProduct */
            $remProduct = $this->productRepository->search($remCriteria, $salesChannelContext)->get($remProductFromDto->getId());

            if (!$remProduct) { // Should not happen if DTO is valid, but as a safeguard
                $remProduct = $remProductFromDto;
            }

            $remProductData = $this->_prepareProductDataForPopup($remProduct, $salesChannelContext);

            // Add savings info
            $savingsString = '';
            if ($remDto->getPriceSavingAbsolute() > 0) {
                $abs = $this->currencyFormatter->formatCurrencyByLanguage(
                    $remDto->getPriceSavingAbsolute(), $salesChannelContext->getCurrency()->getIsoCode(), $salesChannelContext->getContext()->getLanguageId(), $salesChannelContext->getContext()
                );
                $perc = round($remDto->getPriceSavingPercentAbsolute(), 1) . '%';
                $savingsString .= "Save {$abs} ({$perc})";
            }
            if ($remDto->isCheaperPerUnit() && $remDto->getCapacityUnit()) {
                if (!empty($savingsString)) $savingsString .= ". ";
                $savingsString .= "Cheaper per {$remDto->getCapacityUnit()}.";
            }
            if (empty($savingsString) && $remDto->getPriceRem() < $remDto->getPriceOem()) {
                $savingsString = "Alternative product is available."; // Generic message if no direct saving calculated but it's cheaper
            }

            $alternativesData[] = [
                'remProductData' => $remProductData,
                'savingsInfo'    => $savingsString,
            ];
        }

        $oemProductData = $this->_prepareProductDataForPopup($oemProduct, $salesChannelContext);

        return new JsonResponse([
            'success'        => true,
            'showPopup'      => true,
            'oemProductData' => $oemProductData,
            'alternatives'   => $alternativesData,
            'totalCount'     => count($bestRemDTOs),
            'currentIndex'   => 0
        ]);
    }


    // Helper function (can be private or in a trait/service)
    private function _prepareProductDataForPopup(SalesChannelProductEntity $productEntity, SalesChannelContext $context): array
    {
        $name = $productEntity->getTranslated()['name'] ?? $productEntity->getName() ?? 'Unknown Product';
        $imageUrl = $productEntity->getCover()?->getMedia()?->getUrl() ?? ''; // Provide a fallback image URL
        $price = $productEntity->getCalculatedPrice()->getUnitPrice();  // Or getTotalPrice()
        $priceFormatted = $this->currencyFormatter->formatCurrencyByLanguage(
            $price,
            $context->getCurrency()->getIsoCode(),
            $context->getContext()->getLanguageId(),
            $context->getContext()
        );

        return [
            'id'             => $productEntity->getId(),
            'name'           => $name,
            'imageUrl'       => $imageUrl,
            'price'          => $price, // not used, but anyway
            'currencyId'     => $context->getCurrency()->getIsoCode(), // not used, but anyway
            'currencySymbol' => $context->getCurrency()->getSymbol(), // not used, but anyway
            'priceFormatted' => $priceFormatted,
            'productNumber'  => $productEntity->getProductNumber(),
        ];
    }

    #[Route(
        path: "/topdata-linked-oem-rem/add-to-cart",
        name: "frontend.checkout.alternative.add-to-cart",
        options: ["seo" => false],
        defaults: ["XmlHttpRequest" => true],
        methods: ["POST"]
    )]
    public function addAlternativeToCart(Request $request, SalesChannelContext $context): Response
    {
        $productId = $request->request->get('productId');
        $quantity = $request->request->getInt('quantity');

        $validationMessages = [];
        if (!is_string($productId) || empty($productId)) {
            $validationMessages[] = 'productId must be a non-empty string.';
        } elseif (!Uuid::isValid($productId)) { // Only check Uuid if productId is a non-empty string
            $validationMessages[] = 'productId must be a valid UUID.';
        }

        if ($quantity <= 0) {
            $validationMessages[] = 'quantity must be an integer greater than 0.';
        }

        if (!empty($validationMessages)) {
            return new JsonResponse([
                'success' => false,
                'message' => 'Invalid parameters: ' . implode(' ', $validationMessages),
            ], Response::HTTP_BAD_REQUEST);
        }

        try {
            $lineItem = new LineItem($productId, LineItem::PRODUCT_LINE_ITEM_TYPE, $productId, $quantity);
            $lineItem->setRemovable(true);
            $lineItem->setStackable(true);

            $cart = $this->cartService->getCart($context->getToken(), $context);
            $this->cartService->add($cart, $lineItem, $context);

            // $request->getSession()->remove('oem_alternative_popup_data');

            $redirectAction = $this->systemConfigService->getString(
                'TopdataLinkedOemRemSW6.config.oemAlternativeRedirectAction',
                $context->getSalesChannel()->getId()
            );

            switch ($redirectAction) {
                case 'detail':
                    return new JsonResponse([
                        'success'     => true,
                        'refreshCart' => false,
                        'redirectTo'  => 'frontend.detail.page',
                        'parameters'  => ['productId' => $productId],
                        'message'     => 'Product added. Redirecting to product page.'
                    ]);
                case 'cart':
                    return new JsonResponse([
                        'success'     => true,
                        'refreshCart' => false,
                        'redirectTo'  => 'frontend.checkout.cart.page',
                        'message'     => 'Product added. Redirecting to cart.'
                    ]);
                case 'close':
                default:
                    return new JsonResponse([
                        'success'     => true,
                        'refreshCart' => true,
                        'redirectTo'  => null,
                        'message'     => 'Product added. Refreshing cart.'
                    ]);
            }
        } catch (\Exception $e) {
            // It's good practice to log the exception.
            // If you have a logger service injected, you would use it here.
            // Example: $this->logger->error($e->getMessage(), ['exception' => $e]);
            return new JsonResponse([
                'success' => false,
                'message' => 'Error adding product to cart: ' . $e->getMessage(),
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }
}