import Plugin from 'src/plugin-system/plugin.class';
import HttpClient from 'src/service/http-client.service';

export default class OemAlternativePopupPlugin extends Plugin {
    static options = {
        modalSelector: '#oemAlternativePopupModal',
        redirectAction: 'detail',
    };

    init() {
        console.log('[OEM Popup] Initializing Plugin...');

        this.modalElement = document.querySelector(this.options.modalSelector);
        if (!this.modalElement) {
            console.error('[OEM Popup] CRITICAL: Modal element not found with selector:', this.options.modalSelector);
            return;
        }
        console.log('[OEM Popup] Modal element found:', this.modalElement);

        const { Modal } = window.bootstrap;
        this.bsModal = new Modal(this.modalElement);
        this.productAddedToCart = false;

        this._bindModalEvents();
        this._bindPopupActionButtons();

        if (this.el.dataset.redirectAction) {
            this.options.redirectAction = this.el.dataset.redirectAction;
        }

        console.log('[OEM Popup] Plugin initialized and ready.');
    }

    _bindModalEvents() {
        this.modalElement.addEventListener('hide.bs.modal', this._handleModalCloseAttempt.bind(this));
        console.log('[OEM Popup] Bound modal close event.');
    }

    _bindPopupActionButtons() {
        console.log('[OEM Popup] Binding action buttons...');
        const prevButton = this.modalElement.querySelector('.carousel-prev');
        console.log('[OEM Popup] Found "Previous" button:', prevButton);
        if (prevButton) {
            prevButton.addEventListener('click', this._previousAlternative.bind(this));
        }

        const nextButton = this.modalElement.querySelector('.carousel-next');
        console.log('[OEM Popup] Found "Next" button:', nextButton);
        if (nextButton) {
            nextButton.addEventListener('click', this._nextAlternative.bind(this));
        }

        const continueWithOriginalButton = this.modalElement.querySelector('.continue-with-original');
        console.log('[OEM Popup] Found "Continue" button:', continueWithOriginalButton);
        if (continueWithOriginalButton) {
            continueWithOriginalButton.addEventListener('click', this._handleContinueWithOriginal.bind(this));
        }

        const addThisAlternativeButton = this.modalElement.querySelector('.add-current-alternative');
        console.log('[OEM Popup] Found "Add Alternative" button:', addThisAlternativeButton);
        if (addThisAlternativeButton) {
            addThisAlternativeButton.addEventListener('click', this._handleAddThisAlternative.bind(this));
        }
    }

    show(oemProductData, alternatives, oemId, quantity) {
        console.groupCollapsed(`[OEM Popup] show() called at ${new Date().toLocaleTimeString()}`);
        console.log('Received OEM Data:', oemProductData);
        console.log('Received Alternatives Array:', alternatives);
        console.log(`Total alternatives: ${alternatives.length}`);
        console.log(`Requested Quantity: ${quantity}`);
        console.groupEnd();

        this.currentOemProductData = oemProductData;
        this.alternatives = alternatives;
        this.totalAlternatives = alternatives.length;
        this.currentIndex = 0;
        this.currentOemId = oemId;
        this.currentQuantity = quantity || 1; // Store the quantity
        this.productAddedToCart = false;

        this._populateOemProduct();
        this._initializeCarousel();
        this._showAlternative(0);
        this.bsModal.show();
        console.log('[OEM Popup] Modal display triggered.');
    }

    _populateOemProduct() {
        console.groupCollapsed('[OEM Popup] Populating OEM Product Data');
        if (!this.currentOemProductData) {
            console.warn('[OEM Popup] No OEM product data available to populate.');
            console.groupEnd();
            return;
        }

        const oemImageContainerEl = this.modalElement.querySelector('.oem-product-image');
        console.log('[OEM Popup] OEM image container:', oemImageContainerEl);

        const oemNameEl = this.modalElement.querySelector('.oem-product-name');
        console.log('[OEM Popup] OEM name element:', oemNameEl);

        const oemPriceEl = this.modalElement.querySelector('.oem-product-price');
        console.log('[OEM Popup] OEM price element:', oemPriceEl);

        this._setImageInContainer(oemImageContainerEl, this.currentOemProductData.imageUrl, this.currentOemProductData.name, 'OEM Product Image');
        if (oemNameEl) oemNameEl.textContent = this.currentOemProductData.name;
        if (oemPriceEl) oemPriceEl.textContent = this.currentOemProductData.price;
        console.groupEnd();
    }

    _populateAlternativeProduct(alternative) {
        console.groupCollapsed('[OEM Popup] Populating Alternative Product Data');
        console.log('[OEM Popup] Data for current alternative:', alternative);

        // *** THE FIX IS HERE ***
        // Selectors must match the class names in your `_oem_alternative_popup.html.twig` file.
        const altImageContainerEl = this.modalElement.querySelector('.rem-product-image');
        const altNameEl = this.modalElement.querySelector('.rem-product-name');
        const altPriceEl = this.modalElement.querySelector('.rem-product-price');
        const altSavingsEl = this.modalElement.querySelector('.rem-product-savings');

        console.log('[OEM Popup] Looking for .rem-product-image, found:', altImageContainerEl);
        console.log('[OEM Popup] Looking for .rem-product-name, found:', altNameEl);
        console.log('[OEM Popup] Looking for .rem-product-price, found:', altPriceEl);
        console.log('[OEM Popup] Looking for .rem-product-savings, found:', altSavingsEl);


        if (alternative && alternative.remProductData) {
            const remData = alternative.remProductData;
            console.log('[OEM Popup] Applying data from remProductData:', remData);

            this._setImageInContainer(altImageContainerEl, remData.imageUrl, remData.name, 'Alternative Product Image');
            if (altNameEl) altNameEl.textContent = remData.name || 'N/A';
            if (altPriceEl) altPriceEl.textContent = remData.price || 'N/A';
            if (altSavingsEl) altSavingsEl.textContent = alternative.savingsInfo || '';
            console.log('[OEM Popup] Successfully applied data to elements.');
        } else {
            console.warn('[OEM Popup] No alternative data or nested remProductData found. Clearing fields.');
            this._setImageInContainer(altImageContainerEl, null, '', 'Alternative Product Image');
            if (altNameEl) altNameEl.textContent = 'N/A';
            if (altPriceEl) altPriceEl.textContent = 'N/A';
            if (altSavingsEl) altSavingsEl.textContent = '';
        }
        console.groupEnd();
    }

    _initializeCarousel() {
        console.log('[OEM Popup] Initializing carousel. Total alternatives:', this.totalAlternatives);
        const prevButton = this.modalElement.querySelector('.carousel-prev');
        const nextButton = this.modalElement.querySelector('.carousel-next');

        if (this.totalAlternatives > 1) {
            console.log('[OEM Popup] More than one alternative. Showing navigation buttons.');
            if (prevButton) prevButton.classList.remove('d-none');
            if (nextButton) nextButton.classList.remove('d-none');
        } else {
            console.log('[OEM Popup] One or zero alternatives. Hiding navigation buttons.');
            if (prevButton) prevButton.classList.add('d-none');
            if (nextButton) nextButton.classList.add('d-none');
        }
    }

    _showAlternative(index) {
        if (index < 0 || index >= this.totalAlternatives) {
            console.warn(`[OEM Popup] _showAlternative called with invalid index: ${index}`);
            return;
        }

        console.log(`[OEM Popup] Showing alternative at index: ${index}`);
        this.currentIndex = index;
        const alternative = this.alternatives[index];

        this._populateAlternativeProduct(alternative);
        this._updateNavigationState();
        this._updateCounter();
    }

    _nextAlternative() {
        console.log('[OEM Popup] Next button clicked.');
        if (this.currentIndex < this.totalAlternatives - 1) {
            this._showAlternative(this.currentIndex + 1);
        } else {
            console.log('[OEM Popup] Already at the last alternative.');
        }
    }

    _previousAlternative() {
        console.log('[OEM Popup] Previous button clicked.');
        if (this.currentIndex > 0) {
            this._showAlternative(this.currentIndex - 1);
        } else {
            console.log('[OEM Popup] Already at the first alternative.');
        }
    }

    _updateNavigationState() {
        const prevButton = this.modalElement.querySelector('.carousel-prev');
        const nextButton = this.modalElement.querySelector('.carousel-next');

        if (prevButton) {
            prevButton.disabled = this.currentIndex === 0;
            console.log(`[OEM Popup] Previous button disabled: ${prevButton.disabled}`);
        }

        if (nextButton) {
            nextButton.disabled = this.currentIndex === this.totalAlternatives - 1;
            console.log(`[OEM Popup] Next button disabled: ${nextButton.disabled}`);
        }
    }

    _updateCounter() {
        const currentSlideEl = this.modalElement.querySelector('.current-slide');
        const totalSlidesEl = this.modalElement.querySelector('.total-slides');

        if (currentSlideEl && totalSlidesEl) {
            currentSlideEl.textContent = this.currentIndex + 1;
            totalSlidesEl.textContent = this.totalAlternatives;
            console.log(`[OEM Popup] Counter updated to: ${this.currentIndex + 1}/${this.totalAlternatives}`);
        }
    }

    _handleAddThisAlternative() {
        if (!this.alternatives || !this.alternatives[this.currentIndex]) return;
        const alternativeData = this.alternatives[this.currentIndex].remProductData;
        if (!alternativeData || !alternativeData.id) {
            console.error('[OEM Popup] CRITICAL: Could not find ID for the current alternative to add to cart.', this.alternatives[this.currentIndex]);
            return;
        }
        console.log('[OEM Popup] User chose to add alternative product to cart. ID:', alternativeData.id);
        this._addProductToCart(alternativeData.id);
    }

    _handleContinueWithOriginal() {
        console.log('[OEM Popup] User chose to continue with original product. ID:', this.currentOemId);
        this._addProductToCart(this.currentOemId);
    }

    _handleModalCloseAttempt(event) {
        if (this.productAddedToCart) {
            console.log('[OEM Popup] Modal closing naturally because a product was added.');
            return;
        }
        console.log('[OEM Popup] Modal close was attempted by user (ESC/backdrop). Adding original product by default.');
        event.preventDefault();
        this._handleContinueWithOriginal();
    }

    _addProductToCart(productId) {
        if (this.productAddedToCart) {
            console.warn('[OEM Popup] Add to cart action already initiated. Ignoring subsequent clicks.');
            return;
        }
        this.productAddedToCart = true;
        console.log(`[OEM Popup] Sending POST request to add product ID ${productId} with quantity ${this.currentQuantity} to cart.`);

        const client = new HttpClient();
        const data = new FormData();
        data.append('productId', productId);
        data.append('quantity', this.currentQuantity); // Use the stored quantity

        const route = '/topdata-linked-oem-rem/add-to-cart';

        client.post(route, data, (responseString) => {
            console.log('[OEM Popup] Received "add to cart" response:', responseString);
            try {
                const response = JSON.parse(responseString);
                if (response.success) {
                    console.log('[OEM Popup] Product added successfully. Hiding modal and opening off-canvas cart.');
                    this.bsModal.hide();
                    this._openOffCanvasCart();
                } else {
                    this.productAddedToCart = false; // Allow user to try again if it failed
                    this._displayFlashMessage('danger', `[OEM Popup] Error: ${response.message || 'Could not add product.'}`);
                }
            } catch (e) {
                this.productAddedToCart = false;
                console.error('[OEM Popup] Failed to parse "add to cart" response:', e, responseString);
                this._displayFlashMessage('danger', 'An unexpected error occurred.');
            }
        }, (error) => {
            this.productAddedToCart = false;
            console.error('[OEM Popup] AJAX error adding product to cart:', error);
            this._displayFlashMessage('danger', 'A network error occurred.');
        });
    }

    // --- HELPER METHODS (UNCHANGED) ---

    _openOffCanvasCart() {
        console.log('[OEM Popup] Attempting to open off-canvas cart...');
        const offCanvasCartPluginInstances = window.PluginManager.getPluginInstances('OffCanvasCart');
        if (!offCanvasCartPluginInstances || offCanvasCartPluginInstances.length === 0) {
            console.error('[OEM Popup] OffCanvasCartPlugin instance not found.');
            return;
        }
        const offCanvasCartPlugin = offCanvasCartPluginInstances[0];
        if (offCanvasCartPlugin && typeof offCanvasCartPlugin.openOffCanvas === 'function') {
            const offcanvasCartUrl = window.router['frontend.cart.offcanvas'];
            if (!offcanvasCartUrl) {
                console.error('[OEM Popup] Off-canvas cart URL not found in window.router.');
                return;
            }
            offCanvasCartPlugin.openOffCanvas(offcanvasCartUrl, false, () => {
                console.log('[OEM Popup] Off-canvas cart opened.');
            });
        } else {
            console.error('[OEM Popup] Could not access OffCanvasCartPlugin or its openOffCanvas method.');
        }
    }

    _setImageInContainer(containerEl, imageUrl, altText, defaultAltText = 'Product image') {
        if (!containerEl) return;
        containerEl.innerHTML = '';
        if (imageUrl) {
            const imgElement = document.createElement('img');
            imgElement.src = imageUrl;
            imgElement.alt = altText || defaultAltText;
            imgElement.classList.add('img-fluid');
            containerEl.appendChild(imgElement);
        }
    }

    _displayFlashMessage(type, message) {
        const toastElement = document.createElement('div');
        toastElement.className = `toast align-items-center text-white bg-${type} border-0`;
        toastElement.setAttribute('role', 'alert');
        toastElement.innerHTML = `<div class="d-flex"><div class="toast-body">${message}</div><button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button></div>`;
        let toastContainer = document.querySelector('.toast-container');
        if (!toastContainer) {
            toastContainer = document.createElement('div');
            toastContainer.className = 'toast-container position-fixed top-0 end-0 p-3';
            document.body.appendChild(toastContainer);
        }
        toastContainer.appendChild(toastElement);
        const toast = new bootstrap.Toast(toastElement, { autohide: true, delay: 5000 });
        toast.show();
        toastElement.addEventListener('hidden.bs.toast', () => { toastElement.remove(); });
    }
}