import Plugin from 'src/plugin-system/plugin.class';
import HttpClient from 'src/service/http-client.service'; // For fetching alternative data

export default class BuyButtonClickLoggerPlugin extends Plugin {
    static options = {
        // Selector for the element where OemAlternativePopupPlugin is initialized
        popupPluginHolderSelector: '[data-oem-alternative-popup]',
    };

    init() {
        this._registerEvents();
        // console.log('BuyButtonClickLoggerPlugin initialized on:', this.el);
    }

    _registerEvents() {
        this.el.addEventListener('click', this._handleButtonClick.bind(this));
    }

    _handleButtonClick(event) {
        // console.log('_handleButtonClick', event);
        const buyButton = event.target.closest('.btn-buy');

        if (buyButton && this.el.contains(buyButton)) {
            const form = buyButton.closest('form');
            if (form) {
                const remAlternativesInput = form.querySelector('input[name="topdata-rem-alternatives"]');

                if (remAlternativesInput) {
                    event.preventDefault();
                    event.stopPropagation();
                    console.log('Form contains "topdata-rem-alternatives" input. Preventing default and fetching data.');

                    const oemProductId = this._getProductId(buyButton, form);
                    if (!oemProductId) {
                        console.error('Could not determine OEM Product ID. Cannot show popup.');
                        this._fallbackSubmit(form); // Allow original action if ID is missing
                        return;
                    }

                    // Get the quantity from the form
                    const quantity = this._getQuantity(form);

                    // Fetch details for the OEM and its best REM.
                    this._fetchAlternativeDataAndShowPopup(oemProductId, quantity, form);

                } else {
                    // console.log('Form does NOT contain "topdata-rem-alternatives" input. Allowing default action.');
                }
            } else {
                // console.log('.btn-buy button is not inside a form. Allowing default action.');
            }
        }
    }

    _getProductId(buyButton, form) { // Now accepts form as an argument
        if (buyButton.dataset.productId) {
            return buyButton.dataset.productId;
        }
        if (form) { // Use the passed form
            const productIdInput = form.querySelector('input[name="productId"]');
            if (productIdInput && productIdInput.value) {
                return productIdInput.value;
            }
            const lineItemIdInput = form.querySelector('input[type="hidden"][name$="[id]"]');
            if (lineItemIdInput && lineItemIdInput.value) {
                return lineItemIdInput.value;
            }
        }
        return null;
    }

    _getQuantity(form) {
        // Shopware's standard add-to-cart forms use a name like `lineItems[...][quantity]`
        // so we use a selector that checks for a name ending with `[quantity]`.
        const quantityInput = form.querySelector('input[name$="[quantity]"]');
        if (quantityInput && quantityInput.value) {
            const quantity = parseInt(quantityInput.value, 10);
            return quantity > 0 ? quantity : 1;
        }
        return 1; // Default to 1 if not found
    }

    _fetchAlternativeDataAndShowPopup(oemProductId, quantity, originalForm) {
        const client = new HttpClient();
        const route = `/topdata-linked-oem-rem/rem-alternatives-to-oem-product/${oemProductId}`;

        console.log(`Fetching alternative data for OEM ID: ${oemProductId} from ${route}`);

        client.get(route, (response) => {
            try {
                const data = JSON.parse(response);

                if (data.success && data.showPopup) {
                    const popupPluginEl = document.querySelector(this.options.popupPluginHolderSelector);
                    if (popupPluginEl) {
                        const popupInstance = window.PluginManager.getPluginInstanceFromElement(popupPluginEl, 'OemAlternativePopupPlugin');
                        if (popupInstance) {
                            // Pass the quantity to the popup's show method
                            popupInstance.show(
                                data.oemProductData,
                                data.alternatives,
                                oemProductId,
                                quantity
                            );
                        } else {
                            console.error('OemAlternativePopupPlugin instance not found.');
                        }
                    } else {
                        console.error(`Element ${this.options.popupPluginHolderSelector} not found.`);
                    }
                } else {
                    console.log('No popup required or data fetch failed, proceeding with original add to cart.', data);
                    this._fallbackSubmit(originalForm);
                }
            } catch (e) {
                console.error('Failed to parse JSON response:', e);
                console.error('Raw Server Response:', response);
                this._fallbackSubmit(originalForm);
            }
        }, (errorResponse) => {
            console.error('AJAX error fetching alternative data:', errorResponse);
            this._fallbackSubmit(originalForm);
        });
    }

    _fallbackSubmit(formElement) {
        alert("something went wrong, please check the console");
        return;

        if (formElement && typeof formElement.submit === 'function') {
            console.log('Executing fallback: submitting form programmatically.');
            formElement.submit();
        } else {
            console.warn('Fallback: Could not submit form programmatically. User may need to click again.');
        }
    }
}