<?php declare(strict_types=1);

namespace Topdata\TopdataLinkedOemRemSW6\Subscriber;

use Shopware\Core\Checkout\Cart\Event\BeforeLineItemAddedEvent;
use Shopware\Core\Checkout\Cart\LineItem\LineItem;
use Shopware\Core\Content\Product\SalesChannel\SalesChannelProductEntity;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\System\Currency\CurrencyFormatter;
use Shopware\Core\System\SalesChannel\Entity\SalesChannelRepository;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Topdata\TopdataLinkedOemRemSW6\Service\OemRemSavingsService;
use TopdataSoftwareGmbH\Util\UtilDebug;

// Ensure this is the updated DTO

/**
 * This class is a subscriber that listens to the BeforeLineItemAddedEvent.
 * It checks if an added product is an OEM product with REM alternatives.
 * If so, it determines the best REM alternatives, stores their data in the session for a frontend popup,
 * and prevents the original OEM product from being added to the cart.
 */
class TRASH____CartManipulationSubscriber implements EventSubscriberInterface
{
    private const SESSION_POPUP_FLAG            = 'show_oem_rem_popup';
    private const SESSION_OEM_PRODUCT_ID        = 'oem_product_id_for_popup';
    private const SESSION_OEM_PRODUCT_DATA      = 'oem_product_data_for_popup';
    private const SESSION_REM_ALTERNATIVES_DATA = 'rem_alternatives_data_for_popup';
    private const MAX_ALTERNATIVES_FOR_POPUP    = 5; // Configure how many alternatives to show

    public function __construct(
        private readonly OemRemSavingsService   $savingsService,
        private readonly RequestStack           $requestStack,
        private readonly SalesChannelRepository $productRepository, // For fetching full product entities
        private readonly CurrencyFormatter      $currencyFormatter
    )
    {
    }

    public static function getSubscribedEvents(): array
    {
        return [
            BeforeLineItemAddedEvent::class => 'onBeforeLineItemAdded',
        ];
    }

    public function onBeforeLineItemAdded(BeforeLineItemAddedEvent $event): void
    {
        $lineItem = $event->getLineItem();
        $salesChannelContext = $event->getSalesChannelContext();

        if ($lineItem->getType() !== LineItem::PRODUCT_LINE_ITEM_TYPE) {
            return;
        }

        $oemProductId = $lineItem->getReferencedId();
        if ($oemProductId === null) {
            return;
        }

        // Fetch the OEM product entity being added to the cart.
        // It needs 'properties' and 'cover.media' associations for the DTO and popup.
        $oemProductCriteria = (new Criteria([$oemProductId]))
            ->addAssociation('properties') // For SavingsOverOemDTO
            ->addAssociation('cover.media'); // For popup image

        /** @var SalesChannelProductEntity|null $oemProductEntity */
        $oemProductEntity = $this->productRepository->search($oemProductCriteria, $salesChannelContext)->get($oemProductId);

        if (!$oemProductEntity) {
            // Should not happen if it's a valid product line item, but good to check.
            return;
        }

        // Use the new method to find best REM alternatives for this OEM product.
        // This method already handles finding REM products and creating SavingsOverOemDTOs.
        $bestAlternativeDTOs = $this->savingsService->findBestRemAlternatives(
            $oemProductEntity, // Pass the full OEM product entity with associations
            $salesChannelContext,
            self::MAX_ALTERNATIVES_FOR_POPUP
        );

        if (empty($bestAlternativeDTOs)) {
            // No suitable REM alternatives found, or no savings. Let the OEM product be added.
            return;
        }

//        UtilDebug::dd($bestAlternativeDTOs);

        // Prepare data for the popup
        $session = $this->requestStack->getSession();
        $this->clearPopupSessionData($session); // Clear any old data first

        // Prepare OEM product data for the popup
        $oemProductDataForPopup = $this->prepareProductDataForPopup($oemProductEntity, $salesChannelContext);
        $session->set(self::SESSION_OEM_PRODUCT_DATA, $oemProductDataForPopup);
        $session->set(self::SESSION_OEM_PRODUCT_ID, $oemProductId); // Keep original OEM ID

        // Prepare REM alternatives data
        $remAlternativesDataForPopup = [];
        $remProductIdsToFetch = [];
        foreach ($bestAlternativeDTOs as $dto) {
            // The DTO already contains the REM product entity, which should have properties loaded.
            // We might need to ensure its cover is loaded if not already by findBestRemAlternatives.
            $remProductIdsToFetch[] = $dto->getRemProduct()->getId();
        }

        // Fetch full REM product entities if their cover images are not guaranteed to be loaded by the DTO's REM product
        // If OemRemSavingsService ensures REM products in DTOs have 'cover.media', this separate fetch might be optimized.
        // For now, let's assume we need to fetch them with cover for safety.
        if (!empty($remProductIdsToFetch)) {
            $remProductsCriteria = (new Criteria(array_unique($remProductIdsToFetch)))
                ->addAssociation('cover.media'); // Crucial for REM product images
            $remProductEntities = $this->productRepository->search($remProductsCriteria, $salesChannelContext)->getElements();
        } else {
            $remProductEntities = [];
        }


        foreach ($bestAlternativeDTOs as $dto) {
            /** @var SalesChannelProductEntity|null $remEntityFromDto */
            $remEntityFromDto = $dto->getRemProduct(); // This should have properties

            /** @var SalesChannelProductEntity|null $fullRemEntity */
            $fullRemEntity = $remProductEntities[$remEntityFromDto->getId()] ?? $remEntityFromDto; // Prefer fully fetched entity for cover

            if ($fullRemEntity) {
                $remAlternativesDataForPopup[] = [
                    'productData'      => $this->prepareProductDataForPopup($fullRemEntity, $salesChannelContext),
                    'productId'        => $fullRemEntity->getId(),
                    'savingsAbsolute'  => $this->currencyFormatter->formatCurrencyByLanguage(
                        $dto->getPriceSavingAbsolute(),
                        $salesChannelContext->getCurrency()->getIsoCode(),
                        $salesChannelContext->getContext()->getLanguageId(),
                        $salesChannelContext->getContext(),
                    ),
                    'savingsPercent'   => round($dto->getPriceSavingPercentAbsolute(), 2) . '%',
                    // Add per-unit savings if available and you want to display them
                    'pricePerUnitRem'  => $dto->getPricePerUnitRem() !== null ? $this->currencyFormatter->formatCurrencyByLanguage(
                        $dto->getPricePerUnitRem(),
                        $salesChannelContext->getCurrency()->getIsoCode(),
                        $salesChannelContext->getContext()->getLanguageId(),
                        $salesChannelContext->getContext(),
                    ) : null,
                    'pricePerUnitOem'  => $dto->getPricePerUnitOem() !== null ? $this->currencyFormatter->formatCurrencyByLanguage(
                        $dto->getPricePerUnitOem(),
                        $salesChannelContext->getCurrency()->getIsoCode(),
                        $salesChannelContext->getContext()->getLanguageId(),
                        $salesChannelContext->getContext(),
                    ) : null,
                    'capacityUnit'     => $dto->getCapacityUnit(),
                    'capacityRem'      => $dto->getCapacityRem(),
                    'capacityOem'      => $dto->getCapacityOem(),
                    'isCheaperPerUnit' => $dto->isCheaperPerUnit(),
                ];
            }
        }


        if (!empty($remAlternativesDataForPopup)) {
            $session->set(self::SESSION_REM_ALTERNATIVES_DATA, $remAlternativesDataForPopup);
            $session->set(self::SESSION_POPUP_FLAG, true);

            // Prevent the original OEM product from being added to the cart.
            // The cart object is live, so removing the line item should work.
            $cart = $event->getCart();
            if ($cart->has($lineItem->getId())) { // Check if it's still in cart (might be removed by other subscribers)
                $cart->remove($lineItem->getId());
                // After removing, we might need to tell Shopware not to proceed with adding it again.
                // For BeforeLineItemAddedEvent, simply removing it from the cart might be enough,
                // or you might need to manipulate the event's line item list if the event allows.
                // Shopware often re-evaluates the cart. The most robust way is to ensure it's GONE.
                // If problems persist, consider throwing a specific exception that your controller can catch
                // to redirect or display a message, though session flash is usually preferred for popups.
            }
            // If $event had a method like $event->setLineItems([]) or $event->stopPropagation() that would be cleaner.
            // For now, removing from cart is the direct approach.
            // Potentially, we could also set the quantity of the lineItem to 0, but remove is cleaner.
        }
    }

    /**
     * Helper to prepare product data for frontend popup.
     *
     * @param SalesChannelProductEntity $productEntity
     * @param SalesChannelContext $salesChannelContext
     * @return array
     */
    private function prepareProductDataForPopup(SalesChannelProductEntity $productEntity, SalesChannelContext $salesChannelContext): array
    {
        $name = $productEntity->getTranslated()['name'] ?? $productEntity->getName() ?? 'Unknown Product';
        $imageUrl = $productEntity->getCover()?->getMedia()?->getUrl() ?? ''; // Provide a fallback image URL if necessary
        $price = $this->currencyFormatter->formatCurrencyByLanguage(
            $productEntity->getCalculatedPrice()->getUnitPrice(), // Or getTotalPrice() depending on your needs
            $salesChannelContext->getCurrency()->getIsoCode(),
            $salesChannelContext->getContext()->getLanguageId(),
            $salesChannelContext->getContext(),
        );
        $productNumber = $productEntity->getProductNumber();

        return [
            'name'          => $name,
            'imageUrl'      => $imageUrl,
            'price'         => $price,
            'productNumber' => $productNumber,
            // Add any other details needed for the popup, e.g., product URL
            // 'url' => $this->generateUrl('frontend.detail.page', ['productId' => $productEntity->getId()]) // If you have router
        ];
    }

    private function clearPopupSessionData(\Symfony\Component\HttpFoundation\Session\SessionInterface $session): void
    {
        $session->remove(self::SESSION_POPUP_FLAG);
        $session->remove(self::SESSION_OEM_PRODUCT_ID);
        $session->remove(self::SESSION_OEM_PRODUCT_DATA);
        $session->remove(self::SESSION_REM_ALTERNATIVES_DATA);
        // Remove any other related session keys
        $session->remove('rem_product_id_for_popup'); // old key
        $session->remove('oem_product_data_for_popup'); // old key
        $session->remove('rem_product_data_for_popup'); // old key
        $session->remove('rem_product_savings_for_popup'); // old key
        $session->remove('oem_alternative_popup_data'); // old key
    }
}
