import Plugin from 'src/plugin-system/plugin.class';
// DO NOT import from 'bootstrap'. This is the source of the conflict.
// import { Modal } from 'bootstrap';
import HttpClient from 'src/service/http-client.service';

export default class OemAlternativePopupPlugin extends Plugin {
    static options = {
        modalSelector: '#oemAlternativePopupModal', // From your Twig partial
        // Configuration for plugin actions after choosing an alternative (read from config.xml)
        // These will be passed via data attributes on the `[data-oem-alternative-popup]` element
        // or you can fetch them via an AJAX call if they are complex.
        // For now, we'll assume they might be set via data attributes for simplicity.
        redirectAction: 'detail', // Default, can be overridden
    };

    init() {
        this.modalElement = document.querySelector(this.options.modalSelector);
        if (!this.modalElement) {
            console.error('OEM Alternative Popup Modal element not found:', this.options.modalSelector);
            return;
        }


        // Initialize the Bootstrap Modal instance
        // THE KEY FIX: Use the Modal class from the global scope
        const { Modal } = window.bootstrap;
        this.bsModal = new Modal(this.modalElement);
        this.productAddedToCart = false; // Flag to track if modal was closed by its own buttons

        this._bindModalEvents();
        this._bindPopupActionButtons();

        // Read redirect config (example of reading from data attribute, set this in base.html.twig on the plugin's div)
        // <div data-oem-alternative-popup data-redirect-action="{{ config('TopdataLinkedOemRemSW6.config.oemAlternativeRedirectAction') }}"></div>
        if (this.el.dataset.redirectAction) {
            this.options.redirectAction = this.el.dataset.redirectAction;
        }

        console.log('OemAlternativePopupPlugin initialized and modal is ready.');
    }

    _bindModalEvents() {
        // Listen for when Bootstrap intends to hide the modal (e.g. when user presses ESC or clicks the backdrop)
        this.modalElement.addEventListener('hide.bs.modal', this._handleModalCloseAttempt.bind(this));
    }

    _bindPopupActionButtons() {
        const switchToAlternativeButton = this.modalElement.querySelector('.switch-to-alternative');
        if (switchToAlternativeButton) {
            switchToAlternativeButton.addEventListener('click', this._handleSwitchToAlternative.bind(this));
        }

        const continueWithOriginalButton = this.modalElement.querySelector('.continue-with-original');
        if (continueWithOriginalButton) {
            continueWithOriginalButton.addEventListener('click', this._handleContinueWithOriginal.bind(this));
        }
    }

    /**
     * Public method to show the popup.
     * Called by BuyButtonClickLoggerPlugin.
     *
     * @param {object} oemProductData - Data for the OEM product.
     * @param {object} remProductData - Data for the best REM alternative.
     * @param {string} savingsInfo - Formatted savings string.
     * @param {string} oemId - The ID of the original OEM product.
     * @param {string} remId - The ID of the REM alternative to be added if chosen.
     */
    show(oemProductData, remProductData, savingsInfo, oemId, remId) {
        this.currentOemProductData = oemProductData;
        this.currentRemProductData = remProductData;
        this.currentSavingsInfo = savingsInfo;
        this.currentOemId = oemId;
        this.currentRemId = remId;
        this.productAddedToCart = false; // Reset flag before showing

        this._populatePopup();
        this.bsModal.show();

        // this._displayFlashMessage('success', 'Operation completed successfully!');
        // this._displayFlashMessage('danger', 'An error occurred!');
        // this._displayFlashMessage('warning', 'Please check your input!');
        // this._displayFlashMessage('info', 'Information message!');

        console.log('OEM Alternative Popup shown with data:', { oemId, remId });
    }




    _openOffCanvasCart() {
        console.log('attempting to open off-canvas cart...');
        const offCanvasCartPluginInstances = window.PluginManager.getPluginInstances('OffCanvasCart');

        if (!offCanvasCartPluginInstances || offCanvasCartPluginInstances.length === 0) {
            console.error('OffCanvasCartPlugin instance not found. Is the default cart button present and initialized?');
            return;
        }

        // Usually, there's only one main off-canvas cart. Take the first one.
        const offCanvasCartPlugin = offCanvasCartPluginInstances[0];

        if (offCanvasCartPlugin && typeof offCanvasCartPlugin.openOffCanvas === 'function') {
            const offcanvasCartUrl = window.router['frontend.cart.offcanvas'];
            if (!offcanvasCartUrl) {
                console.error('Off-canvas cart URL (frontend.cart.offcanvas) not found in window.router.');
                return;
            }

            // Call the public method from the original plugin
            offCanvasCartPlugin.openOffCanvas(offcanvasCartUrl, false, () => {
                console.log('Off-canvas cart opened via custom trigger.');
            });
        } else {
            console.error('Could not access OffCanvasCartPlugin or its openOffCanvas method.');
        }
    }


    /**
     * Helper method to create and set an image within a container element.
     * Clears the container first.
     * @param {HTMLElement} containerEl The div element to place the image in.
     * @param {string|null} imageUrl The URL of the image.
     * @param {string} altText The alt text for the image (e.g., product name).
     * @param {string} defaultAltText Fallback alt text if the primary altText is falsy.
     */
    _setImageInContainer(containerEl, imageUrl, altText, defaultAltText = 'Product image') {
        if (!containerEl) return;

        containerEl.innerHTML = ''; // Clear any existing content

        if (imageUrl) {
            const imgElement = document.createElement('img');
            imgElement.src = imageUrl;
            imgElement.alt = altText || defaultAltText;
            imgElement.classList.add('img-fluid'); // <--- ADD THIS LINE
            containerEl.appendChild(imgElement);
        } else {
            // Optional: What to do if there's no image URL
            // containerEl.textContent = 'Image not available';
        }
    }

    _populatePopup() {
        if (!this.modalElement) return;

        // OEM Product Elements
        const oemImageContainerEl = this.modalElement.querySelector('.oem-product-image');
        const oemNameEl = this.modalElement.querySelector('.oem-product-name');
        const oemPriceEl = this.modalElement.querySelector('.oem-product-price');

        // REM Product Elements
        const remImageContainerEl = this.modalElement.querySelector('.rem-product-image');
        const remNameEl = this.modalElement.querySelector('.rem-product-name');
        const remPriceEl = this.modalElement.querySelector('.rem-product-price');
        const remSavingsEl = this.modalElement.querySelector('.rem-product-savings');

        // --- Populate OEM Product Data ---
        if (this.currentOemProductData) {
            this._setImageInContainer(
                oemImageContainerEl,
                this.currentOemProductData.imageUrl,
                this.currentOemProductData.name,
                'OEM Product Image'
            );
            if (oemNameEl) oemNameEl.textContent = this.currentOemProductData.name || 'N/A';
            if (oemPriceEl) oemPriceEl.textContent = this.currentOemProductData.price || 'N/A';
        } else {
            this._setImageInContainer(oemImageContainerEl, null, '', 'OEM Product Image');
            if (oemNameEl) oemNameEl.textContent = 'N/A';
            if (oemPriceEl) oemPriceEl.textContent = 'N/A';
        }

        // --- Populate REM Product Data ---
        if (this.currentRemProductData) {
            this._setImageInContainer(
                remImageContainerEl,
                this.currentRemProductData.imageUrl,
                this.currentRemProductData.name,
                'Remanufactured Product Image'
            );
            if (remNameEl) remNameEl.textContent = this.currentRemProductData.name || 'N/A';
            if (remPriceEl) remPriceEl.textContent = this.currentRemProductData.price || 'N/A';
            if (remSavingsEl) remSavingsEl.textContent = this.currentSavingsInfo || '';
        } else {
            this._setImageInContainer(remImageContainerEl, null, '', 'Remanufactured Product Image');
            if (remNameEl) remNameEl.textContent = 'N/A';
            if (remPriceEl) remPriceEl.textContent = 'N/A';
            if (remSavingsEl) remSavingsEl.textContent = '';
        }
    }


    _handleModalCloseAttempt() {
        // this._displayFlashMessage('info', '_handleModalCloseAttempt');
        console.log('_handleModalCloseAttempt ................. productAddedToCart: ' + (this.productAddedToCart ? 'true' : 'false'), this);
        if (this.productAddedToCart) {
            console.log('[OEM Alternative Popup] Closing popup - Product was already added to cart.');
            return false;
        }
        // if the product was not added to cart, we add it now
        this._handleContinueWithOriginal();
    }

    _handleContinueWithOriginal() {
        console.log('Adding original OEM product:', this.currentOemId);
        this._addProductToCart(this.currentOemId);
    }

    _handleSwitchToAlternative() {
        console.log('Switching to REM alternative:', this.currentRemId);
        this._addProductToCart(this.currentRemId);
    }

    _addProductToCart(productId) {
        this.productAddedToCart = true;
        const client = new HttpClient();
        const data = new FormData();
        data.append('productId', productId);
        data.append('quantity', 1); // Assuming quantity 1 for now

        const route = '/topdata-linked-oem-rem/add-to-cart';

        client.post(route, data, (responseText) => {
            const responseData = JSON.parse(responseText);
            if (responseData.success) {
                // this._displayFlashMessage('success', `[OEM Alternative Popup] Product ${productId} added to cart successfully. Server response:` + JSON.stringify(responseData));
                this.bsModal.hide();
                this._openOffCanvasCart();
            } else {
                // hopefully this never happens, but if it happens the customer will see an error message, not just a console error
                this._displayFlashMessage('error', `[OEM Alternative Popup] Failed to add product ${productId} to cart:`, responseData.message || 'Unknown error');
            }
        });
    }

    /**
     * Create and show a bootstrap toast notification
     *
     * @param {string} type - 'success', 'danger', 'warning', 'info'
     * @param {string} message - The message to display
     */
    _displayFlashMessage(type, message) {
        // Create toast element
        const toastElement = document.createElement('div');
        toastElement.className = `toast align-items-center text-white bg-${type} border-0`;
        toastElement.setAttribute('role', 'alert');
        toastElement.innerHTML = `
        <div class="d-flex">
            <div class="toast-body">${message}</div>
            <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
        </div>
    `;

        // Add to toast container or create one
        let toastContainer = document.querySelector('.toast-container');
        if (!toastContainer) {
            toastContainer = document.createElement('div');
            toastContainer.className = 'toast-container position-fixed top-0 end-0 p-3';
            document.body.appendChild(toastContainer);
        }

        toastContainer.appendChild(toastElement);

        // Initialize and show toast
        const toast = new bootstrap.Toast(toastElement, {
            autohide: true,
            delay: 5000
        });
        toast.show();

        // Remove element after hide
        toastElement.addEventListener('hidden.bs.toast', () => {
            toastElement.remove();
        });
    }


}