<?php

namespace Topdata\TopdataLinkedOemRemSW6\DTO;

use Shopware\Core\Content\Product\SalesChannel\SalesChannelProductEntity;
use Shopware\Core\Content\Property\Aggregate\PropertyGroupOption\PropertyGroupOptionEntity;
use Shopware\Core\Checkout\Cart\Price\Struct\CalculatedPrice; // For type hinting
use Topdata\TopdataFoundationSW6\Util\CliLogger; // Assuming you have this for logging
use Topdata\TopdataLinkedOemRemSW6\Traits\CreateFromDictTrait;

/**
 * 11/2023 created
 * 04/2024 TODO: compare per prices relative to PaperCapacity (Now addressed)
 * 05/2024 Enhanced for per-unit price comparison
 *
 * @method static SavingsOverOemDTO createFromDict(array $dict)
 */
class SavingsOverOemDTO
{
    use CreateFromDictTrait;

    private SalesChannelProductEntity $oemProduct;
    private SalesChannelProductEntity $remProduct; // Added for clarity and access
    private float $priceOem;
    private float $priceRem;
    private ?int $capacityOem = null; // e.g., pages, ml
    private ?int $capacityRem = null; // e.g., pages, ml
    private ?string $capacityUnit = null; // Heuristic attempt: "pages", "ml", etc.
    private bool $haveComparableCapacity;

    private float $priceSavingAbsolute;
    private float $priceSavingPercentAbsolute;

    // Per-unit calculations
    private ?float $pricePerUnitOem = null;
    private ?float $pricePerUnitRem = null;
    private ?float $priceSavingPerUnitAbsolute = null;
    private ?float $priceSavingPerUnitRelative = null;

    /**
     * Factory method
     *
     * @param SalesChannelProductEntity $oemProduct
     * @param SalesChannelProductEntity $remProduct
     * @param string|null $sw6IdCapacity The UUID of the property group representing capacity.
     */
    public static function createFromOemProductAndRemProduct(
        SalesChannelProductEntity $oemProduct,
        SalesChannelProductEntity $remProduct,
        ?string $sw6IdCapacity
    ): ?self // Can return null if essential data (like price) is missing
    {
        $oemPrice = self::getProductPrice($oemProduct);
        $remPrice = self::getProductPrice($remProduct);

        if ($oemPrice === null || $remPrice === null) {
            CliLogger::warning("Could not determine price for OEM ({$oemProduct->getId()}) or REM ({$remProduct->getId()}). Cannot create SavingsOverOemDTO.");
            return null;
        }

        $dict = [
            'oemProduct' => $oemProduct,
            'remProduct' => $remProduct,
            'priceOem' => $oemPrice,
            'priceRem' => $remPrice,
            'priceSavingAbsolute' => $oemPrice - $remPrice,
            'priceSavingPercentAbsolute' => ($oemPrice > 0) ? (($oemPrice - $remPrice) / $oemPrice) * 100 : 0,
            'capacityOem' => null,
            'capacityRem' => null,
            'capacityUnit' => null,
            'haveComparableCapacity' => false,
            'pricePerUnitOem' => null,
            'pricePerUnitRem' => null,
            'priceSavingPerUnitAbsolute' => null,
            'priceSavingPerUnitRelative' => null,
        ];

        if ($sw6IdCapacity) {
            $capacityOemData = self::extractCapacityData($oemProduct, $sw6IdCapacity);
            $capacityRemData = self::extractCapacityData($remProduct, $sw6IdCapacity);

            $dict['capacityOem'] = $capacityOemData['value'];
            $dict['capacityRem'] = $capacityRemData['value'];

            // Attempt to determine a common unit (simplistic approach)
            // A more robust system might involve a predefined list of units or conversion factors
            if ($capacityOemData['unit'] !== null) {
                $dict['capacityUnit'] = $capacityOemData['unit'];
            } elseif ($capacityRemData['unit'] !== null) {
                $dict['capacityUnit'] = $capacityRemData['unit'];
            }

            // Check for comparable capacity: both must have capacity and units must match if both specified
            if ($dict['capacityOem'] !== null && $dict['capacityRem'] !== null) {
                if ($capacityOemData['unit'] !== null && $capacityRemData['unit'] !== null && $capacityOemData['unit'] !== $capacityRemData['unit']) {
                    $dict['haveComparableCapacity'] = false; // Units differ
                    CliLogger::info("Capacity units differ for OEM {$oemProduct->getId()} ('{$capacityOemData['unit']}') and REM {$remProduct->getId()} ('{$capacityRemData['unit']}'). Treating as non-comparable for per-unit price.");
                } else {
                    $dict['haveComparableCapacity'] = true;
                }
            } else {
                $dict['haveComparableCapacity'] = false; // One or both lack capacity
            }


            if ($dict['haveComparableCapacity']) {
                if ($dict['capacityOem'] > 0) {
                    $dict['pricePerUnitOem'] = $dict['priceOem'] / $dict['capacityOem'];
                }
                if ($dict['capacityRem'] > 0) {
                    $dict['pricePerUnitRem'] = $dict['priceRem'] / $dict['capacityRem'];
                }

                if ($dict['pricePerUnitOem'] !== null && $dict['pricePerUnitRem'] !== null) {
                    $dict['priceSavingPerUnitAbsolute'] = $dict['pricePerUnitOem'] - $dict['pricePerUnitRem'];
                    if ($dict['pricePerUnitOem'] > 0) {
                        $dict['priceSavingPerUnitRelative'] = ($dict['priceSavingPerUnitAbsolute'] / $dict['pricePerUnitOem']) * 100;
                    }
                }
            }
        }

        return self::createFromDict($dict);
    }

    private static function getProductPrice(SalesChannelProductEntity $product): ?float
    {
        $calculatedPrice = $product->getCalculatedPrice();
        if ($calculatedPrice instanceof CalculatedPrice) {
            // Use unit price if available and seems appropriate (e.g. for base product)
            // For variants, getTotalPrice() might be more reliable
            // This depends on your product setup. TotalPrice is generally safer.
            return $calculatedPrice->getTotalPrice();
        }
        CliLogger::warning("Product {$product->getId()} does not have a CalculatedPrice.");
        return null;
    }

    /**
     * Extracts an integer capacity and a unit string from a property value.
     * Example property value: "1000 Blatt", "approx. 500 pages", "XL 70ml Black"
     * @return array{value: ?int, unit: ?string}
     */
    private static function extractCapacityData(SalesChannelProductEntity $product, string $propertyGroupIdCapacity): array
    {
        $properties = $product->getProperties();
        if ($properties === null) {
            return ['value' => null, 'unit' => null];
        }

        /** @var PropertyGroupOptionEntity|null $propertyOption */
        $propertyOption = $properties->filterByGroupId($propertyGroupIdCapacity)->first();

        if ($propertyOption === null || $propertyOption->getName() === null) {
            return ['value' => null, 'unit' => null];
        }

        $name = $propertyOption->getTranslated()['name'] ?? $propertyOption->getName(); // Prefer translated name
        $value = null;
        $unit = null;

        // Try to extract number: "1000", "5.400", "1,200"
        if (preg_match('/(\d+[\.,]?\d*)/', $name, $matches)) {
            $numericPart = str_replace(['.', ','], ['', '.'], $matches[1]); // Normalize to use . as decimal separator
            // We expect integer capacities for pages/ml, so we can cast to int after removing decimals for simplicity.
            // If fractional capacities are possible, use (float) and adjust property type.
            $value = (int) filter_var($numericPart, FILTER_SANITIZE_NUMBER_INT);
        }


        // Try to extract common units (extend this list as needed)
        // This is a heuristic. A more robust solution would be a dedicated "unit" property
        // or more structured capacity property values.
        $nameLower = strtolower($name);
        if (str_contains($nameLower, 'blatt') || str_contains($nameLower, 'seiten') || str_contains($nameLower, 'pages')) {
            $unit = 'pages';
        } elseif (str_contains($nameLower, 'ml')) {
            $unit = 'ml';
        } elseif ($value !== null && ($nameLower === (string)$value || preg_match("/^\s*\d+\s*$/", $nameLower))) {
            // If the property is just a number, we don't know the unit
            $unit = 'units'; // Generic unit
        }
        // Add more unit detection logic if needed

        return ['value' => $value > 0 ? $value : null, 'unit' => $unit];
    }


    // --- Getter methods ---
    public function getOemProduct(): SalesChannelProductEntity { return $this->oemProduct; }
    public function getRemProduct(): SalesChannelProductEntity { return $this->remProduct; }
    public function getPriceOem(): float { return $this->priceOem; }
    public function getPriceRem(): float { return $this->priceRem; }
    public function getCapacityOem(): ?int { return $this->capacityOem; }
    public function getCapacityRem(): ?int { return $this->capacityRem; }
    public function getCapacityUnit(): ?string { return $this->capacityUnit; }
    public function haveComparableCapacity(): bool { return $this->haveComparableCapacity; }
    public function getPriceSavingAbsolute(): float { return $this->priceSavingAbsolute; }
    public function getPriceSavingPercentAbsolute(): float { return $this->priceSavingPercentAbsolute; }
    public function getPricePerUnitOem(): ?float { return $this->pricePerUnitOem; }
    public function getPricePerUnitRem(): ?float { return $this->pricePerUnitRem; }
    public function getPriceSavingPerUnitAbsolute(): ?float { return $this->priceSavingPerUnitAbsolute; }
    public function getPriceSavingPerUnitRelative(): ?float { return $this->priceSavingPerUnitRelative; }

    /**
     * Helper to check if per-unit comparison is meaningful and REM is cheaper per unit.
     */
    public function isCheaperPerUnit(): bool
    {
        return $this->haveComparableCapacity && $this->pricePerUnitRem !== null && $this->pricePerUnitOem !== null && $this->pricePerUnitRem < $this->pricePerUnitOem;
    }
}