<?php declare(strict_types=1);

namespace Topdata\TopdataLinkedOemRemSW6\Service;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Exception;
use Shopware\Core\System\SystemConfig\SystemConfigService;
use Topdata\TopdataFoundationSW6\Util\CliLogger;
use Topdata\TopdataLinkedOemRemSW6\Constants\GlobalPluginConstants;

/**
 * Service for managing the discovery, validation, and persistence of links between OEM and REM products
 * in the `topdata_lor_oem_to_rem` table.
 *
 * 05/2025 created as part of OemRemService refactoring
 */
class OemRemLinkerService
{
    public function __construct(
        private readonly Connection $connection,
        private readonly SystemConfigService $systemConfigService
    ) {
    }

    /**
     * Finds products that are linked between OEM and REM based on property values.
     * This method performs a complex database query to identify REM products that reference OEM products
     * and vice versa, using property groups for MPN and ReferenceOEM.
     * 
     * @return array Array of linked product pairs with their details
     */
    private function _findLinkedProducts(): array
    {
        CliLogger::info("Starting _findLinkedProducts...");

        // ---- Retrieve configured and constant IDs ----
        $propertyGroupIdMpn = $this->systemConfigService->get('TopdataLinkedOemRemSW6.config.propertyGroupIdMpn');
        $propertyGroupIdReferenceOem = $this->systemConfigService->get('TopdataLinkedOemRemSW6.config.propertyGroupIdReferenceOem');
        $languageIdGerman = GlobalPluginConstants::LANGUAGE_ID_GERMAN;

        // ---- Pre-flight checks for critical IDs ----
        $allCriticalIdsValid = true;

        // Check Property Group ID for MPN
        if (empty($propertyGroupIdMpn)) {
            CliLogger::error("SystemConfig 'TopdataLinkedOemRemSW6.config.propertyGroupIdMpn' is not set or is empty.");
            $allCriticalIdsValid = false;
        } else {
            $existsMpnGroup = $this->connection->fetchOne(
                "SELECT 1 FROM property_group WHERE id = UNHEX(:id)",
                ['id' => $propertyGroupIdMpn]
            );
            if (!$existsMpnGroup) {
                CliLogger::warning("Configured PropertyGroupID for MPN ('{$propertyGroupIdMpn}') not found in 'property_group' table.");
                $allCriticalIdsValid = false;
            } else {
                CliLogger::info("Configured PropertyGroupID for MPN ('{$propertyGroupIdMpn}') confirmed in DB.");
            }
        }

        // Check Property Group ID for Reference OEM
        if (empty($propertyGroupIdReferenceOem)) {
            CliLogger::error("SystemConfig 'TopdataLinkedOemRemSW6.config.propertyGroupIdReferenceOem' is not set or is empty.");
            $allCriticalIdsValid = false;
        } else {
            $existsRefOemGroup = $this->connection->fetchOne(
                "SELECT 1 FROM property_group WHERE id = UNHEX(:id)",
                ['id' => $propertyGroupIdReferenceOem]
            );
            if (!$existsRefOemGroup) {
                CliLogger::warning("Configured PropertyGroupID for ReferenceOEM ('{$propertyGroupIdReferenceOem}') not found in 'property_group' table.");
                $allCriticalIdsValid = false;
            } else {
                CliLogger::info("Configured PropertyGroupID for ReferenceOEM ('{$propertyGroupIdReferenceOem}') confirmed in DB.");
            }
        }

        // Check Language ID for German
        if (empty($languageIdGerman)) {
            // This should ideally not happen if the constant is defined
            CliLogger::error("GlobalPluginConstants::LANGUAGE_ID_GERMAN is empty. This is unexpected.");
            $allCriticalIdsValid = false;
        } else {
            $existsLanguage = $this->connection->fetchOne(
                "SELECT 1 FROM language WHERE id = UNHEX(:id)",
                ['id' => $languageIdGerman]
            );
            if (!$existsLanguage) {
                CliLogger::warning("Defined LanguageID for German ('{$languageIdGerman}') not found in 'language' table.");
                $allCriticalIdsValid = false;
            } else {
                CliLogger::info("Defined LanguageID for German ('{$languageIdGerman}') confirmed in DB.");
            }
        }

        if (!$allCriticalIdsValid) {
            CliLogger::error("One or more critical IDs (PropertyGroup, Language) are missing from config/constants or not found in DB. Aborting link finding process. Please check plugin configuration and data integrity.");
            return [];
        }

        // ---- step 1 find rem products which have a reference oem
        $SQL = "
                SELECT 
                        LOWER(HEX(pp1.product_id)) AS remProductId, 
                        -- p1.product_number          AS remProductNumber, 
                        LOWER(HEX(pp2.product_id)) AS oemProductId, 
                        -- p2.product_number          AS oemProductNumber, 
                        pgot1.name                 AS referenceOem
                
                         FROM product_property                  pp1
                         JOIN property_group_option             pgo1  ON pgo1.id = pp1.property_group_option_id AND pgo1.property_group_id = UNHEX(:propertyGroupIdReferenceOem)
                         JOIN property_group_option_translation pgot1 ON pgot1.property_group_option_id=pgo1.id AND pgot1.language_id=UNHEX(:languageIdGerman)
                         -- too slow: JOIN product                           p1    ON p1.id = pp1.product_id AND p1.active = 1
                    
                         JOIN property_group_option             pgo2  ON pgo2.property_group_id = UNHEX(:propertyGroupIdMpn)  
                         JOIN property_group_option_translation pgot2 ON pgot2.property_group_option_id=pgo2.id AND pgot2.language_id=UNHEX(:languageIdGerman) AND pgot2.name = pgot1.name
                         JOIN product_property                  pp2   ON pp2.property_group_option_id = pgo2.id
                         -- too slow: JOIN product                           p2    ON p2.id = pp2.product_id AND p2.active = 1
                
                ORDER BY referenceOem ASC
        ";

        $bind = [
            'propertyGroupIdMpn'          => $propertyGroupIdMpn,
            'propertyGroupIdReferenceOem' => $propertyGroupIdReferenceOem,
            'languageIdGerman'            => $languageIdGerman,
        ];
        $rows = $this->connection->fetchAllAssociative($SQL, $bind);

        if(count($rows) === 0) {
            CliLogger::warning("Main SQL query returned 0 rows for linked REM/OEM products.");
            CliLogger::info("This means no products could be linked based on the current property configuration and data.");
            CliLogger::info("Further debugging information for property options will follow.");
            CliLogger::info("Main SQL for linking: " . $SQL);
            CliLogger::dump("Bind parameters for main SQL: ", $bind);
            $this->_debugPropertyOptions($bind); // Call the deeper debug method
            return []; // Return early as no links were found
        }

        CliLogger::info("Found " . count($rows) . " potential REM/OEM product links (before checking product activity).");

        // ---- step 2 - fetch products from db and make loopup map
        $productIdsRem = array_column($rows, 'remProductId');
        $productIdsOem = array_column($rows, 'oemProductId');
        $productIds = array_unique(array_merge($productIdsRem, $productIdsOem));

        if (empty($productIds)) {
            CliLogger::info("No product IDs extracted from SQL results. This is unexpected if rows > 0.");
            return [];
        }

        $rows2 = $this->connection->fetchAllAssociative("SELECT LOWER(HEX(id)) AS id, product_number, active FROM product WHERE id IN (:ids)",
            ['ids' => array_map('hex2bin', $productIds)],
            ['ids' => Connection::PARAM_STR_ARRAY]
        );
        $mapProductDetails = array_column($rows2, null, 'id'); // Create a map with product ID as key
        CliLogger::info("Fetched details for " . count($mapProductDetails) . " products from DB for " . count($productIds) . " unique IDs.");


        // ---- step 3 - combine data from step1 and step2, filtering by active status
        $rows3 = [];
        foreach ($rows as $idx => $row) {
            $remProductDetail = $mapProductDetails[$row['remProductId']] ?? null;
            $oemProductDetail = $mapProductDetails[$row['oemProductId']] ?? null;

            if (!$remProductDetail) {
                CliLogger::warning("REM product details not found for ID {$row['remProductId']}. Skipping row: " . json_encode($row));
                continue;
            }
            if (!$oemProductDetail) {
                CliLogger::warning("OEM product details not found for ID {$row['oemProductId']}. Skipping row: " . json_encode($row));
                continue;
            }

            $remActive = (int)$remProductDetail['active'];
            $oemActive = (int)$oemProductDetail['active'];

            if (!$remActive || !$oemActive) {
                CliLogger::info("Skipping link: REM Product ({$row['remProductId']}) active: {$remActive}, OEM Product ({$row['oemProductId']}) active: {$oemActive}. Row: " . json_encode($row));
                continue;
            }

            $rows3[] = [
                'referenceOem'     => $row['referenceOem'],
                'remProductNumber' => $remProductDetail['product_number'] ?? '',
                'oemProductNumber' => $oemProductDetail['product_number'] ?? '',
                'remProductId'     => $row['remProductId'],
                'oemProductId'     => $row['oemProductId'],
            ];
        }
        CliLogger::info("Found " . count($rows3) . " active and validly linked REM/OEM product pairs.");
        return $rows3;
    }

    /**
     * Helper method to debug property options if the main query yields no results.
     * This is called if primary ID checks pass but the linking query still finds nothing.
     * Performs detailed diagnostics on property groups, options, and product assignments
     * to help identify why product linking might be failing.
     *
     * @param array $bindParams Parameters used in the main SQL (propertyGroupIds, languageId)
     */
    private function _debugPropertyOptions(array $bindParams): void
    {
        CliLogger::info("--- Deeper Debugging: Checking Property Options ---");

        // Check options for Reference OEM Property Group
        $sqlRefOemOptions = "
            SELECT LOWER(HEX(pgo.id)) as option_id, pgot.name 
            FROM property_group_option pgo
            JOIN property_group_option_translation pgot ON pgot.property_group_option_id = pgo.id
            WHERE pgo.property_group_id = UNHEX(:propertyGroupIdReferenceOem)
              AND pgot.language_id = UNHEX(:languageIdGerman)
            LIMIT 10
        ";
        $refOemOptions = $this->connection->fetchAllAssociative($sqlRefOemOptions, [
            'propertyGroupIdReferenceOem' => $bindParams['propertyGroupIdReferenceOem'],
            'languageIdGerman' => $bindParams['languageIdGerman'],
        ]);
        if (empty($refOemOptions)) {
            CliLogger::warning("No property options found for ReferenceOEM group (ID: {$bindParams['propertyGroupIdReferenceOem']}) with German language (ID: {$bindParams['languageIdGerman']}). This is a likely cause for no links.");
        } else {
            CliLogger::info(count($refOemOptions) . " property options found for ReferenceOEM group and German language. First few examples: " . json_encode($refOemOptions));
        }

        // Check options for MPN Property Group
        $sqlMpnOptions = "
            SELECT LOWER(HEX(pgo.id)) as option_id, pgot.name 
            FROM property_group_option pgo
            JOIN property_group_option_translation pgot ON pgot.property_group_option_id = pgo.id
            WHERE pgo.property_group_id = UNHEX(:propertyGroupIdMpn)
              AND pgot.language_id = UNHEX(:languageIdGerman)
            LIMIT 10
        ";
        $mpnOptions = $this->connection->fetchAllAssociative($sqlMpnOptions, [
            'propertyGroupIdMpn' => $bindParams['propertyGroupIdMpn'],
            'languageIdGerman' => $bindParams['languageIdGerman'],
        ]);
        if (empty($mpnOptions)) {
            CliLogger::warning("No property options found for MPN group (ID: {$bindParams['propertyGroupIdMpn']}) with German language (ID: {$bindParams['languageIdGerman']}). This is a likely cause for no links.");
        } else {
            CliLogger::info(count($mpnOptions) . " property options found for MPN group and German language. First few examples: " . json_encode($mpnOptions));
        }

        // Check for matching names if both have options (conceptual check based on a sample)
        if (!empty($refOemOptions) && !empty($mpnOptions)) {
            // This is a more complex check, let's just see if there *could* be matches.
            // For a full check, we'd need all options, not just a sample.
            $refOemSampleNames = array_column($refOemOptions, 'name');
            $mpnSampleNames = array_column($mpnOptions, 'name');
            $intersectingSampleNames = array_intersect($refOemSampleNames, $mpnSampleNames);
            if (empty($intersectingSampleNames)) {
                CliLogger::info("Based on a sample of 10 options from each group, no common 'name' values were found. The main query relies on 'pgot2.name = pgot1.name'. If no names match across all options, no links will be found.");
            } else {
                CliLogger::info("Common 'name' values found in samples between ReferenceOEM and MPN options. Examples: " . json_encode(array_values($intersectingSampleNames)));
            }
        }

        // Check if any products are actually assigned these properties
        $sqlProductPropertyCheckRefOem = "
            SELECT COUNT(DISTINCT pp.product_id)
            FROM product_property pp
            JOIN property_group_option pgo ON pgo.id = pp.property_group_option_id
            WHERE pgo.property_group_id = UNHEX(:propertyGroupIdReferenceOem)
        ";
        $countProductsWithRefOemProp = $this->connection->fetchOne($sqlProductPropertyCheckRefOem, ['propertyGroupIdReferenceOem' => $bindParams['propertyGroupIdReferenceOem']]);
        CliLogger::info("Number of distinct products associated with ANY option in the ReferenceOEM group: " . $countProductsWithRefOemProp);
        if ($countProductsWithRefOemProp == 0) {
            CliLogger::warning("No products are assigned properties from the ReferenceOEM group. This will result in no links.");
        }


        $sqlProductPropertyCheckMpn = "
            SELECT COUNT(DISTINCT pp.product_id)
            FROM product_property pp
            JOIN property_group_option pgo ON pgo.id = pp.property_group_option_id
            WHERE pgo.property_group_id = UNHEX(:propertyGroupIdMpn)
        ";
        $countProductsWithMpnProp = $this->connection->fetchOne($sqlProductPropertyCheckMpn, ['propertyGroupIdMpn' => $bindParams['propertyGroupIdMpn']]);
        CliLogger::info("Number of distinct products associated with ANY option in the MPN group: " . $countProductsWithMpnProp);
        if ($countProductsWithMpnProp == 0) {
            CliLogger::warning("No products are assigned properties from the MPN group. This will result in no links.");
        }

        CliLogger::info("--- End Deeper Debugging ---");
    }

    /**
     * Inserts or updates the OEM-to-REM product relationships in the database.
     * Handles the persistence of linked products in the topdata_lor_oem_to_rem table,
     * including cleanup of stale entries from previous runs.
     *
     * @param array $rows Array of product relationship data to be persisted
     * @return void
     */
    private function _upsert(array $rows): void
    {
        if (empty($rows)) {
            CliLogger::info("_upsert called with empty rows. No data to insert/update.");
            // Clean up stale entries even if new data is empty
            $refreshRunId = date('YmdHis');
            $stmtDelete = $this->connection->prepare("DELETE FROM topdata_lor_oem_to_rem WHERE refresh_run_id != :refreshRunId OR refresh_run_id IS NULL");
            try {
                $numRowsDeleted = $stmtDelete->executeStatement(['refreshRunId' => $refreshRunId]);
                CliLogger::info("deleted $numRowsDeleted stale entries (called from empty upsert)");
            } catch (\Throwable $e) {
                CliLogger::error("Error deleting stale entries: " . $e->getMessage());
            }
            return;
        }

        $SQL = "INSERT INTO topdata_lor_oem_to_rem (refresh_run_id, reference_oem, oem_product_number, rem_product_number, oem_product_id, rem_product_id, created_at) VALUES
               (:refreshRunId, :referenceOem, :oemProductNumber, :remProductNumber, UNHEX(:oemProductId), UNHEX(:remProductId), NOW())
               ON DUPLICATE KEY UPDATE updated_at = NOW(), reference_oem = VALUES(reference_oem), oem_product_number = VALUES(oem_product_number), rem_product_number = VALUES(rem_product_number), refresh_run_id = VALUES(refresh_run_id)
               "; // Used VALUES() for ON DUPLICATE KEY UPDATE for robustness


        // Begin the transaction for bulk insertion
        $this->connection->beginTransaction();

        // Prepare the SQL statement
        $refreshRunId = date('YmdHis'); // we use for deleting stale entries

        $stmtUpsert = $this->connection->prepare($SQL);
        // Delete entries that are not part of the current refresh run.
        // It's safer to delete entries whose refresh_run_id IS NULL or not equal to the current one.
        $stmtDelete = $this->connection->prepare("DELETE FROM topdata_lor_oem_to_rem WHERE refresh_run_id != :refreshRunId OR refresh_run_id IS NULL");


        try {
            $insertedCount = 0;
            // Execute the prepared statement for each set of data
            foreach ($rows as $data) {
                $stmtUpsert->executeStatement(array_merge(
                    $data,
                    ['refreshRunId' => $refreshRunId],
                ));
                $insertedCount++;
            }
            CliLogger::info("Upserted $insertedCount rows into topdata_lor_oem_to_rem.");


            $numRowsDeleted = $stmtDelete->executeStatement(['refreshRunId' => $refreshRunId]);
            CliLogger::info("deleted $numRowsDeleted stale entries from topdata_lor_oem_to_rem.");

            // Commit the transaction
            $this->connection->commit();
        } catch (\Throwable $e) {
            // Rollback the transaction if an error occurs
            $this->connection->rollBack();
            CliLogger::error("Error during _upsert: " . $e->getMessage());
            CliLogger::error("SQL: " . $SQL);
            CliLogger::dump("Sample Data (first row if available):", $rows[0] ?? null);
            throw $e; // Re-throw the exception after logging
        }
    }

    /**
     * Refreshes all links between OEM and REM products in the database.
     * This is the main entry point for updating the product relationship table.
     * The process involves finding all linked products and then upserting them
     * into the topdata_lor_oem_to_rem table.
     *
     * @return void
     * @throws \Throwable If any database operation fails
     */
    public function refreshLinks()
    {
        // ---- step 1 - find rem products which have a reference oem
        $rows = $this->_findLinkedProducts();
        CliLogger::info("refreshLinks: _findLinkedProducts returned " . count($rows) . " linked products.");

        // ---- step 2 - populate tbl topdata_lor_oem_to_rem
        // _upsert handles empty $rows internally now
        $this->_upsert($rows);
        CliLogger::info("refreshLinks completed.");
    }
}
