import Plugin from 'src/plugin-system/plugin.class';
import HttpClient from 'src/service/http-client.service'; // For fetching alternative data

export default class BuyButtonClickLoggerPlugin extends Plugin {
    static options = {
        // Selector for the element where OemAlternativePopupPlugin is initialized
        popupPluginHolderSelector: '[data-oem-alternative-popup]',
    };

    init() {
        this._registerEvents();
        // console.log('BuyButtonClickLoggerPlugin initialized on:', this.el);
    }

    _registerEvents() {
        this.el.addEventListener('click', this._handleBuyButtonClick.bind(this));
    }

    _handleBuyButtonClick(event) {
        const buyButton = event.target.closest('.btn-buy');

        if (buyButton && this.el.contains(buyButton)) {
            const form = buyButton.closest('form');
            if (form) {
                const remAlternativesInput = form.querySelector('input[name="topdata-rem-alternatives"]');

                if (remAlternativesInput) {
                    event.preventDefault();
                    event.stopPropagation();
                    console.log('Form contains "topdata-rem-alternatives" input. Preventing default and fetching data.');

                    const oemProductId = this._getProductId(buyButton, form);
                    if (!oemProductId) {
                        console.error('Could not determine OEM Product ID. Cannot show popup.');
                        this._fallbackSubmit(form); // Allow original action if ID is missing
                        return;
                    }

                    // In a real scenario, the value of 'remAlternativesInput' might be a JSON string
                    // of pre-fetched alternative IDs, or just a flag.
                    // For now, we assume we need to fetch details for the OEM and its best REM.
                    this._fetchAlternativeDataAndShowPopup(oemProductId, form);

                } else {
                    // console.log('Form does NOT contain "topdata-rem-alternatives" input. Allowing default action.');
                }
            } else {
                // console.log('.btn-buy button is not inside a form. Allowing default action.');
            }
        }
    }

    _getProductId(buyButton, form) { // Now accepts form as an argument
        if (buyButton.dataset.productId) {
            return buyButton.dataset.productId;
        }
        if (form) { // Use the passed form
            const productIdInput = form.querySelector('input[name="productId"]');
            if (productIdInput && productIdInput.value) {
                return productIdInput.value;
            }
            const lineItemIdInput = form.querySelector('input[type="hidden"][name$="[id]"]');
            if (lineItemIdInput && lineItemIdInput.value) {
                return lineItemIdInput.value;
            }
        }
        return null;
    }

    _fetchAlternativeDataAndShowPopup(oemProductId, originalForm) {
        const client = new HttpClient();
        // You need a new controller endpoint to fetch this data
        // This endpoint should return JSON for oemProductData, bestRemProductData, savingsInfo, and remId
        const route = `/topdata-linked-oem-rem/rem-alternatives-to-oem-product/${oemProductId}`; // Example route

        console.log(`Fetching alternative data for OEM ID: ${oemProductId} from ${route}`);

        client.get(route, (responseText) => {
            try {
                const data = JSON.parse(responseText);
                if (data.success && data.showPopup) {
                    const popupPluginEl = document.querySelector(this.options.popupPluginHolderSelector);
                    if (popupPluginEl) {
                        const popupInstance = window.PluginManager.getPluginInstanceFromElement(popupPluginEl, 'OemAlternativePopupPlugin');
                        if (popupInstance) {
                            popupInstance.show(
                                data.oemProductData,
                                data.remProductData,
                                data.savingsInfo,
                                oemProductId, // Original OEM ID
                                data.remProductData.id // ID of the best REM alternative
                            );
                        } else {
                            console.error('OemAlternativePopupPlugin instance not found.');
                            // this._fallbackSubmit(originalForm);
                        }
                    } else {
                        console.error(`Element ${this.options.popupPluginHolderSelector} not found.`);
                    //    this._fallbackSubmit(originalForm);
                    }
                } else {
                    console.log('No popup required or data fetch failed, proceeding with original add to cart.', data.message || '');
                    this._fallbackSubmit(originalForm);
                }
            } catch (e) {
                console.error('Error parsing alternative data response:', e, responseText);
                this._fallbackSubmit(originalForm);
            }
        }, (errorResponse) => {
            console.error('AJAX error fetching alternative data:', errorResponse);
            this._fallbackSubmit(originalForm);
        });
    }

    _fallbackSubmit(formElement) {
        // Re-enable default submission if we prevented it and then failed
        // This is tricky because we've already called preventDefault.
        // The most reliable way to "re-submit" an add-to-cart form is usually
        // to trigger Shopware's own add-to-cart JavaScript, or if it's a simple form,
        // by directly calling form.submit() if it's not an AJAX form.

        if (formElement && typeof formElement.submit === 'function') {
            // This might bypass Shopware's AJAX handling if it's an AJAX form.
            // For true AJAX forms, you'd need to find and call the Shopware JS handler.
            console.log('Executing fallback: submitting form programmatically.');
            formElement.submit();
        } else {
            console.warn('Fallback: Could not submit form programmatically. User may need to click again.');
            // If it's a purely JS-driven AJAX add-to-cart (no actual form.submit()),
            // this fallback is insufficient. You'd need to call the original JS handler.
        }
    }
}