<?php declare(strict_types=1);

namespace Topdata\TopdataLinkedOemRemSW6\Command;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Exception;
use Shopware\Core\Content\Product\SalesChannel\SalesChannelProductEntity;
use Shopware\Core\Framework\Api\Context\SystemSource;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\System\SalesChannel\Context\SalesChannelContextService;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Topdata\TopdataFoundationSW6\Command\AbstractTopdataCommand;
use Topdata\TopdataLinkedOemRemSW6\Service\OemRemProductFinderService;
use Topdata\TopdataLinkedOemRemSW6\Service\OemRemSavingsService;
use Topdata\TopdataLinkedOemRemSW6\DTO\SavingsOverOemDTO;
use Topdata\TopdataLinkedOemRemSW6\Service\SalesChannelSelectionService;
use TopdataSoftwareGmbH\Util\UtilDebug;

/**
 * Command to list all linked OEM and REM products.
 */
#[AsCommand(
    name: 'topdata:linked-oem-rem:list-links',
    description: 'Lists all linked OEM and REM products from the topdata_lor_oem_to_rem table with pagination.'
)]
class Command_ListLinks extends AbstractTopdataCommand
{
    private const DEFAULT_LIMIT = 10;

    public function __construct(
        private readonly Connection                   $connection,
        private readonly OemRemProductFinderService    $productFinderService,
        private readonly OemRemSavingsService          $savingsService,
        private readonly SalesChannelContextService    $salesChannelContextService,
        private readonly SalesChannelSelectionService  $salesChannelSelectionService,
    )
    {
        parent::__construct();
        $this->addOption(
            'page',
            'p',
            InputOption::VALUE_REQUIRED,
            'The page number to display.',
            '1' // Default value
        );
    }

    /**
     * Executes the command.
     *
     * @param InputInterface $input
     * @param OutputInterface $output
     * @return int
     * @throws Exception
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $this->cliStyle->title('Linked OEM and REM Products');

        $salesChannelContext = $this->salesChannelSelectionService->askToGetSalesChannelContext();


        $page = (int)$input->getOption('page');
        if ($page < 1) {
            $this->cliStyle->error('Page number must be a positive integer.');
            return Command::FAILURE;
        }

        $limit = self::DEFAULT_LIMIT;
        $offset = ($page - 1) * $limit;

        $totalLinks = $this->_fetchTotalLinksCount();
        if ($totalLinks === 0) {
            $this->cliStyle->info('No linked products found in the topdata_lor_oem_to_rem table.');
            $this->done();
            return Command::SUCCESS;
        }

        $totalPages = (int)ceil($totalLinks / $limit);
        if ($page > $totalPages && $totalLinks > 0) {
            $this->cliStyle->warning(sprintf('Page %d is out of range. Total pages: %d.', $page, $totalPages));
            $this->done();
            return Command::SUCCESS;
        }

        $links = $this->_fetchLinks($limit, $offset);

        if (empty($links)) {
            $this->cliStyle->info(sprintf('No linked products found on page %d.', $page));
            $this->done();
            return Command::SUCCESS;
        }

        $globalStartIndex = $offset + 1;
        $tableData = [];
        foreach ($links as $index => $link) {

            // Get the REM product to fetch its price
            $remProducts = $this->productFinderService->findMatchingRemProductsFlat([$link['oem_product_id']], $salesChannelContext);
            $remProduct = $remProducts[0] ?? null;

            $savingsInfo = 'N/A';
            $oemPrice = 'N/A';
            $remPrice = 'N/A';
            $savingsPercent = 'N/A';

            if ($remProduct) {
                $savingsDto = $this->savingsService->getSavingsOverOem($remProduct, $salesChannelContext);

                if ($savingsDto) {
                    $oemPrice = number_format($savingsDto->getPriceOem(), 2, ',', '.');
                    $remPrice = number_format($savingsDto->getPriceRem(), 2, ',', '.');
                    $savingsAbs = $savingsDto->getPriceSavingAbsolute();
                    $savingsInfo = number_format($savingsAbs, 2, ',', '.');
                    $savingsPercent = number_format($savingsDto->getPriceSavingPercentAbsolute(), 2, ',', '.') . '%';
                }
            }

            $tableData[] = [
                $globalStartIndex + $index, // Global row number
                $link['rem_product_number'],
                $link['oem_product_number'],
                $oemPrice,
                $remPrice,
                $savingsInfo,
                $savingsPercent,
            ];
        }

        $this->cliStyle->text(sprintf('Displaying page %d of %d (Total links: %d, %d per page)', $page, $totalPages, $totalLinks, $limit));
        $this->cliStyle->table(
            ['#',
                'productNr REM',
                'productNr OEM',
                'OEM Price',
                'REM Price',
                'Savings (abs)',
                'Savings (%)'],
            $tableData
        );

        $this->done();
        return Command::SUCCESS;
    }

    /**
     * Fetches the linked product numbers from the database with pagination.
     *
     * @param int $limit
     * @param int $offset
     * @return array<int, array<string, string>>
     * @throws Exception
     */
    private function _fetchLinks(int $limit, int $offset): array
    {
        $sql = "SELECT 
                    rem_product_number, 
                    oem_product_number,
                    HEX(rem_product_id) as rem_product_id,
                    HEX(oem_product_id) AS oem_product_id
                FROM topdata_lor_oem_to_rem 
                ORDER BY rem_product_number ASC, oem_product_number ASC
                LIMIT :limit OFFSET :offset";

        return $this->connection->fetchAllAssociative($sql, [
            'limit'  => $limit,
            'offset' => $offset,
        ], [
            'limit'  => \PDO::PARAM_INT,
            'offset' => \PDO::PARAM_INT,
        ]);
    }

    /**
     * Fetches the total count of linked products.
     *
     * @return int
     * @throws Exception
     */
    private function _fetchTotalLinksCount(): int
    {
        $sql = "SELECT COUNT(*) FROM topdata_lor_oem_to_rem";
        return (int)$this->connection->fetchOne($sql);
    }
}