<?php

namespace Topdata\TopdataLinkedOemRemSW6\DTO;


use Shopware\Core\Content\Product\SalesChannel\SalesChannelProductEntity;
use Shopware\Core\Content\Property\Aggregate\PropertyGroupOption\PropertyGroupOptionEntity;
use Topdata\TopdataLinkedOemRemSW6\Traits\CreateFromDictTrait;

/**
 * 11/2023 created
 * 04/2024 TODO: compare per prices relative to PaperCapacity
 *
 * @method static SavingsOverOemDTO createFromDict(array $dict)
 */
class SavingsOverOemDTO
{
    use CreateFromDictTrait;

    private SalesChannelProductEntity $oemProduct;
    private float $priceOem;
    private float $priceRem;
    private bool $haveSameCapacity; # if both have same capacity, we can compare product prices, if not we compare prices relative to capacities
    private float $priceSavingAbsolute;
    private float $priceSavingPercentAbsolute;

    /**
     * factory method
     * 11/2023 created
     *
     * @param SalesChannelProductEntity $oemProduct
     * @param SalesChannelProductEntity $remProduct
     */
    public static function createFromOemProductAndRemProduct(
        SalesChannelProductEntity $oemProduct,
        SalesChannelProductEntity $remProduct,
        ?string                   $sw6IdCapacity
    ): SavingsOverOemDTO
    {
        /** @var PropertyGroupOptionEntity $propCatOem */
        $propCatOem = $oemProduct->getProperties()->filterByGroupId($sw6IdCapacity)->first();
        /** @var PropertyGroupOptionEntity $propCapRem */
        $propCapRem = $remProduct->getProperties()->filterByGroupId($sw6IdCapacity)->first();
        if ($propCatOem && $propCapRem) {
            $capOem = self::extractIntegerFromString($propCatOem->getName());
            $capRem = self::extractIntegerFromString($propCapRem->getName());
            $haveSameCapacity = $capOem === $capRem;
        } else {
            $haveSameCapacity = false;
        }

        // ---- compare absolute prices
        $priceOem = $oemProduct->getCalculatedPrice()->getTotalPrice();
        $priceRem = $remProduct->getCalculatedPrice()->getTotalPrice();
        $priceSavingAbsolute = $priceOem - $priceRem;
        $priceSavingPercentAbsolute = $priceSavingAbsolute / $priceOem * 100;

        return self::createFromDict([
            'oemProduct'                 => $oemProduct,
            'priceOem'                   => $priceOem,
            'priceRem'                   => $priceRem,
            'priceSavingAbsolute'        => $priceSavingAbsolute,
            'priceSavingPercentAbsolute' => $priceSavingPercentAbsolute,
            'haveSameCapacity'           => $haveSameCapacity,
        ]);
    }

    /**
     * 04/2024 created
     * @param string|null $value eg "1000 Blatt", "1000", "5.400 Seiten", ...
     */
    private static function extractIntegerFromString(?string $value): ?int
    {
        if ($value === null) {
            return null;
        }

        $value = preg_replace('/[^0-9]/', '', $value);
        if ($value === '') {
            return null;
        }

        return (int)$value;
    }

    public function getOemProduct(): SalesChannelProductEntity
    {
        return $this->oemProduct;
    }

    public function getPriceOem(): float
    {
        return $this->priceOem;
    }

    public function getPriceRem(): float
    {
        return $this->priceRem;
    }

    public function getPriceSavingAbsolute(): float
    {
        return $this->priceSavingAbsolute;
    }

    public function getPriceSavingPercentAbsolute(): float
    {
        return $this->priceSavingPercentAbsolute;
    }


}