<?php

namespace Topdata\TopdataLinkedOemRemSW6\Service;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Exception;
use Shopware\Core\Content\Product\SalesChannel\SalesChannelProductEntity;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\EntitySearchResult;
use Shopware\Core\System\SalesChannel\Entity\SalesChannelRepository;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Core\System\SystemConfig\SystemConfigService;
use Topdata\TopdataFoundationSW6\Trait\CliStyleTrait;
use Topdata\TopdataLinkedOemRemSW6\Constants\GlobalPluginConstants;
use Topdata\TopdataLinkedOemRemSW6\DTO\SavingsOverOemDTO;
use Topdata\TopdataLinkedOemRemSW6\Traits\SetIoTrait;

/**
 * - methods starting with find* DO query the DB
 * - methods starting with get* do NOT query the DB
 *
 * 11/2023 created
 */
class OemRemService
{
    use SetIoTrait;
    use CliStyleTrait;

    public function __construct(
        private readonly SalesChannelRepository $productRepository,
        private readonly Connection             $connection,
        private readonly SystemConfigService    $systemConfigService
    )
    {
    }

    private function _findLinkedProducts(): array
    {
        // ---- step 1 find rem products which have a reference oem
        $SQL = "
                        
                SELECT 
                        LOWER(HEX(pp1.product_id)) AS remProductId, 
                        -- p1.product_number          AS remProductNumber, 
                        LOWER(HEX(pp2.product_id)) AS oemProductId, 
                        -- p2.product_number          AS oemProductNumber, 
                        pgot1.name                 AS referenceOem
                
                         FROM product_property                  pp1
                         JOIN property_group_option             pgo1  ON pgo1.id = pp1.property_group_option_id AND pgo1.property_group_id = UNHEX(:propertyGroupIdReferenceOem)
                         JOIN property_group_option_translation pgot1 ON pgot1.property_group_option_id=pgo1.id AND pgot1.language_id=UNHEX(:languageIdGerman)
                         -- too slow: JOIN product                           p1    ON p1.id = pp1.product_id AND p1.active = 1
                    
                         JOIN property_group_option             pgo2  ON pgo2.property_group_id = UNHEX(:propertyGroupIdMpn)  
                         JOIN property_group_option_translation pgot2 ON pgot2.property_group_option_id=pgo2.id AND pgot2.language_id=UNHEX(:languageIdGerman) AND pgot2.name = pgot1.name
                         JOIN product_property                  pp2   ON pp2.property_group_option_id = pgo2.id
                         -- too slow: JOIN product                           p2    ON p2.id = pp2.product_id AND p2.active = 1
                
                
                ORDER BY referenceOem ASC
        
        ";
        // TODO: these property group ids need to be fetched dynamically
        $rows = $this->connection->fetchAllAssociative($SQL, [
            'propertyGroupIdMpn'          => $this->systemConfigService->get('TopdataLinkedOemRemSW6.config.propertyGroupIdMpn'), // 'b21b95c56b714adeaae7fdeb0e08b56b'
            'propertyGroupIdReferenceOem' => $this->systemConfigService->get('TopdataLinkedOemRemSW6.config.propertyGroupIdReferenceOem'),// '75ee2810a8b846069be33a5bd2c639f7',
            'languageIdGerman'            => GlobalPluginConstants::LANGUAGE_ID_GERMAN,
        ]);
        $this->cliStyle->info("found " . count($rows) . " rows");
        // dump(count($rows), $rows[0]);

        // ---- step 2 - fetch products from db and make loopup map
        $productIdsRem = array_column($rows, 'remProductId');
        $productIdsOem = array_column($rows, 'oemProductId');
        $productIds = array_unique(array_merge($productIdsRem, $productIdsOem));
        $rows2 = $this->connection->fetchAllAssociative("SELECT LOWER(HEX(id)) AS id, product_number, active FROM product WHERE id IN (:ids)",
            ['ids' => array_map('hex2bin', $productIds)],
            ['ids' => Connection::PARAM_STR_ARRAY]
        );
        $ids = array_column($rows2, 'id');
        $map = array_combine($ids, $rows2);
        $this->cliStyle->info("found " . count($map) . " products in db for " . count($ids) . " ids");


        // ---- step 3 - combine data from step1 and step2
        $rows3 = [];
        foreach ($rows as $idx => $row) {
            $remActive = (int)$map[$row['remProductId']]['active'] ?? 0;
            $oemActive = (int)$map[$row['oemProductId']]['active'] ?? 0;
            if (!$remActive || !$oemActive) {
                // $this->cliStyle->warning("rem or oem product is not active. Skipping row: " . json_encode($row));
                continue;
            }

            $rows3[] = [
                'referenceOem'     => $row['referenceOem'],
                'remProductNumber' => $map[$row['remProductId']]['product_number'] ?? '',
                'oemProductNumber' => $map[$row['oemProductId']]['product_number'] ?? '',
                'remProductId'     => $row['remProductId'],
                'oemProductId'     => $row['oemProductId'],
            ];
        }

        return $rows3;
    }

    /**
     * fills topdata_lor_oem_to_rem
     *
     * 11/2023 created
     *
     * @param array $rows
     * @return void
     */
    private function _upsert(array $rows): void
    {
        $SQL = "INSERT INTO topdata_lor_oem_to_rem (refresh_run_id, reference_oem, oem_product_number, rem_product_number, oem_product_id, rem_product_id, created_at) VALUES
               (:refreshRunId, :referenceOem, :oemProductNumber, :remProductNumber, UNHEX(:oemProductId), UNHEX(:remProductId), NOW())
               ON DUPLICATE KEY UPDATE updated_at = NOW(), reference_oem = :referenceOem, oem_product_number = :oemProductNumber, rem_product_number = :remProductNumber, refresh_run_id = :refreshRunId
               ";


        // Begin the transaction for bulk insertion
        $this->connection->beginTransaction();

        // Prepare the SQL statement
        $refreshRunId = date('YmdHis'); // we use for deleting stale entries

        $stmtUpsert = $this->connection->prepare($SQL);
        $stmtDelete = $this->connection->prepare("DELETE FROM topdata_lor_oem_to_rem WHERE refresh_run_id != :refreshRunId");


        try {

            // Execute the prepared statement for each set of data
            foreach ($rows as $data) {
                $stmtUpsert->executeStatement(array_merge(
                    $data,
                    ['refreshRunId' => $refreshRunId],
                ));
            }


            $numRowsDeleted = $stmtDelete->executeStatement(['refreshRunId' => $refreshRunId]);
            $this->cliStyle->info("deleted $numRowsDeleted stale entries");

            // Commit the transaction
            $this->connection->commit();
        } catch (\Throwable $e) {
            // Rollback the transaction if an error occurs
            $this->connection->rollBack();
            throw $e; // Handle the error as needed
        }
    }


    /**
     * refresh links between oem and rem products in tbl topdata_lor_oem_to_rem
     *
     * 11/2023 created
     *
     * @return void
     * @throws \Throwable
     */
    public function refreshLinks()
    {
        // ---- step 1 - find rem products which have a reference oem
        $rows = $this->_findLinkedProducts();
        dump($rows[0], count($rows));

        // ---- step 2 - populate tbl topdata_lor_oem_to_rem
        $this->_upsert($rows);
    }


    /**
     * example output:
     *
     * [ '0012345678' => [ '12345678', '12345679' ] ]
     *
     * 11/2023 created
     *
     * @param array $oemProductIds
     * @return array a map of oem product ids to rem product ids
     * @throws \Doctrine\DBAL\Exception
     */
    private function _buildMapOemProductIdToRemProductIds(array $oemProductIds): array
    {
        if (empty($oemProductIds)) {
            return [];
        }

        // ---- find matching REM product ids
        $SQL = "    
                    SELECT 
                        LOWER(HEX(rem_product_id)) AS remProductId, 
                        LOWER(HEX(oem_product_id)) AS oemProductId 
                    FROM topdata_lor_oem_to_rem 
                    WHERE oem_product_id IN (:oemProductIds)";
        $rows = $this->connection->fetchAllAssociative($SQL, ['oemProductIds' => array_map('hex2bin', $oemProductIds)], ['oemProductIds' => Connection::PARAM_STR_ARRAY]);

        if (empty($rows)) {
            return [];
        }

        // ---- build Lookup map
        $mapOemProductIdToRemProductIds = [];
        foreach ($rows as $row) {
            if (!isset($mapOemProductIdToRemProductIds[$row['oemProductId']])) {
                $mapOemProductIdToRemProductIds[$row['oemProductId']] = [];
            }
            $mapOemProductIdToRemProductIds[$row['oemProductId']][] = $row['remProductId'];
        }

        // $remProductIds = array_column($rows, 'remProductId');

        return $mapOemProductIdToRemProductIds;
    }


    /**
     * example output:
     *
     * [ '0012345678' => [ '12345678', '12345679' ] ]
     *
     * 11/2023 created
     *
     * @param array $remProductIds
     * @return array a map of rem product ids to oem product ids
     * @throws \Doctrine\DBAL\Exception
     */
    private function _buildMapRemProductIdToOemProductIds(array $remProductIds): array
    {
        if (empty($remProductIds)) {
            return [];
        }

        // ---- find matching OEM product ids
        $SQL = "    
                    SELECT 
                        LOWER(HEX(oem_product_id)) AS oemProductId, 
                        LOWER(HEX(rem_product_id)) AS remProductId 
                    FROM topdata_lor_oem_to_rem 
                    WHERE rem_product_id IN (:remProductIds)";
        $rows = $this->connection->fetchAllAssociative($SQL, ['remProductIds' => array_map('hex2bin', $remProductIds)], ['remProductIds' => Connection::PARAM_STR_ARRAY]);
        if (empty($rows)) {
            return [];
        }

        // ---- build Lookup map
        $mapRemProductIdToOemProductIds = [];
        foreach ($rows as $row) {
            if (!isset($mapRemProductIdToOemProductIds[$row['remProductId']])) {
                $mapRemProductIdToOemProductIds[$row['remProductId']] = [];
            }
            $mapRemProductIdToOemProductIds[$row['remProductId']][] = $row['oemProductId'];
        }

        // $oemProductIds = array_column($rows, 'oemProductId');

        return $mapRemProductIdToOemProductIds;
    }


    /**
     * DOES query the DB
     *
     * wenn OEM Produkte im Warenkorb sind, REM Alternativprodukt/e anzeigen lassen
     *
     * 11/2023 created
     *
     * @param string[] $oemProductIds
     * @param SalesChannelContext $salesChannelContext
     * @return array a map
     * @throws Exception
     */
    public function findMatchingRemProductsMap(array $oemProductIds, SalesChannelContext $salesChannelContext): array
    {
        $mapOemProductIdToRemProductIds = $this->_buildMapOemProductIdToRemProductIds($oemProductIds);

        // ---- fetch the products
        $remProductIds = array_merge(...array_values($mapOemProductIdToRemProductIds));
        if (empty($remProductIds)) {
            return [];
        }
        $criteria = new Criteria($remProductIds);
        $associatedProducts = $this->productRepository->search($criteria, $salesChannelContext)->getEntities();


        // ---- build the map
        $mapFinal = [];
        foreach ($mapOemProductIdToRemProductIds as $oemProductId => $remProductIds) {
            $mapFinal[$oemProductId] = array_values($associatedProducts->filter(fn($product) => in_array($product->getId(), $remProductIds))->getElements());
        }

        return $mapFinal;
    }

    /**
     * DOES query the DB
     *
     * 11/2023 created
     *
     * @param string[] $oemProductIds
     * @param SalesChannelContext $ctx
     * @return SalesChannelProductEntity[]
     */
    public function findMatchingRemProductsFlat(array $oemProductIds, SalesChannelContext $salesChannelContext): array
    {
        return array_merge(...array_values($this->findMatchingRemProductsMap($oemProductIds, $salesChannelContext)));
    }


    /**
     * DOES query the DB
     *
     * wenn REM Produkte im Warenkorb sind, OEM Alternativprodukt/e anzeigen lassen
     *
     * 11/2023 created
     *
     * @param string[] $remProductIds
     * @param SalesChannelContext $salesChannelContext
     * @return array a map
     * @throws Exception
     */
    public function findMatchingOemProductsMap(array $remProductIds, SalesChannelContext $salesChannelContext): array
    {
        $mapRemProductIdToOemProductIds = $this->_buildMapRemProductIdToOemProductIds($remProductIds);
        // ---- fetch the products
        $oemProductIds = array_merge(...array_values($mapRemProductIdToOemProductIds));
        if (empty($oemProductIds)) {
            return [];
        }
        $criteria = new Criteria($oemProductIds);
        $associatedProducts = $this->productRepository->search($criteria, $salesChannelContext)->getEntities();


        // ---- build the map
        $mapFinal = [];
        foreach ($mapRemProductIdToOemProductIds as $remProductId => $oemProductIds) {
            $mapFinal[$remProductId] = array_values($associatedProducts->filter(fn($product) => in_array($product->getId(), $oemProductIds))->getElements());
        }

        return $mapFinal;
    }


    /**
     * 11/2023 created
     *
     * @param string[] $array
     * @param SalesChannelContext $salesChannelContext
     * @return array
     */
    public function findMatchingOemProductsFlat(array $remProductIds, SalesChannelContext $salesChannelContext)
    {
        return array_merge(...array_values($this->findMatchingOemProductsMap($remProductIds, $salesChannelContext)));
    }

    /**
     * 11/2023 created
     *
     * @param SalesChannelProductEntity $remProduct
     * @param SalesChannelContext $salesChannelContext
     * @return SavingsOverOemDTO|null
     */
    public function getSavingsOverOem(SalesChannelProductEntity $remProduct, SalesChannelContext $salesChannelContext): ?SavingsOverOemDTO
    {
        $oemProducts = $this->findMatchingOemProductsFlat([$remProduct->getId()], $salesChannelContext);
        if (empty($oemProducts)) {
            return null;
        }

        // ---- calculate price savings
        $ret = [];
        $sw6IdCapacity = $this->systemConfigService->get('TopdataLinkedOemRemSW6.config.propertyGroupIdCapacity');
        foreach ($oemProducts as $oemProduct) {
            $ret[] = SavingsOverOemDTO::createFromOemProductAndRemProduct($oemProduct, $remProduct, $sw6IdCapacity);
        }

        // ---- if more than one OEM products are found, return the one with the highest saving
        usort($ret, fn(SavingsOverOemDTO $item1, SavingsOverOemDTO $item2) => $item2->getPriceSavingAbsolute() <=> $item1->getPriceSavingAbsolute());

        return $ret[0];
    }

    /**
     * does NOT query the DB
     *
     * 11/2023 created
     *
     * @param EntitySearchResult $searchResults
     * @param SalesChannelContext $salesChannelContext
     * @return array
     * @throws Exception
     */
    public function getSavingsOverOemMap(EntitySearchResult $searchResults, SalesChannelContext $salesChannelContext): array
    {
        $map = $this->findMatchingOemProductsMap($searchResults->getIds(), $salesChannelContext);

        $savingsMap = [];
        $sw6IdCapacity = $this->systemConfigService->get('TopdataLinkedOemRemSW6.config.propertyGroupIdCapacity');
        foreach ($map as $remProductId => $oemProducts) {
            $savings = [];
            $remProduct = $searchResults->get($remProductId);
            foreach ($oemProducts as $oemProduct) {
                $savings[] = SavingsOverOemDTO::createFromOemProductAndRemProduct($oemProduct, $remProduct, $sw6IdCapacity);
            }

            // ---- if more than one OEM products are found, return the one with the highest saving
            usort($savings, fn(SavingsOverOemDTO $item1, SavingsOverOemDTO $item2) => $item2->getPriceSavingAbsolute() <=> $item1->getPriceSavingAbsolute());
            $savingsMap[$remProductId] = $savings[0];
        }

        return $savingsMap;
    }


}