<?php declare(strict_types=1);


namespace Topdata\TopdataLinkedOemRemSW6\Command;

use Shopware\Core\Framework\Api\Context\SystemSource;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\System\SalesChannel\Context\CachedSalesChannelContextFactory;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Topdata\TopdataFoundationSW6\Command\AbstractTopdataCommand;
use Topdata\TopdataLinkedOemRemSW6\Service\OemRemService;

/**
 * 11/2023 created
 */
#[AsCommand(
    name: 'topdata:linked-oem-rem:test-fetch-linked-products',
    description: 'test fetch linked products'
)]
class TestFetchLinkedProductsCommand extends AbstractTopdataCommand
{
    public function __construct(
        private readonly OemRemService                    $oemRemService,
        private readonly CachedSalesChannelContextFactory $salesChannelContextFactory,
        private readonly EntityRepository                 $salesChannelRepository
    )
    {
        parent::__construct();
    }

    private function getSalesChannelContext(string $salesChannelId, Context $context): SalesChannelContext
    {
        return $this->salesChannelContextFactory->create(
            '',  // token can be empty for admin context
            $salesChannelId,
            [],  // no specific options needed
            $context
        );
    }

    /**
     * ==== MAIN ====
     */
    public function execute(InputInterface $input, OutputInterface $output): int
    {
        $this->oemRemService->setIo($this->cliStyle);
        $oemProductIds = ['00427c2d5bcd445f9fe33afccea249b7', '0054dab2b71f49d7bc2312b8da566d02'];

        $salesChannelId = $this->askForSalesChannelId();


        // Create sales channel context
        $context = new Context(new SystemSource());
        $salesChannelContext = $this->getSalesChannelContext($salesChannelId, $context);

        $map = $this->oemRemService->findMatchingRemProductsMap($oemProductIds, $salesChannelContext);

        $this->cliStyle->dumpDict($map);

        $this->done();

        return Command::SUCCESS;
    }

    public function askForSalesChannelId(): string
    {
        // Get available sales channels
        $criteria = new Criteria();
        $criteria->addAssociation('domains');


        $salesChannels = $this->salesChannelRepository->search($criteria, Context::createDefaultContext());

        $choices = [];
        foreach ($salesChannels as $salesChannel) {
            $domain = $salesChannel->getDomains()->first();
            $url = $domain ? $domain->getUrl() : 'no domain';
            $choices[$salesChannel->getId()] = sprintf('%s (%s)', $salesChannel->getName(), $url);
        }

        if (empty($choices)) {
            throw new \RuntimeException('No sales channels found');
        }

        $defaultSalesChannelId = array_keys($choices)[0];
        $salesChannelId = $this->cliStyle->choice('Select sales channel', $choices, $defaultSalesChannelId);

        $this->cliStyle->info('Selected sales channel: ' . $salesChannelId);

        return $salesChannelId;
    }

}
