<?php declare(strict_types=1);

namespace Topdata\FreeTopdataCompareProducts\Subscriber;

use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Topdata\FreeTopdataCompareProducts\Component\Collection;
use Topdata\FreeTopdataCompareProducts\Component\Helper;
use Shopware\Storefront\Page\PageLoadedEvent;
use Shopware\Storefront\Pagelet\Header\HeaderPageletLoadedEvent;
use Topdata\TopdataProductMenu\Storefront\Page\ProductMenu\ProductMenuLoadedEvent;
use Shopware\Core\Framework\Adapter\Translation\Translator;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;

/**
 * Subscriber class for handling events related to product comparison functionality.
 * This class subscribes to storefront events to add product comparison features to the header, product menu, and pages.
 */
class Subscriber implements EventSubscriberInterface
{

    public function __construct(
        private readonly Translator            $translator,
        private readonly UrlGeneratorInterface $router
    )
    {
    }

    /**
     * Returns an array of event names this subscriber wants to listen to.
     *
     * @return array<string, string>
     */
    public static function getSubscribedEvents(): array
    {
        return [
            HeaderPageletLoadedEvent::class => 'addProductComparer',
            ProductMenuLoadedEvent::class   => 'addMenuItems',
            PageLoadedEvent::class          => 'addProductComparerToPage',
        ];
    }

    /**
     * Adds a menu item for comparing products to the product menu.
     *
     * @param ProductMenuLoadedEvent $event
     */
    public function addMenuItems(ProductMenuLoadedEvent $event): void
    {
        $event->getPage()->setMenuItem(
            $this->translator->trans('topdata-compare-products.compare'),
            '#',
            [
                'path' => $this->router->generate(
                    'frontend.topdata-compare-products.api.product-add',
                    ['productId' => $event->getPage()->getProductId()]
                ),
            ],
            ['topdata-compare-products-add']
        );
    }

    /**
     * Adds the product comparer to the header pagelet.
     *
     * @param HeaderPageletLoadedEvent $event
     */
    public function addProductComparer(HeaderPageletLoadedEvent $event): void
    {
        // ---- Create a new collection to store the product comparer data
        $struct = new Collection();
        // ---- Set the count of products in the comparer
        $struct->set('count', Helper::getCompareProductsTotalCount($event->getRequest()));
        // ---- Set the product IDs in the comparer
        $struct->set('productIds', Helper::getProductCompareIds($event->getRequest()));
        // ---- Add the product comparer to the header pagelet as an extension
        $event->getPagelet()->addExtension('productComparer', $struct);
    }

    /**
     * Adds the product comparer to the page.
     *
     * @param PageLoadedEvent $event
     */
    public function addProductComparerToPage(PageLoadedEvent $event): void
    {
        // ---- Create a new collection to store the product comparer data
        $struct = new Collection();
        // ---- Set the count of products in the comparer
        $struct->set('count', Helper::getCompareProductsTotalCount($event->getRequest()));
        // ---- Set the product IDs in the comparer
        $struct->set('productIds', Helper::getProductCompareIds($event->getRequest()));
        // ---- Add the product comparer to the page as an extension
        $event->getPage()->addExtension('productComparer', $struct);
    }
}