<?php declare(strict_types=1);

namespace Topdata\FreeTopdataCompareProducts\Storefront\Page\PopupCompareProductsWidget;

use Shopware\Core\Content\Property\PropertyGroupEntity;
use Shopware\Core\Framework\DataAbstractionLayer\EntityCollection;
use Shopware\Core\System\SalesChannel\Entity\SalesChannelRepository;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Page\GenericPageLoader;
use Symfony\Component\HttpFoundation\Request;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Content\Product\SalesChannel\ProductAvailableFilter;
use Shopware\Core\Content\Product\Aggregate\ProductVisibility\ProductVisibilityDefinition;
use Shopware\Core\Content\Product\SalesChannel\SalesChannelProductEntity;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Sorting\FieldSorting;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Topdata\FreeTopdataCompareProducts\Component\Helper;
use Topdata\FreeTopdataCompareProducts\Component\SettingsService;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\Adapter\Translation\Translator;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Topdata\FreeTopdataCompareProducts\Storefront\Page\PopupCompareProductsWidget\PopupCompareProductsWidget;

/**
 * This class is responsible for loading the data required for the popup compare products widget.
 * It fetches products, property groups, and handles adding/removing products from the compare list.
 */
class PopupCompareProductsWidgetLoader
{
    const COMPARE_PRODUCTS_LIMIT = 4;
    const COMPARE_GROUPS_LIMIT   = 4;

    /* product compare strucrure:
    $array = [
                0 => [
                    'productArt'=>'productArt1',
                    'productIds'=>['id1', 'id2', 'id3', ...],
                    'propertyGroups'=>['prop1', 'prop2', 'prop3', ...],
                    'propertyGroupsDifferent'=>['prop2', 'prop4', ...],
                ],
                1 => [...],
                ...
            ];
    */

    private ?PopupCompareProductsWidget $page;
    private ?string $groupingGroup = null;

    public function __construct(
        private readonly GenericPageLoader        $genericLoader,
        private readonly SalesChannelRepository   $productRepository,
        private readonly EventDispatcherInterface $eventDispatcher,
        private readonly SettingsService          $settings,
        private readonly EntityRepository         $propertyGroupRepository,
        private readonly Translator               $translator,
        private readonly UrlGeneratorInterface    $router,
        private readonly EntityRepository         $customFieldRepository
    )
    {
    }


    /**
     * Retrieves the name of the property group used for grouping products in the compare list.
     *
     * @param SalesChannelContext $salesChannelContext
     * @return string
     */
    public function getGroupingGroupName(SalesChannelContext $salesChannelContext): string
    {
        // ---- Check if the grouping group name is already cached
        if (null !== $this->groupingGroup) {
            return $this->groupingGroup;
        }

        // ---- If no grouping property is set in the settings, return an empty string
        if (!$this->settings->getString('groupingProperty')) {
            $this->groupingGroup = '';
            return $this->groupingGroup;
        }

        $propertyGroupId = $this->settings->getString('groupingProperty');

        /** @var PropertyGroupEntity|null $propertyGroup */
        $propertyGroup = $this
            ->propertyGroupRepository
            ->search(new Criteria([$propertyGroupId]), $salesChannelContext->getContext())
            ->first();

        $this->groupingGroup = (string)$propertyGroup?->getName();

        return $this->groupingGroup;
    }


    /**
     * Changes the order of product compare groups by moving a specific group to the beginning of the list.
     *
     * @param Request $request
     * @param SalesChannelContext $salesChannelContext
     * @param int $group
     * @return bool
     */
    public function changeGroup(Request $request, SalesChannelContext $salesChannelContext, int $group = 0): bool
    {
        $productCompareGroups = Helper::getProductCompare($request);
        // ---- Check if the specified group exists in the product compare groups
        if ($group && isset($productCompareGroups[$group])) {
            $temp = $productCompareGroups[$group];
            unset($productCompareGroups[$group]);
            array_unshift($productCompareGroups, $temp);
            Helper::setProductCompare($productCompareGroups, $request);
            return true;
        }
        return false;
    }


    /**
     * Loads the data required for the popup compare products widget.
     *
     * @param Request $request
     * @param SalesChannelContext $salesChannelContext
     * @return \Topdata\FreeTopdataCompareProducts\Storefront\Page\PopupCompareProductsWidget\PopupCompareProductsWidget
     */
    public function load(Request $request, SalesChannelContext $salesChannelContext): PopupCompareProductsWidget
    {
        $this->setPage($request, $salesChannelContext);

        $productCompareGroups = Helper::getProductCompare($request);

        $this->page->group = (int)$request->get('group');
        // ---- Check if the requested group exists, otherwise default to the first group
        if (!isset($productCompareGroups[$this->page->group])) {
            $this->page->group = 0;
        }

        $this->page->compare = $productCompareGroups;

        // ---- Load product data for the selected group
        if (isset($productCompareGroups[$this->page->group])) {
            $this->page->products = $this->loadProducts($productCompareGroups[$this->page->group]['productIds'], $salesChannelContext);
            $this->page->propertyGroups = $productCompareGroups[$this->page->group]['propertyGroups'];
            $this->page->propertyGroupsDifferent = $productCompareGroups[$this->page->group]['propertyGroupsDifferent'];
            $this->page->productArts = $this->loadProductArts($request);
        }

        $this->eventDispatcher->dispatch(
            new PopupCompareProductsWidgetLoadedEvent($this->page, $salesChannelContext, $request)
        );

        $this->page->temp = $productCompareGroups;
        return $this->page;
    }

    /**
     * Checks if the translations of property group names have changed and rebuilds the compare list if necessary.
     *
     * @param Request $request
     * @param SalesChannelContext $salesChannelContext
     * @return void
     */
    public function checkTranslation(Request $request, SalesChannelContext $salesChannelContext): void
    {
        $productCompareGroups = Helper::getProductCompare($request);
        $mapIdName = [];
        // ---- Iterate through the compare groups to check for translation changes
        foreach ($productCompareGroups as $group) {
            // ---- Check if the "all products" translation has changed
            if ($group['groupKey'] === 'all') {
                if ($group['productArt'] !== $this->translator->trans('topdata-compare-products.allProducts')) {
                    $this->rebuildCompare($request, $salesChannelContext);
                    return;
                }
            }

            // ---- Map property group IDs to their names
            foreach ($group['propertyGroups'] as $key => $value) {
                if (Helper::isValidUuid($key)) {
                    $mapIdName[$key] = $value;
                }
            }
        }

        //find language change in other groups
        if (count($mapIdName)) {
            $propertyGroups = $this
                ->propertyGroupRepository
                ->search(new Criteria(array_keys($mapIdName)), $salesChannelContext->getContext());
            foreach ($propertyGroups as $propertyGroup) {
                // ---- Check if the property group name has changed
                if (isset($mapIdName[$propertyGroup->getId()])
                    && $propertyGroup->getName() !== $mapIdName[$propertyGroup->getId()]
                ) {
                    $this->rebuildCompare($request, $salesChannelContext);
                    return;
                }
            }
        }
    }


    /**
     * Rebuilds the compare list by re-adding all products to ensure correct translations.
     *
     * @param Request $request
     * @param SalesChannelContext $salesChannelContext
     * @return void
     */
    private function rebuildCompare(Request $request, SalesChannelContext $salesChannelContext): void
    {
        $productIds = Helper::getProductCompareIds($request);
        $emptyArray = [];
        Helper::setProductCompare($emptyArray, $request);
        // ---- Re-add each product to the compare list
        foreach ($productIds as $productId) {
            $this->addProductById($productId, $request, $salesChannelContext);
        }
    }


    /**
     * Loads the product arts (group names) for each compare group.
     *
     * @param Request $request
     * @return array
     */
    private function loadProductArts(Request $request): array
    {
        $return = [];
        $productCompareGroups = Helper::getProductCompare($request);
        // ---- Build an array of product arts with their product counts
        foreach ($productCompareGroups as $group) {
            $return[] = $group['productArt'] . ' <span class="badge badge-primary">' . count($group['productIds']) . '</span>';
        }
        return $return;
    }


    /**
     * Loads the property groups for a given array of products.
     *
     * @param array $products
     * @return array
     */
    public function loadProperyGroups(array $products): array
    {
        $return = [];
        if (!count($products)) {
            return [];
        }

        $skipPropertuIds = $this->settings->getValue('skipPropertyIds');

        $extra = [
            PopupCompareProductsWidget::COMPARE_WEIGHT => false,
            PopupCompareProductsWidget::COMPARE_WIDTH  => false,
            PopupCompareProductsWidget::COMPARE_HEIGHT => false,
            PopupCompareProductsWidget::COMPARE_LENGTH => false,
        ];

        $custom = [];

        // ---- Iterate through the products to extract property groups
        foreach ($products as $product) {

            // ---- Check for extra properties (weight, width, height, length) based on settings
            if ($this->settings->getBool('weight') && $product->getWeight()) {
                $extra[PopupCompareProductsWidget::COMPARE_WEIGHT] = true;
            }
            if ($this->settings->getBool('width') && $product->getWidth()) {
                $extra[PopupCompareProductsWidget::COMPARE_WIDTH] = true;
            }
            if ($this->settings->getBool('height') && $product->getHeight()) {
                $extra[PopupCompareProductsWidget::COMPARE_HEIGHT] = true;
            }
            if ($this->settings->getBool('length') && $product->getLength()) {
                $extra[PopupCompareProductsWidget::COMPARE_LENGTH] = true;
            }

            // ---- Check for custom fields based on settings
            if ($this->page->compareCustomFields) {
                foreach ($this->page->compareCustomFields as $customFieldEntity) {
                    foreach ($product->getCustomFields() as $prodCustomName => $prodCustomValue) {
                        if ($customFieldEntity->getName() == $prodCustomName) {
                            $custom[PopupCompareProductsWidget::COMPARE_CUSTOM_PREFIX . $prodCustomName] = true;
                        }
                    }
                }
            }
            // ---- Extract property groups from the product
            foreach ($product->getSortedProperties()->getElements() as $group) {

                // ---- Skip property groups that are in the skip list
                if (is_array($skipPropertuIds)
                    && in_array($group->getId(), $skipPropertuIds)) {
                    continue;
                }
                if (isset($group->getTranslated()['name']) && (string)$group->getTranslated()['name'] != '')
                    $return[$group->getId()] = (string)$group->getTranslated()['name'];
                elseif ($group->getName() && (string)$group->getName() != '')
                    $return[$group->getId()] = (string)$group->getName();  //misa translation
                else
                    $return[$group->getId()] = (string)$group->getId();

            }
        }
        asort($return);

        // ---- Remove extra properties if they are not present in any product
        foreach ($extra as $key => $val) {
            if (!$val) {
                unset($extra[$key]);
            }
        }

        // ---- Merge custom fields and extra properties into the return array
        if (count($custom)) {
            $return = array_merge(array_keys($custom), $return);
        }

        if (count($extra)) {
            $return = array_merge(array_keys($extra), $return);
        }


        return array_unique($return);
    }


    /**
     * Loads the property groups that have different values across the compared products.
     *
     * @param array $products
     * @return array
     */
    public function loadProperyGroupsDifferent(array $products): array
    {
        $return = [];
        $allPropertyGroups = $this->loadProperyGroups($products);
        // ---- Iterate through all property groups
        foreach ($allPropertyGroups as $groupName) {
            $groupName = (string)$groupName;
            $valuesInGroup = [];
            // ---- Collect the values of the property group for each product
            foreach ($products as $product) {
                $valuesInGroup[] = $this->page->getProductParameterByName($product, $groupName);
            }
            // ---- If the property group has different values across the products, add it to the return array
            if (count(array_unique($valuesInGroup)) > 1) {
                $return[] = $groupName;
            }
        }

        return $return;
    }


    /**
     * Loads an array of products by their IDs.
     *
     * @param array $productIds
     * @param SalesChannelContext $salesChannelContext
     * @return array
     */
    public function loadProducts(array $productIds, SalesChannelContext $salesChannelContext): array
    {
        $products = [];
        // ---- Load each product by its ID
        foreach ($productIds as $prodId) {
            $products[] = $this->loadProduct($prodId, $salesChannelContext);

        }
        return $products;
    }


    /**
     * Loads a single product by its ID.
     *
     * @param string $productId
     * @param SalesChannelContext $salesChannelContext
     * @return SalesChannelProductEntity|null
     */
    public function loadProduct(string $productId, SalesChannelContext $salesChannelContext): ?SalesChannelProductEntity
    {
        $criteria = (new Criteria([$productId]))
            ->addFilter(new ProductAvailableFilter($salesChannelContext->getSalesChannel()->getId(), ProductVisibilityDefinition::VISIBILITY_LINK))
            ->addAssociation('prices')
            ->addAssociation('manufacturer')
            ->addAssociation('cover.media')
            ->addAssociation('media.media')
            ->addAssociation('properties.group')
            ->addAssociation('mainCategories.category');

        $criteria->getAssociation('media')->addSorting(new FieldSorting('position'));


        /** @var SalesChannelProductEntity|null $product */
        $product = $this->productRepository->search($criteria, $salesChannelContext)->first();

        return $product;
    }


    /**
     * Retrieves the product groups for a given product.
     *
     * @param string $productId
     * @param SalesChannelProductEntity|null $product
     * @param SalesChannelContext $salesChannelContext
     * @return array
     */
    public function getProduktGroups(string $productId, ?SalesChannelProductEntity $product, SalesChannelContext $salesChannelContext): array
    {
        $return = [];

        /** @var SalesChannelProductEntity $product */
        if ($product === null) {
            $product = $this->loadProduct($productId, $salesChannelContext);
        }

        if ($product === null) {
            return $return;
        }

        $groupingPropertyGroupName = $this->getGroupingGroupName($salesChannelContext);

        // ---- If a grouping property is defined, extract the product groups based on that property
        if ($groupingPropertyGroupName) {
            foreach ($product->getSortedProperties()->getElements() as $group) {
                $gname = $group->getId();
                if (isset($group->getTranslated()['name']) && (string)$group->getTranslated()['name'] != '')
                    $gname = (string)$group->getTranslated()['name'];
                elseif ($group->getName() && (string)$group->getName() != '')
                    $gname = (string)$group->getName();  //misa translation


                if ($gname == $groupingPropertyGroupName) {
                    foreach ($group->getOptions()->getElements() as $opt) {
                        if (isset($opt->getTranslated()['name']) && (string)$opt->getTranslated()['name'] != '')
                            $return[$opt->getId()] = (string)$opt->getTranslated()['name'];
                        elseif ($opt->getName() && (string)$opt->getName() != '')
                            $return[$opt->getId()] = (string)$opt->getName();
                        else
                            $return[$opt->getId()] = (string)$opt->getId();
                    }
                }
            }
        }

        // ---- If no product groups are found and the setting to allow products without a property is disabled, return an empty array
        if (!$return && !$this->settings->getBool('allowWithoutProperty')) {
            return $return;
        }

        $return['all'] = $this->translator->trans('topdata-compare-products.allProducts');

        return $return;
    }


    /**
     * Adds a product to the compare list by its ID.
     *
     * @param string $productId
     * @param Request $request
     * @param SalesChannelContext $salesChannelContext
     * @return array
     */
    public function addProductById(string $productId, Request $request, SalesChannelContext $salesChannelContext): array
    {
        $this->setPage($request, $salesChannelContext);

        $product = $this->loadProduct($productId, $salesChannelContext);
        // ---- If the product is not found, return an error
        if (!$product) {
            return [
                'success' => false,
                'text'    => $this->translator->trans('topdata-compare-products.error.productNotFound')
            ];
        }

        $productGroups = $this->getProduktGroups($productId, $product, $salesChannelContext);

        // ---- If the product does not belong to any product group, return an error
        if (!$productGroups) {
            return [
                'success' => false,
                'text'    => $this->translator->trans(
                    'topdata-compare-products.error.groupingPropertyOnly',
                    ['%propertyName%' => $this->getGroupingGroupName($salesChannelContext)]
                )
            ];
        }

        $productCompare = Helper::getProductCompare($request);
        $productAddedGroups = [];
        $popupAddedWindow = false;
        // ---- Iterate through the product groups to add the product to each group
        foreach ($productGroups as $groupKey => $productArt) {
            $currentProductCompare = null;

            // ---- Find the existing compare group for the current product art
            foreach ($productCompare as $key => $productCompareGroup) {
                if ($productCompareGroup['productArt'] == $productArt) {
                    $currentProductCompare = $productCompareGroup;
                    unset($productCompare[$key]);
                    break;
                }
            }

            // ---- If no compare group exists for the current product art, create a new one
            if ($currentProductCompare === null) {
                $currentProductCompare = [
                    'groupKey'   => $groupKey,
                    'productArt' => $productArt,
                    'productIds' => []
                ];
            }

            $currentProductCompare['productIds'][] = $productId;
            $currentProductCompare['productIds'] = array_unique($currentProductCompare['productIds']);

            // ---- If the compare group is full, return an error
            if (count($currentProductCompare['productIds']) > $this->getMaxProductsInGroup()) {
                return [
                    'success'    => false,
                    'text'       => $this->translator->trans(
                        'topdata-compare-products.error.groupIsFull',
                        [
                            '%groupName%'     => $productArt,
                            '%productsLimit%' => $this->getMaxProductsInGroup()
                        ]
                    ),
                    'modalTitle' => $this->translator->trans('topdata-compare-products.error.groupIsFullTitle')
                ];
            }

            $currentProducts = $this->loadProducts($currentProductCompare['productIds'], $salesChannelContext);

            $currentProductCompare['propertyGroups'] = $this->loadProperyGroups($currentProducts);
            $currentProductCompare['propertyGroupsDifferent'] = $this->loadProperyGroupsDifferent($currentProducts);
            array_unshift($productCompare, $currentProductCompare);
            // ---- If the maximum number of groups is reached, return an error
            if (count($productCompare) > $this->getMaxGroups()) {
                return [
                    'success'    => false,
                    'text'       => $this->translator->trans(
                        'topdata-compare-products.error.maxGroups',
                        [
                            '%groupsLimit%' => $this->getMaxGroups()
                        ]
                    ),
                    'modalTitle' => $this->translator->trans('topdata-compare-products.error.groupIsFullTitle'),
                ];
            }

            if (count($currentProductCompare['productIds']) > 1) {
                $popupAddedWindow = true;
            }

            $productAddedGroups[] = $productArt;
        }

        Helper::setProductCompare($productCompare, $request, $this->translator->trans('topdata-compare-products.allProducts'));
        $groups = [];
        // ---- Generate the HTML for the compare group links
        if ((count($productCompare) == 2)
            && isset($productCompare[0])
            && isset($productCompare[1])
            && (count($productCompare[0]['productIds']) == count($productCompare[1]['productIds']))
        ) {
            $url = $this->router->generate('frontend.topdata-compare-products.api.load-popup', ['group' => 0]);
            $groups[] = '<a href="javascript:;" class="btn btn-primary btn-block topdata-compare-products-popup-compare-group topdata-compare-products-close-modal" data-path="' . $url . '" data-group="0">' . $productCompare[0]['productArt'] . ' <span class="badge badge-light">' . count($productCompare[0]['productIds']) . '</span></a>';
        } else {
            foreach ($productCompare as $key => $group) {
                $url = $this->router->generate('frontend.topdata-compare-products.api.load-popup', ['group' => $key]);
                $cssClass = $key > 0 ? 'btn-primary' : 'btn-secondary';
                $groups[] = '<a href="javascript:;" class="btn ' . $cssClass . ' btn-block topdata-compare-products-popup-compare-group topdata-compare-products-close-modal" data-path="' . $url . '" data-group="' . $key . '" >' . $group['productArt'] . ' <span class="badge badge-light">' . count($group['productIds']) . '</span></a>';
            }
        }

        $text = implode('', $groups);

        if (count($currentProductCompare['productIds']) == 1) {
            $popupAddedWindow = true;
            $text = $this->translator->trans('topdata-compare-products.addMoreProducts');
        }

        return [
            'success'      => true,
            'productId'    => $productId,
            'productName'  => $product->getName(),
            'productImage' => $product->getCover() ? $product->getCover()->getMedia()->getUrl() : '',
            'popup'        => $popupAddedWindow,
            'text'         => $text,
            'count'        => Helper::getCompareProductsTotalCount($request),
            'modalTitle'   => $this->translator->trans('topdata-compare-products.productCompareAdded')
        ];
    }


    /**
     * Removes a product from the compare list by its ID.
     *
     * @param string $productId
     * @param Request $request
     * @param SalesChannelContext $salesChannelContext
     * @return array
     */
    public function removeProductById(string $productId, Request $request, SalesChannelContext $salesChannelContext): array
    {
        $this->setPage($request, $salesChannelContext);
        $product = $this->loadProduct($productId, $salesChannelContext);
        // ---- If the product is not found, return an error
        if (!$product) {
            return [
                'success' => false,
                'text'    => 'Product not found'
            ];
        }

        $productGroups = $this->getProduktGroups($productId, $product, $salesChannelContext);

        // ---- If the product does not belong to any product group, return an error
        if (!$productGroups) {
            return [
                'success' => false,
                'text'    => 'Product groups not found'
            ];
        }

        $productCompare = Helper::getProductCompare($request);
        $affectedGroups = [];

        // ---- Iterate through the product groups to remove the product from each group
        foreach ($productGroups as $productArt) {

            $currentProductCompare = null;
            $currentProductCompareKey = null;
            foreach ($productCompare as $key => $productCompareGroup) {
                if ($productCompareGroup['productArt'] == $productArt) {
                    $currentProductCompare = $productCompareGroup;
                    $currentProductCompareKey = $key;
                    break;
                }
            }

            // ---- If no compare group exists for the current product art, continue to the next group
            if ($currentProductCompare === null) {
                continue;
            }

            $productKey = array_search($productId, $currentProductCompare['productIds']);
            // ---- If the product is not in the compare group, continue to the next group
            if ($productKey === false) {
                continue;
            }
            unset($currentProductCompare['productIds'][$productKey]);

            // ---- Update the compare group with the removed product
            if (count($currentProductCompare['productIds'])) {
                $currentProducts = $this->loadProducts($currentProductCompare['productIds'], $salesChannelContext);
                $currentProductCompare['propertyGroups'] = $this->loadProperyGroups($currentProducts);
                $currentProductCompare['propertyGroupsDifferent'] = $this->loadProperyGroupsDifferent($currentProducts);
                $productCompare[$currentProductCompareKey] = $currentProductCompare;
                $affectedGroups[] = [
                    'productArt' => $productArt,
                    'productIds' => $currentProductCompare['productIds'],
                    'isEmpty'    => false
                ];
            } else {
                unset($productCompare[$currentProductCompareKey]);
                $productCompare = array_values($productCompare);
                $affectedGroups[] = [
                    'productArt' => $productArt,
                    'productIds' => [],
                    'isEmpty'    => true
                ];
            }

        }

        Helper::setProductCompare($productCompare, $request, $this->translator->trans('topdata-compare-products.allProducts'));

        return [
            'success'      => true,
            'productId'    => $productId,
            'count'        => Helper::getCompareProductsTotalCount($request),
            'productImage' => $product->getCover() ? $product->getCover()->getMedia()->getUrl() : '',
            'text'         => $product->getName(),
            'modalTitle'   => $this->translator->trans('topdata-compare-products.productCompareRemoved'),
        ];
    }


    /**
     * Gets the current count of products in the compare list.
     *
     * @param Request $request
     * @return int
     */
    public function getCompareProductsCurrentCount(Request $request): int
    {
        $productCompare = Helper::getProductCompare($request);
        return isset($productCompare[0]) ? count($productCompare[0]['productIds']) : 0;
    }


    /**
     * Gets the maximum number of products allowed in a compare group.
     *
     * @return int
     */
    private function getMaxProductsInGroup(): int
    {
        return $this->settings->getInt('maxProductsInGroup') ?: self::COMPARE_PRODUCTS_LIMIT;
    }


    /**
     * Gets the maximum number of compare groups allowed.
     *
     * @return int
     */
    private function getMaxGroups(): int
    {
        return $this->settings->getInt('maxGroups') ?: self::COMPARE_GROUPS_LIMIT;
    }

    /**
     * Sets the page object with data loaded from the generic page loader and custom fields.
     *
     * @param Request $request
     * @param SalesChannelContext $salesChannelContext
     * @return void
     */
    private function setPage(Request $request, SalesChannelContext $salesChannelContext): void
    {
        $this->page = PopupCompareProductsWidget::createFrom(
            $this->genericLoader->load($request, $salesChannelContext)
        );
        $this->page->translator = $this->translator;
        $this->page->compareCustomFields = $this->loadCompareCustomFields($salesChannelContext);
    }


    /**
     * Loads a page with a specific set of products for comparison.
     *
     * @param Request $request
     * @param SalesChannelContext $salesChannelContext
     * @param array $productIds
     * @return \Topdata\FreeTopdataCompareProducts\Storefront\Page\PopupCompareProductsWidget\PopupCompareProductsWidget
     */
    public function loadProductsPage(Request $request, SalesChannelContext $salesChannelContext, array $productIds): PopupCompareProductsWidget
    {
        $this->setPage($request, $salesChannelContext);
        $this->page->products = $this->loadProducts($productIds, $salesChannelContext);
        $this->page->propertyGroups = $this->loadProperyGroups($this->page->products);
        $this->page->propertyGroupsDifferent = $this->loadProperyGroupsDifferent($this->page->products);
        $this->page->getMetaInformation()->setRobots('noindex,nofollow');
        $this->page->isPopup = false;
        return $this->page;
    }


    /**
     * Loads a page with a list of products, typically used for displaying a product listing.
     *
     * @param Request $request
     * @param SalesChannelContext $salesChannelContext
     * @param array $productIds
     * @return \Topdata\FreeTopdataCompareProducts\Storefront\Page\PopupCompareProductsWidget\PopupCompareProductsWidget
     */
    public function loadProductsListPage(Request $request, SalesChannelContext $salesChannelContext, array $productIds): PopupCompareProductsWidget
    {
        $this->setPage($request, $salesChannelContext);
        $this->page->products = $this->loadProducts($productIds, $salesChannelContext);
        $this->page->getMetaInformation()->setRobots('noindex,nofollow');
        $this->page->isPopup = false;
        return $this->page;
    }


    /**
     * Loads the custom fields that are configured for comparison.
     *
     * @param SalesChannelContext $salesChannelContext
     * @return EntityCollection|null
     */
    private function loadCompareCustomFields(SalesChannelContext $salesChannelContext): ?EntityCollection
    {
        $ids = $this->settings->getValue('customFieldIds');
        if (!is_array($ids)) {
            return null;
        }
        $entityCollection = $this
            ->customFieldRepository
            ->search(new Criteria($ids), $salesChannelContext->getContext());

        /*
         * Currently only maintained field types
         */
        // ---- Filter out custom fields that are not of a supported type
        foreach ($entityCollection as $id => $field) {
            $type = $field->getConfig()['customFieldType'] ?? false;
            if (in_array($type, ['text', 'select', 'checkbox', 'switch', 'number', 'date', 'colorpicker'])) {
                continue;
            }
            $entityCollection->remove($id);
        }

        return $entityCollection;
    }
}