import Plugin from 'src/plugin-system/plugin.class';
import HttpClient from 'src/service/http-client.service';

/**
 * CompareProductsModalPlugin handles the comparison modal functionality for products.
 * It manages modal opening/closing, product comparison operations, and image galleries.
 *
 * @extends Plugin
 */
export default class CompareProductsModalPlugin extends Plugin {

    // ---- INITIALIZATION SECTION ----

    /**
     * Initializes the plugin by setting up modal triggers and event subscriptions.
     * Registers event listeners for modal interactions and product comparisons.
     */
    init() {
        console.log('%c[DEBUG] ModalPlugin.init() started.', 'color: blue; font-weight: bold;');

        this.modalTrigger = document.querySelector('#topdata-compare-products-modal-trigger');
        console.log('[DEBUG] ModalPlugin.init(): Searching for modal trigger...', { modalTrigger: this.modalTrigger });
        if (!this.modalTrigger) {
            console.error('[DEBUG] ModalPlugin.init(): CRITICAL - Modal trigger element not found!');
        }

        this.$emitter.subscribe('openModalEvent', (event) => {
            this.openModal(event?.detail?.group, event?.detail?.open);
        });

        this._registerEventListeners();
    }

    // ---- EVENT LISTENERS SECTION ----

    /**
     * Registers all event listeners for modal interactions.
     * Sets up delegated click handlers and modal trigger listeners.
     * @private
     */
    _registerEventListeners() {
        console.log('[DEBUG] ModalPlugin._registerEventListeners(): Attaching listeners.');

        // This is the main delegated listener.
        document.body.addEventListener('click', this._onDelegatedHeaderClick.bind(this));
        console.log('[DEBUG] ModalPlugin._registerEventListeners(): Attached delegated listener to document.body.');

        // This listener waits for the main one to trigger it.
        if (this.modalTrigger) {
            this.modalTrigger.addEventListener('click', this._onTriggerClick.bind(this));
        }
    }

    /**
     * Handles delegated click events on the document body to detect header link clicks.
     * Prevents default behavior and triggers the modal when the compare products link is clicked.
     * @param {Event} event - The click event
     * @private
     */
    _onDelegatedHeaderClick(event) {
        // This log should appear for EVERY SINGLE click anywhere on the page.
        console.log('%c[DEBUG] Body click detected.', 'color: orange;', { target: event.target });

        // Check if the click originated from our specific header link.
        const headerLink = event.target.closest('.topdata-compare-products-show-products-compare');
        console.log('[DEBUG] ... searching for header link from click target. Result:', { headerLink });

        if (!headerLink) {
            // This is normal for clicks on other elements. We just stop here.
            return;
        }

        // If we reach this point, the click was on our link.
        console.log('%c[SUCCESS] Header link click identified! Proceeding to open modal.', 'color: green; font-weight: bold;');
        event.preventDefault();

        if (this.modalTrigger) {
            console.log('[DEBUG] Triggering click on the hidden modal trigger element.');
            this.modalTrigger.click();
        } else {
            console.error('[DEBUG] CRITICAL - Cannot open modal because this.modalTrigger is null.');
        }
    }

    /**
     * Handles the hidden modal trigger click event.
     * This is the final step that actually opens the modal.
     * @private
     */
    _onTriggerClick() {
        // This log confirms the final step is working.
        console.log('%c[SUCCESS] Hidden modal trigger was clicked. Loading modal content.', 'color: green; font-weight: bold;');
        this.openModal(null);
    }

    // ---- PRODUCT COMPARISON MANAGEMENT SECTION ----

    /**
     * Handles the response when a product is removed from comparison.
     * Updates the UI to reflect the new product count and visibility states.
     * @param {string} data - JSON string containing removal response data
     */
    productCompareRemoved(data) {
        data = JSON.parse(data);
        if (data.success === true) {
            document.querySelector('.topdata-compare-products-count-total').innerHTML = data.count;
            if (data.count) {
                document.querySelector('.topdata-compare-products-path').classList.remove('topdata-compare-products-hidden');
            } else {
                document.querySelector('.topdata-compare-products-path').classList.add('topdata-compare-products-hidden');
                this.el.querySelector('.btn-close').click();
            }
            document.querySelector('.topdata-compare-products-product-comparing-' + data.productId).classList.add('topdata-compare-products-hidden');
            document.querySelector('.topdata-compare-products-product-compare-' + data.productId).classList.remove('topdata-compare-products-hidden');
        }
        this.openModal();
    }

    /**
     * Opens the comparison modal and loads content for the specified group.
     * Makes an AJAX request to fetch modal content and updates the UI.
     * @param {string|null} group - The product group to load, defaults to element's data-group
     * @param {boolean} open - Whether to automatically open the modal after loading
     */
    openModal(group = null, open = false) {
        if (group === null) {
            group = this.el.dataset.group;
        }
        if (group === '-1') {
            group = '0';
        }
        const containerTitle = this.el.querySelector('.modal-title');
        const containerBody = this.el.querySelector('.modal-body');
        const path = document.querySelector('.topdata-compare-products-path').dataset.path + '?group=' + group;
        this.showCompareContentLoading(true);
        let productComparePopupHttpClient = new HttpClient(window.accessKey, window.contextToken);
        productComparePopupHttpClient.get(path, (response) => {
            response = JSON.parse(response);
            if (response.success === true) {
                containerBody.innerHTML = response.body;
                containerTitle.innerHTML = response.title;
                this.onAjaxContentEvents();
            }
            this.showCompareContentLoading(false);
            this.el.dataset.empty = false;
            this.el.dataset.group = group;
            if (open) {
                document.querySelector('#topdata-compare-products-modal-trigger').click();
            }
        });
    }

    /**
     * Shows or hides the loading state for modal content.
     * Toggles loading CSS classes on title and body containers.
     * @param {boolean} visible - Whether to show the loading state
     */
    showCompareContentLoading(visible) {
        const containerTitle = this.el.querySelector('.modal-title');
        const containerBody = this.el.querySelector('.modal-body');

        if (visible) {
            containerTitle.classList.add('topdata-compare-products-loading');
            containerBody.classList.add('topdata-compare-products-loading');
        } else {
            containerTitle.classList.remove('topdata-compare-products-loading');
            containerBody.classList.remove('topdata-compare-products-loading');
        }
    }

    // ---- AJAX CONTENT EVENT HANDLERS SECTION ----

    /**
     * Registers event listeners for all AJAX-loaded content within the modal.
     * Called after modal content is dynamically loaded via AJAX.
     */
    onAjaxContentEvents() {
        this.onRemoveProduct();
        this.onGroupChange();
        this.onCompareSettings();
        this.onAddProducts();
        this.onImageChange(); // <-- ADD THIS
    }

    // ---- IMAGE GALLERY SECTION ----

    /**
     * Registers event listeners for the next/previous image buttons.
     */
    onImageChange() {
        const nextElements = this.el.getElementsByClassName('topdata-compare-products-product-box-image-next');
        for (let i = 0; i < nextElements.length; i++) {
            nextElements[i].addEventListener('click', this._onImageNext.bind(this));
        }
        const prevElements = this.el.getElementsByClassName('topdata-compare-products-product-box-image-prev');
        for (let i = 0; i < prevElements.length; i++) {
            prevElements[i].addEventListener('click', this._onImagePrev.bind(this));
        }
    }

    /**
     * Handles the 'next' image button click.
     * @param {Event} event
     * @private
     */
    _onImageNext(event) {
        this._onChangeImage(event, 1);
    }

    /**
     * Handles the 'previous' image button click.
     * @param {Event} event
     * @private
     */
    _onImagePrev(event) {
        this._onChangeImage(event, -1);
    }

    /**
     * Core logic for changing the displayed product image in a gallery.
     * @param {Event} event - The click event from the next/prev button.
     * @param {number} direction - 1 for next, -1 for previous.
     * @private
     */
    _onChangeImage(event, direction) {
        const container = event.currentTarget.closest('.topdata-compare-products-product-box-image');
        if (!container) return;

        // Since sw_thumbnails creates <picture><img...></picture>, we need to toggle a class
        // on a common element. The template adds 'is-standart' to the <img> tags.
        const images = container.querySelectorAll('.is-standart');
        const dots = container.querySelectorAll('.topdata-compare-products-product-box-image-dot');
        if (images.length <= 1) return;

        let currentIndex = parseInt(container.dataset.current, 10) || 0;
        let nextIndex = currentIndex + direction;

        // Handle wrapping around the gallery
        if (nextIndex >= images.length) {
            nextIndex = 0;
        } else if (nextIndex < 0) {
            nextIndex = images.length - 1;
        }

        // Update image visibility by toggling the hidden class
        images[currentIndex].classList.add('topdata-compare-products-hidden');
        images[nextIndex].classList.remove('topdata-compare-products-hidden');

        // Update dot indicators
        if (dots.length > 0 && dots[currentIndex] && dots[nextIndex]) {
            dots[currentIndex].classList.remove('current');
            dots[nextIndex].classList.add('current');
        }

        // Update the current index state on the container
        container.dataset.current = nextIndex;
    }

    // ---- PRODUCT REMOVAL SECTION ----

    /**
     * Registers event listeners for product removal buttons.
     * Sets up click handlers for all remove product elements in the modal.
     */
    onRemoveProduct() {
        let rmElements = this.el.getElementsByClassName('topdata-compare-products-remove');
        for (let i = 0; i < rmElements.length; i++) {
            rmElements[i].addEventListener('click', this._onProductRemove.bind(this))
        }
    }

    /**
     * Handles product removal when a remove button is clicked.
     * Makes an AJAX request to remove the product from comparison.
     * @param {Event} event - The click event from the remove button
     * @private
     */
    _onProductRemove(event) {
        let path = event.currentTarget.dataset.path
        const httpClient = new HttpClient(window.accessKey, window.contextToken);
        httpClient.get(path, (response) => {
            this.productCompareRemoved(response)
        });
    }

    // ---- GROUP SELECTION SECTION ----

    /**
     * Registers event listeners for product group selection.
     * Sets up click handlers for group selector elements.
     */
    onGroupChange() {
        let groupSelectorElements = this.el.getElementsByClassName('topdata-compare-products-group-selector');
        for (let i = 0; i < groupSelectorElements.length; i++) {
            groupSelectorElements[i].addEventListener('click', this._onChangeGroup.bind(this))
        }
    }

    /**
     * Handles group change when a group selector is clicked.
     * Reloads the modal with products from the selected group.
     * @param {Event} event - The click event from the group selector
     * @private
     */
    _onChangeGroup(event) {
        let group = event.currentTarget.dataset.group;
        this.openModal(group);
    }

    // ---- COMPARISON SETTINGS SECTION ----

    /**
     * Registers event listeners for comparison settings.
     * Sets up handlers for property visibility toggles.
     */
    onCompareSettings() {
        this._onCompareSettingsProperties()
    }

    /**
     * Registers event listeners for property visibility toggle buttons.
     * Sets up click handlers for elements that can show/hide comparison properties.
     * @private
     */
    _onCompareSettingsProperties() {
        let settingsPropertiesSelectorElements = this.el.getElementsByClassName('topdata-compare-products-hiddable-trigger');
        for (let i = 0; i < settingsPropertiesSelectorElements.length; i++) {
            settingsPropertiesSelectorElements[i].addEventListener('click', this.__onClickProperty.bind(this))
        }
    }

    /**
     * Handles property visibility toggle when a settings trigger is clicked.
     * Shows or hides comparison properties based on the group or globally.
     * @param {Event} event - The click event from the property toggle trigger
     * @private
     */
    __onClickProperty(event) {
        let group = event.currentTarget.dataset.group;
        if (group) {
            let elements = document.querySelectorAll('.topdata-compare-products-hiddable[data-group="' + group + '"]');
            for (let i = 0; i < elements.length; i++) {
                if (elements[i].classList.contains('topdata-compare-products-hidden')) {
                    elements[i].classList.remove('topdata-compare-products-hidden');
                } else {
                    elements[i].classList.add('topdata-compare-products-hidden');
                }
            }
        } else {
            let elements = document.querySelectorAll('.topdata-compare-products-hiddable');
            for (let i = 0; i < elements.length; i++) {
                if (elements[i].classList.contains('topdata-compare-products-hidden')) {
                    elements[i].classList.remove('topdata-compare-products-hidden');
                } else {
                    elements[i].classList.add('topdata-compare-products-hidden');
                }
            }
        }
    }

    // ---- PRODUCT ADDITION SECTION ----

    /**
     * Registers event listeners for adding products to comparison.
     * Sets up click handlers for elements that close the modal to add more products.
     */
    onAddProducts() {
        let addProductElements = this.el.getElementsByClassName('topdata-compare-products-close-modal');
        for (let i = 0; i < addProductElements.length; i++) {
            addProductElements[i].addEventListener('click', this._onAddProduct.bind(this))
        }
    }

    /**
     * Handles the add product action by closing the modal.
     * Allows users to return to the main page to add more products to compare.
     * @param {Event} event - The click event from the add product button
     * @private
     */
    _onAddProduct(event) {
        this.el.querySelector('.btn-close').click();
    }
}