import Plugin from 'src/plugin-system/plugin.class';
import HttpClient from 'src/service/http-client.service';
import DomAccess from 'src/helper/dom-access.helper';

/**
 * Plugin for loading and managing the state of product comparison functionality.
 * Handles loading product comparison data from server and updating DOM elements
 * to reflect the current comparison state.
 *
 * @extends Plugin
 */
export default class CompareProductsLoadStatePlugin extends Plugin {
    /**
     * Initialize the plugin by setting up the element functionality.
     */
    init() {
        this.initElement()
    }

    // ---- Element Initialization and State Loading ----

    /**
     * Initialize the comparison element by fetching comparison state from server
     * and updating the DOM to reflect the current product comparison status.
     *
     * Performs the following operations:
     * - Finds the compare products path element
     * - Extracts the API path from data attribute
     * - Makes HTTP request to load comparison state
     * - Updates product count displays
     * - Shows/hides comparison elements based on state
     * - Updates individual product comparison buttons
     */
    initElement() {
        try {
            const element = document.querySelector('.topdata-compare-products-path');
            const path = DomAccess.getAttribute(element, 'data-pathstate');

            if (path) {
                const httpClient = new HttpClient(window.accessKey, window.contextToken);
                httpClient.get(path, (response) => {
                    response = JSON.parse(response);

                    // ---- Product Count Display Update ----
                    if (response.productCount) {
                        document.querySelectorAll('.topdata-compare-products-count-total').forEach(el => {
                            el.innerHTML = response.productCount;
                        });

                        // ---- Main Compare Container Visibility ----
                        document.querySelector('.topdata-compare-products-path').classList.remove('topdata-compare-products-hidden');

                        // ---- Individual Product State Updates ----
                        response.productIds.forEach(productId => {
                            const comparingElements = document.querySelectorAll('.topdata-compare-products-product-comparing-' + productId);
                            comparingElements.forEach(el => {
                                el.classList.remove('topdata-compare-products-hidden');
                            });

                            const compareElements = document.querySelectorAll('.topdata-compare-products-product-compare-' + productId);
                            compareElements.forEach(el => {
                                el.classList.add('topdata-compare-products-hidden');
                            });
                        });
                    } else {
                        // ---- Hide Compare Container When No Products ----
                        document.querySelector('.topdata-compare-products-path').classList.add('topdata-compare-products-hidden');
                    }
                });
            }
        } catch (e) {
            console.log(e);
        }
    }
}